﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/sf/sf_Types.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/dd/dd_DeviceAddressSpace.h>
#include <nn/nn_SdkLog.h>

#include <nn/audioctrl.h>
#include <nn/audioctrl/audioctrl_PlayReport.h>
#include "detail/audioctrl_CreateAudioController.h"

namespace nn { namespace audioctrl {

namespace {

nn::sf::SharedPointer<detail::IAudioController> CreateAudioController() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    return detail::CreateAudioControllerByHipc();
#else
    return detail::CreateAudioControllerByDfc();
#endif
}

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
bool IsSupportedTarget(AudioTarget target)
{
    return target == AudioTarget_Speaker || target == AudioTarget_Headphone || target == AudioTarget_UsbOutputDevice;
}

bool IsDefaultableTarget(AudioTarget target)
{
    return target == AudioTarget_Speaker || target == AudioTarget_Tv;
}

bool IsValidTargetForAudioOutputMode(AudioTarget target) NN_NOEXCEPT
{
    // TODO: mode に DeviceType_AhubHeadphone の値を設定するようになった場合は要修正
    // nn::audioctrl::AudioTarget ConvertToTarget(AppletVolumeManager::DeviceType device) @ audio_AudioDeviceImpl-os.horizon.cpp の修正が必要
    // See http://spdlybra.nintendo.co.jp/stash/projects/SIGLO/repos/sdk/pull-requests/24691/overview?commentId=268371
    return target == AudioTarget_Speaker || target == AudioTarget_Tv || target == AudioTarget_UsbOutputDevice;
}

bool IsValidTargetForOutputTarget(AudioTarget target) NN_NOEXCEPT
{
    return target == AudioTarget_Invalid || target == AudioTarget_Speaker || target == AudioTarget_Headphone|| target == AudioTarget_Tv;
}

bool IsValidOutputMode(AudioOutputMode mode) NN_NOEXCEPT
{
    return mode == AudioOutputMode_Pcm1ch || mode == AudioOutputMode_Pcm2ch || mode == AudioOutputMode_Pcm6ch || mode == AudioOutputMode_PcmAuto;
}
#endif

} // anonymous namespace

int GetTargetVolume(AudioTarget target) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsSupportedTarget(target));

    int32_t value = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioController()->GetTargetVolume(static_cast<uint32_t>(target), &value));
    return value;
}

void SetTargetVolume(AudioTarget target, int volume) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsSupportedTarget(target));
    NN_SDK_REQUIRES(volume >= nn::audioctrl::GetTargetVolumeMin() && volume <= nn::audioctrl::GetTargetVolumeMax());

    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioController()->SetTargetVolume(static_cast<uint32_t>(target), volume));
}

int GetTargetVolumeMax() NN_NOEXCEPT
{
    int32_t volume = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioController()->GetTargetVolumeMax(&volume));
    return volume;
}

int GetTargetVolumeMin() NN_NOEXCEPT
{
    int32_t volume = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioController()->GetTargetVolumeMin(&volume));
    return volume;
}

bool IsTargetMute(AudioTarget target) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsSupportedTarget(target));
    bool value = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioController()->IsTargetMute(static_cast<uint32_t>(target), &value));
    return value;
}

void SetTargetMute(AudioTarget target, bool isMute) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsSupportedTarget(target));

    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioController()->SetTargetMute(static_cast<uint32_t>(target), isMute));
}

bool IsTargetConnected(AudioTarget target) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsSupportedTarget(target));
    bool value = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioController()->IsTargetConnected(static_cast<uint32_t>(target), &value));
    return value;
}

void SetDefaultTarget(AudioTarget target, nn::TimeSpanType fadeOutTime, nn::TimeSpanType fadeInTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsDefaultableTarget(target));
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioController()->SetDefaultTarget(static_cast<uint32_t>(target), fadeOutTime, fadeInTime));
}

AudioTarget GetDefaultTarget() NN_NOEXCEPT
{
    int32_t target = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioController()->GetDefaultTarget(&target));
    return static_cast<AudioTarget>(target);
}

AudioOutputMode GetOutputModeSetting(AudioTarget target) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValidTargetForAudioOutputMode(target));
    uint32_t mode;
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioController()->GetOutputModeSetting(&mode, static_cast<uint32_t>(target)));
    return static_cast<AudioOutputMode>(mode);
}

void SetOutputModeSetting(AudioTarget target, AudioOutputMode mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValidTargetForAudioOutputMode(target));
    NN_SDK_REQUIRES(IsValidOutputMode(mode));
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioController()->SetOutputModeSetting(static_cast<uint32_t>(target), static_cast<uint32_t>(mode)));
}

AudioOutputMode GetAudioOutputMode(AudioTarget target) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValidTargetForAudioOutputMode(target));
    uint32_t mode;
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioController()->GetAudioOutputMode(&mode, static_cast<uint32_t>(target)));
    return static_cast<AudioOutputMode>(mode);
}

void SetAudioOutputMode(AudioTarget target, AudioOutputMode mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValidTargetForAudioOutputMode(target));
    NN_SDK_REQUIRES(IsValidOutputMode(mode));
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioController()->SetAudioOutputMode(static_cast<uint32_t>(target), static_cast<uint32_t>(mode)));
}

void SetForceMutePolicy(ForceMutePolicy policy) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(policy == ForceMutePolicy_Disable || policy == ForceMutePolicy_SpeakerMuteOnHeadphoneUnplugged);
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioController()->SetForceMutePolicy(static_cast<int32_t>(policy)));
}

ForceMutePolicy GetForceMutePolicy() NN_NOEXCEPT
{
    int32_t policy = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioController()->GetForceMutePolicy(&policy));
    return static_cast<ForceMutePolicy>(policy);
}

void SetOutputTargetForProduction(AudioTarget target) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValidTargetForOutputTarget(target));
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioController()->SetOutputTarget(static_cast<uint32_t>(target)));
}

void SetInputTargetForceEnabledForProduction(bool isEnabled) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioController()->SetInputTargetForceEnabled(isEnabled));
}

void SetHeadphoneOutputLevelMode(HeadphoneOutputLevelMode mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(mode == HeadphoneOutputLevelMode_Normal || mode == HeadphoneOutputLevelMode_HighPower);
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioController()->SetHeadphoneOutputLevelMode(static_cast<uint32_t>(mode)));
}

HeadphoneOutputLevelMode GetHeadphoneOutputLevelMode() NN_NOEXCEPT
{
    uint32_t mode = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioController()->GetHeadphoneOutputLevelMode(&mode));
    return static_cast<HeadphoneOutputLevelMode>(mode);
}

void BindAudioVolumeUpdateEventForPlayReport(::nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    auto handle = ::nn::sf::NativeHandle();
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioController()->AcquireAudioVolumeUpdateEventForPlayReport(&handle));

    ::nn::os::AttachReadableHandleToSystemEvent(pEvent,
                                                handle.GetOsHandle(),
                                                handle.IsManaged(),
                                                nn::os::EventClearMode_AutoClear);

    handle.Detach();
}

void GetAudioVolumeDataForPlayReport(PlayReportAudioVolumeData* pOutData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutData);

    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioController()->GetAudioVolumeDataForPlayReport(pOutData));
}

void BindAudioOutputDeviceUpdateEventForPlayReport(::nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    auto handle = ::nn::sf::NativeHandle();
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioController()->AcquireAudioOutputDeviceUpdateEventForPlayReport(&handle));

    ::nn::os::AttachReadableHandleToSystemEvent(pEvent,
                                                handle.GetOsHandle(),
                                                handle.IsManaged(),
                                                nn::os::EventClearMode_AutoClear);
    handle.Detach();
}

PlayReportAudioOutputTarget GetAudioOutputTargetForPlayReport() NN_NOEXCEPT
{
    uint32_t target = 0;

    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioController()->GetAudioOutputTargetForPlayReport(&target));

    return static_cast<PlayReportAudioOutputTarget>(target);
}

void NotifyHeadphoneVolumeWarningDisplayedEvent() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioController()->NotifyHeadphoneVolumeWarningDisplayedEvent());
}


float GetSystemOutputMasterVolume() NN_NOEXCEPT
{
    float volume = 0.f;
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioController()->GetSystemOutputMasterVolume(&volume));

    return volume;
}

void SetSystemOutputMasterVolume(float volume) NN_NOEXCEPT
{
    NN_SDK_ASSERT_MINMAX(volume, SystemOutputMasterVolumeMin, SystemOutputMasterVolumeMax);
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioController()->SetSystemOutputMasterVolume(volume));
}

void UpdateHeadphoneSettings(bool isRestricted) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateAudioController()->UpdateHeadphoneSettings(isRestricted));
}

}}  // namespace nn::audioctrl
