﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <cinttypes>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Semaphore.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/erpt.h>
#include <nn/err.h>
#include <nn/err/detail/err_ErrorCodeConvert.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/pcv/pcv.h>
#include <nne/audio/audio.h>
#include <nn/codec/server/codec_HardwareOpusDecoderManager.h>
#include <nn/audio/server/audio_FirmwareDebugSettings.h>
#include <nn/audio/audio_ResultForPrivate.h>
#include <nn/audio/detail/audio_Log.h>
#include <nn/settings/system/settings_Audio.h>
#include <nn/gpio/gpio.h>
#include <nn/settings/factory/settings_Speaker.h>
#include <nn/util/util_Endian.h>
#include <nn/fatal/fatal_ApiPrivate.h>
#include "audio_AudioHardware.h"
#include "audio_AppletVolumeManager.h"
#include "detail/audioctrl_AudioControllerService.h"
#include "../dsp/audio_DspExceptionNotifier.h"

#include "audio_AudioHardware-hardware.nx.h"

#if !defined(NN_BUILD_CONFIG_SOC_TEGRA_X1)
#error This SoC is not supported.
#endif

// ADSP firmware image
extern const char adsp_os_bin_begin[];
extern const char adsp_os_bin_end[];
extern const char vector_bin_begin[];
extern const char vector_bin_end[];

namespace nn { namespace audio { namespace server { namespace detail {


namespace {

NN_OS_ALIGNAS_THREAD_STACK uint8_t g_HdaHotplugDetectThreadStack[12 * 1024];
NN_OS_ALIGNAS_THREAD_STACK uint8_t g_CodecAlertHandlerThreadStack[12 * 1024];
const nn::TimeSpan MinimumFadeTime = nn::TimeSpan::FromMilliSeconds(10);
const nn::TimeSpan ThreadTimeOut = nn::TimeSpan::FromSeconds(1);

AudioDeviceController g_DeviceController;

int AdspExceptionHandler(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);
    NN_DETAIL_AUDIO_FATAL("[audio] ADSP Exception Handler is called.\n");

    // Turn off sounds played by Codec IC.
    // (HDA doesn't need to do it.)
    nne::audio::codec::SetOutVolume(0);

    if (arg)
    {
        auto exceptionContext = reinterpret_cast<nne::audio::adsp::ExceptionContext*>(arg);
        static uint8_t contextLocalBuffer[512 * 4];
        nn::erpt::Context context(nn::erpt::CategoryId::AdspErrorInfo, contextLocalBuffer, sizeof(contextLocalBuffer));

        auto result = context.Add(nn::erpt::FieldId::AdspExceptionReason,
                        static_cast<uint32_t>(exceptionContext->exceptionReason));
        NN_SDK_ASSERT(result.IsSuccess());

        result = context.Add(nn::erpt::FieldId::AdspExceptionRegisters,
                        exceptionContext->frame.r,
                        sizeof(exceptionContext->frame.r) / sizeof(exceptionContext->frame.r[0]));
        NN_SDK_ASSERT(result.IsSuccess());

        result = context.Add(nn::erpt::FieldId::AdspExceptionStackPointer,
                        exceptionContext->frame.usp);
        NN_SDK_ASSERT(result.IsSuccess());

        result = context.Add(nn::erpt::FieldId::AdspExceptionLinkRegister,
                        exceptionContext->frame.ulr);
        NN_SDK_ASSERT(result.IsSuccess());

        result = context.Add(nn::erpt::FieldId::AdspExceptionProgramCounter,
                        exceptionContext->frame.pc);
        NN_SDK_ASSERT(result.IsSuccess());

        result = context.Add(nn::erpt::FieldId::AdspExceptionSpsr,
                        exceptionContext->frame.spsr);
        NN_SDK_ASSERT(result.IsSuccess());

        result = context.Add(nn::erpt::FieldId::AdspExceptionArmModeRegisters,
                        reinterpret_cast<uint32_t*>(&exceptionContext->regs),
                        sizeof(nne::audio::adsp::ArmModeRegs) / sizeof(uint32_t));
        NN_SDK_ASSERT(result.IsSuccess());

        result = context.Add(nn::erpt::FieldId::AdspExceptionStackAddress,
                        exceptionContext->stackAddr);
        NN_SDK_ASSERT(result.IsSuccess());

        result = context.Add(nn::erpt::FieldId::AdspExceptionStackDump,
                        exceptionContext->stackDump,
                        sizeof(exceptionContext->stackDump) / sizeof(exceptionContext->stackDump[0]));
        NN_SDK_ASSERT(result.IsSuccess());

        result = context.SubmitContext();
        NN_SDK_ASSERT(result.IsSuccess());
    }

    nn::audio::dsp::DspExceptionNotifier::Notify();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::ResultAudioDspAbort());

    return 0;
}

nn::Result ConvertAdspErrorTypeToNnResult(int type) NN_NOEXCEPT
{
    switch(type)
    {
    case nne::audio::adsp::ADSP_ERROR_TYPE_INIT_FAILURE:
        NN_RESULT_THROW(nn::audio::ResultAudioDspInitilizationFailed());
    case nne::audio::adsp::ADSP_ERROR_TYPE_BOOT_TIMEOUT:
        NN_RESULT_THROW(nn::audio::ResultAudioDspBootTimeout());
    case nne::audio::adsp::ADSP_ERROR_TYPE_SUCCESS:
        NN_RESULT_SUCCESS;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

} // namespace

// ================================================================================
// SleepWakePolicy
// ================================================================================
SleepWakePolicy::SleepWakePolicy() NN_NOEXCEPT
    : m_SleepWakeState(State::Invalid)
    , m_Mutex(false)
{
}

SleepWakePolicy::State SleepWakePolicy::GetSleepWakeState() NN_NOEXCEPT
{
    return m_SleepWakeState;
}

void SleepWakePolicy::Initialize(bool hasCodecIc, bool hasHdmi) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    if (IsReadyToInitialize() == false)
    {
        return;
    }

    m_HasCodecIc = hasCodecIc;
    m_HasHdmi = hasHdmi;

    InitializeImpl();
}

void SleepWakePolicy::Finalize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    if (IsReadyToFinalize() == false)
    {
        return;
    }

    FinalizeImpl();
}

void SleepWakePolicy::Sleep() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    if (IsReadyToSleep() == false)
    {
        return;
    }

    SleepImpl();
}

void SleepWakePolicy::Wake() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    if (IsReadyToWake() == false)
    {
        return;
    }

    WakeImpl();
}

inline bool SleepWakePolicy::CheckState(bool condition, State nextState) NN_NOEXCEPT
{
    if (condition)
    {
        m_SleepWakeState = nextState;
        return true;
    }
    else
    {
        return false;
    }
}

bool SleepWakePolicy::IsReadyToInitialize() NN_NOEXCEPT
{
    return CheckState(
        (m_SleepWakeState == State::Invalid) || (m_SleepWakeState == State::Finalized),
        State::Initialized);
}

bool SleepWakePolicy::IsReadyToSleep() NN_NOEXCEPT
{
    return CheckState(
        (m_SleepWakeState == State::Initialized),
        State::Asleep);
}

bool SleepWakePolicy::IsReadyToWake() NN_NOEXCEPT
{
    return CheckState(
        (m_SleepWakeState == State::Asleep),
        State::Initialized);
}

bool SleepWakePolicy::IsReadyToFinalize() NN_NOEXCEPT
{
    return CheckState(
        (m_SleepWakeState == State::Initialized),
//         (m_SleepWakeState == State::Initialized) || (m_SleepWakeState == State::Asleep)),
        State::Finalized);
}

// ================================================================================
// Gmix
// ================================================================================

void Gmix::InitializeImpl() NN_NOEXCEPT
{
    nne::audio::gmix::Initialize();
    nn::os::InitializeSemaphore(&m_ProcessSemaphore, 0, 1);
    nne::audio::gmix::SetProcessSemaphore(nne::audio::gmix::Session::Name::Default, &m_ProcessSemaphore);

    UpdateDownmixModeForGameRecord(Gmix::DownmixMode::Default);
}

void Gmix::FinalizeImpl() NN_NOEXCEPT
{
    nne::audio::gmix::Finalize();
}

void Gmix::SleepImpl() NN_NOEXCEPT
{
    FinalizeImpl();
}

void Gmix::WakeImpl() NN_NOEXCEPT
{
    InitializeImpl();
}

void Gmix::UpdateDownmixMode(DownmixMode mode) NN_NOEXCEPT
{
    if (mode == DownmixMode::Mono)
    {
        nne::audio::gmix::SetFunctionPtr(
            nne::audio::gmix::Function::Downmix16_6ch,
            nne::audio::gmix::GetFunctionPtr(nne::audio::gmix::Function::Downmix16_6ch_Mono)
        );
    }
    else
    {
        nne::audio::gmix::SetFunctionPtr(
            nne::audio::gmix::Function::Downmix16_6ch,
            nne::audio::gmix::GetFunctionPtr(nne::audio::gmix::Function::Downmix16_6ch_Default)
        );
    }
}

void Gmix::UpdateDownmixModeForGameRecord(DownmixMode mode) NN_NOEXCEPT
{
    if (mode == DownmixMode::Mono)
    {
        nne::audio::gmix::SetFunctionPtr(
            nne::audio::gmix::Function::GameRecordDownmix16_6ch,
            nne::audio::gmix::GetFunctionPtr(nne::audio::gmix::Function::Downmix16_6ch_Mono)
        );
    }
    else
    {
        nne::audio::gmix::SetFunctionPtr(
            nne::audio::gmix::Function::GameRecordDownmix16_6ch,
            nne::audio::gmix::GetFunctionPtr(nne::audio::gmix::Function::Downmix16_6ch_Default)
        );
    }
}

void Gmix::PrintPerformanceStatistics() NN_NOEXCEPT
{
    nne::audio::gmix::stats_t stat;
    nne::audio::gmix::Stats(&stat);

#if 0
    NN_DETAIL_AUDIO_INFO(
        "AudioOutCycles "   "AudioOutPasses "   "AudioRendererCycles "      "AudioRendererPasses "      "SystemSoundCycles "
        "SystemSoundPass "  "FarVoiceCycles "   "FarVoicePasses "           "NearVoiceCycles "          "NearVoicePasses "
        "GameRecordCycles " "GameRecordPasses " "NearVoiceNarrationCycles " "NearVoiceNarrationPasses " "AhubOutCycles "
        "AhubOutPasses "    "AhubInCycles "     "AhubInPasses "             "HdaOutCycles "             "HdaOutPasses "
        "AdspStarve "       "CpuStarve\n");
#endif

    const auto sp = stat.Process;
    NN_UNUSED(sp);
    NN_DETAIL_AUDIO_INFO(
        "%" PRIu64 " %" PRIu32 " %" PRIu64 " %" PRIu32 " %" PRIu64 " "
        "%" PRIu32 " %" PRIu64 " %" PRIu32 " %" PRIu64 " %" PRIu32 " "
        "%" PRIu64 " %" PRIu32 " %" PRIu64 " %" PRIu32 " %" PRIu64 " "
        "%" PRIu32 " %" PRIu64 " %" PRIu32 " %" PRIu64 " %" PRIu64 " "
        "%" PRIu32 " %" PRIu32 " \n",
        sp.AudioOutCycles,    sp.AudioOutPasses,   sp.AudioRendererCycles,      sp.AudioRendererPasses,      sp.SystemSoundCycles,
        sp.SystemSoundPasses, sp.FarVoiceCycles,   sp.FarVoicePasses,           sp.NearVoiceCycles,          sp.NearVoicePasses,
        sp.GameRecordCycles,  sp.GameRecordPasses, sp.NearVoiceNarrationCycles, sp.NearVoiceNarrationPasses, sp.AhubOutCycles,
        sp.AhubOutPasses,     sp.AhubInCycles,     sp.AhubInPasses,             sp.HdaOutCycles,             sp.HdaOutPasses,
        sp.AdspStarve,        sp.CpuStarve
    );
}

// ================================================================================
// Ahub
// ================================================================================

Ahub::Ahub() NN_NOEXCEPT
    : m_pGmixAhub(nullptr)
    , m_pAhub(nullptr)
    , m_ProcessSemaphore(0, 1)
{
}

void Ahub::InitializeImpl() NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        return;
    }

    nne::audio::ahub::Initialize();

    //Start ahub out
    nne::audio::gmix::OpenSession(
        &m_pGmixAhub,
        nne::audio::gmix::Session::Name::Ahub,
        0,
        nne::audio::gmix::Session::Name::Default,
        nne::audio::gmix::Session::Format::Stereo,
        nne::audio::gmix::Session::Mode::Default);
    nne::audio::ahub::OpenSession(&m_pAhub, 0);

    nne::audio::gmix::SetProcessSemaphore(nne::audio::gmix::Session::Name::Ahub, m_ProcessSemaphore.GetBase());

    m_pGmixAhub->Start();
    m_pAhub->Start();
}

void Ahub::FinalizeImpl() NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        return;
    }

    m_pAhub->Stop();
    m_pGmixAhub->Stop();

    nne::audio::ahub::CloseSession(m_pAhub);
    nne::audio::gmix::CloseSession(m_pGmixAhub);

    nne::audio::ahub::Finalize();

    m_pGmixAhub = nullptr;
    m_pAhub = nullptr;
}

void Ahub::SleepImpl() NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        return;
    }

    FinalizeImpl();
}

void Ahub::WakeImpl() NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        return;
    }

    InitializeImpl();
}

void Ahub::SetMute(bool isMute) NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        return;
    }

    const auto mode = isMute ? nne::audio::gmix::Session::Mode::Mute : nne::audio::gmix::Session::Mode::Default;
    m_pGmixAhub->SetMode(mode);
}

void Ahub::SetVolume(float volume, uint32_t curveDurationInMicroseconds) NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        return;
    }

    auto volumeForGmix = static_cast<int32_t>(volume * 256); // 1.0 = 256 in gmix

    NN_SDK_ASSERT_GREATER_EQUAL(volumeForGmix, 0);
    if(volumeForGmix < 0)
    {
        volumeForGmix = 0;
    }

    if(m_pGmixAhub)
    {
        m_pGmixAhub->SetVolume(volumeForGmix, curveDurationInMicroseconds);
    }
}

bool Ahub::IsInitialized() NN_NOEXCEPT
{
    return (GetSleepWakeState() == SleepWakePolicy::State::Initialized);
}

AhubIn::AhubIn() NN_NOEXCEPT
    : m_pGmixAhubIn(nullptr)
    , m_pAhubIn(nullptr)
    , m_Mutex(true)  // m_Mutex is locked recursively in case of "Prepare() -> Start()"
    , m_IsStarted(false)
{
}

void AhubIn::InitializeImpl() NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        return;
    }

    // NN_DETAIL_AUDIO_INFO("[audio] ahubin initialize\n");
    nne::audio::ahub::Initialize();

    //Start ahub in session
    nne::audio::gmix::OpenSession(
        &m_pGmixAhubIn,
        nne::audio::gmix::Session::Name::Ahub,
        1,
        nne::audio::gmix::Session::Name::NearVoice,
        nne::audio::gmix::Session::Format::Stereo,
        nne::audio::gmix::Session::Mode::Default);
    m_pGmixAhubIn->Start();

    // but keep in Mute to prevent noise.
    nne::audio::gmix::SetSessionVolume(
        nne::audio::gmix::Session::Name::Ahub,
        1,  // ahub in index = 1
        0,  // volume = 0
        0); // curve duration.
}

void AhubIn::FinalizeImpl() NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        return;
    }

    // NN_DETAIL_AUDIO_INFO("[audio] ahubin finalize\n");
    Stop();

    m_pGmixAhubIn->Stop();
    nne::audio::gmix::CloseSession(m_pGmixAhubIn);
    m_pGmixAhubIn = nullptr;

    nne::audio::ahub::Finalize();
}

void AhubIn::SleepImpl() NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        return;
    }

    FinalizeImpl();
}

void AhubIn::WakeImpl() NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        return;
    }

    InitializeImpl();
}

void AhubIn::Prepare() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (m_pAhubIn)
    {
        return;
    }

    // enable codec.
    nne::audio::ahub::OpenSession(&m_pAhubIn, 1);

    // un-mute session.
    m_pAhubIn->Start();
}

void AhubIn::Start() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (m_pAhubIn == nullptr)
    {
        return;
    }

    nne::audio::gmix::SetSessionVolume(
        nne::audio::gmix::Session::Name::Ahub,
        1,   // ahub in index = 1
        256, // volume = 1.0
        0);  // curve duration = 0

    if(!m_IsStarted)
    {
        AppletVolumeManager::SignalInputNotification();
    }
    m_IsStarted = true;
}

void AhubIn::Stop() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (m_pAhubIn == nullptr)
    {
        return;
    }

    // disable codec.
    m_pAhubIn->Stop();
    nne::audio::ahub::CloseSession(m_pAhubIn);
    m_pAhubIn = nullptr;

    // Mute session.
    nne::audio::gmix::SetSessionVolume(
        nne::audio::gmix::Session::Name::Ahub,
        1,   // ahub in index = 1
        0,   // volume = 0
        0);  // curve duration = 0

    if(m_IsStarted)
    {
        AppletVolumeManager::SignalInputNotification();
    }
    m_IsStarted = false;
}

bool AhubIn::IsStarted() NN_NOEXCEPT
{
    // Ensure invoking AppletVolumeManager::SignalInputNotification() and setting value to m_IsStarted atomically.
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    return m_IsStarted;
}

bool AhubIn::IsInitialized() NN_NOEXCEPT
{
    return (GetSleepWakeState() == SleepWakePolicy::State::Initialized);
}

nne::audio::gmix::Session* Ahub::GetGmixSession() NN_NOEXCEPT
{
    return m_pGmixAhub;
}

nn::os::Semaphore* Ahub::GetProcessSemaphore() NN_NOEXCEPT
{
    return &m_ProcessSemaphore;
}

// ================================================================================
// Hda
// ================================================================================

Hda::Hda() NN_NOEXCEPT
    : m_pGmixHda(nullptr)
    , m_pHda(nullptr)
    , m_ProcessSemaphore(0, 1)
    , m_IsConnected(false)
    , m_ChannelCount(ChannelCount_Invalid)
    , m_Initialized(false)
    , m_IsDeviceStarted(false)
    , m_IsGmixStarted(false)
    , m_OutputMode(nn::audioctrl::AudioOutputMode_Invalid)
{
}

void Hda::InitializeImpl() NN_NOEXCEPT
{
    auto hdaChannel = ShouldSurroundEnable() ?
        nne::audio::hda::AUDIO_HDA_NUMBEROFCHANNELS_6 :
        nne::audio::hda::AUDIO_HDA_NUMBEROFCHANNELS_2;
    nne::audio::hda::Initialize(
        nne::audio::hda::AUDIO_HDA_FORMAT_LPCM,
        nne::audio::hda::AUDIO_HDA_SAMPLERATE_48000,
        nne::audio::hda::AUDIO_HDA_BITSPERSAMPLE_16,
        hdaChannel);
    m_Initialized = true;
}

void Hda::Start() NN_NOEXCEPT
{
    if(!m_pGmixHda)
    {
        auto format = ShouldSurroundEnable() ?
            nne::audio::gmix::Session::Format::Surround :
            nne::audio::gmix::Session::Format::Stereo;
        nne::audio::gmix::OpenSession(
            &m_pGmixHda,
            nne::audio::gmix::Session::Name::Hda,
            0,
            nne::audio::gmix::Session::Name::Default,
            format,
            nne::audio::gmix::Session::Mode::Default);
    }
    if(!m_IsGmixStarted)
    {
        nne::audio::gmix::SetProcessSemaphore(nne::audio::gmix::Session::Name::Hda, m_ProcessSemaphore.GetBase());
        m_pGmixHda->Start();
        m_IsGmixStarted = true;
    }
    if(!m_pHda)
    {
        nne::audio::hda::OpenSession(&m_pHda, 0);
    }
    if(!m_IsDeviceStarted)
    {
        m_pHda->Start();
        m_IsDeviceStarted = true;
    }
}

void Hda::Stop() NN_NOEXCEPT
{
    if(m_IsDeviceStarted)
    {
        m_pHda->Stop();
        m_IsDeviceStarted = false;
    }
    if(m_IsGmixStarted)
    {
        m_pGmixHda->Stop();
        m_IsGmixStarted = false;
    }
    if(m_pHda)
    {
        nne::audio::hda::CloseSession(m_pHda);
        m_pHda = nullptr;
    }
    if(m_pGmixHda)
    {
        nne::audio::gmix::CloseSession(m_pGmixHda);
        m_pGmixHda = nullptr;
    }
}

void Hda::FinalizeImpl() NN_NOEXCEPT
{
    Stop();

    nne::audio::hda::Finalize();

    m_Initialized = false;
}

void Hda::SleepImpl() NN_NOEXCEPT
{
    FinalizeImpl();
}

void Hda::WakeImpl() NN_NOEXCEPT
{
    InitializeImpl();
}

void Hda::UpdateHotplugState() NN_NOEXCEPT
{
    if (GetSleepWakeState() != SleepWakePolicy::State::Initialized)
    {
        return;
    }

    int count = 0;
    nne::audio::hda::GetHotplugState(&m_IsConnected, &count);
    m_ChannelCount = (count >= 6) ? ChannelCount_6ch : ChannelCount_2ch;
}

bool Hda::ShouldSurroundEnable() const NN_NOEXCEPT
{
    return m_OutputMode == nn::audioctrl::AudioOutputMode_Pcm6ch || (m_OutputMode == nn::audioctrl::AudioOutputMode_PcmAuto && m_ChannelCount == ChannelCount_6ch);
}

void Hda::SetVolume(float volume, uint32_t curveDurationInMicroseconds) NN_NOEXCEPT
{
    auto volumeForGmix = static_cast<int32_t>(volume * 256); // 1.0 = 256 in gmix

    NN_SDK_ASSERT_GREATER_EQUAL(volumeForGmix, 0);
    if(volumeForGmix < 0)
    {
        volumeForGmix = 0;
    }

    if(m_pGmixHda)
    {
        m_pGmixHda->SetVolume(volumeForGmix, curveDurationInMicroseconds);
    }
}

void Hda::SetOutputMode(nn::audioctrl::AudioOutputMode mode) NN_NOEXCEPT
{
    m_OutputMode = mode;
}

bool Hda::IsConnected() const NN_NOEXCEPT
{
    return m_IsConnected;
}

nne::audio::gmix::Session* Hda::GetGmixSession() const NN_NOEXCEPT
{
    return m_pGmixHda;
}

nn::os::Semaphore* Hda::GetProcessSemaphore() NN_NOEXCEPT
{
    return &m_ProcessSemaphore;
}

void Hda::StopDevice() NN_NOEXCEPT
{
    if(m_pHda)
    {
        m_pHda->Stop();
    }
    m_IsDeviceStarted = false;
}

// ================================================================================
// Codec
// ================================================================================

NN_DEFINE_STATIC_CONSTANT(const int Codec::VolumeLevelMin);
NN_DEFINE_STATIC_CONSTANT(const int Codec::VolumeLevelMax);
NN_DEFINE_STATIC_CONSTANT(const int Codec::HeadphoneAmplifierGainMin);
NN_DEFINE_STATIC_CONSTANT(const int Codec::HeadphoneAmplifierGainMax);

// TODO: Hoag 対応時に Icosa, Cooper, Hoag 毎の実装クラスを作成する (SIGLO-62303)
Codec::Codec() NN_NOEXCEPT
    : m_StateCheckMutex(false),
      m_NneAudioCodecMutex(false)
{
}

void Codec::InitializeImpl() NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        return;
    }

    std::lock_guard<nn::os::Mutex> stateCheckLock(m_StateCheckMutex);

    {
        std::lock_guard<nn::os::Mutex> codecHardwareLock(m_NneAudioCodecMutex);
        nne::audio::codec::Initialize();
    }

    // GPIO ライブラリの初期化
    nn::gpio::Initialize();

    // ヘッドホン挿抜検出 GPIO の初期化
    nn::gpio::OpenSession(&m_HeadphoneMicJackGpioSession, nn::gpio::GpioPadName_CodecHpDetIrq);
    nn::gpio::SetDirection(&m_HeadphoneMicJackGpioSession, nn::gpio::Direction_Input);
    nn::gpio::SetInterruptMode(&m_HeadphoneMicJackGpioSession, nn::gpio::InterruptMode_AnyEdge);

    // ヘッドホン挿抜検出 GPIO のチャタリング防止機能の設定
    nn::gpio::SetDebounceEnabled(&m_HeadphoneMicJackGpioSession, true);
    nn::gpio::SetDebounceTime(&m_HeadphoneMicJackGpioSession, 128);
}

void Codec::FinalizeImpl() NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        return;
    }
    std::lock_guard<nn::os::Mutex> stateCheckLock(m_StateCheckMutex);

    // ヘッドホン挿抜検出 GPIO の終了
    nn::gpio::CloseSession(&m_HeadphoneMicJackGpioSession);

    // GPIO ライブラリの終了
    nn::gpio::Finalize();

    {
        std::lock_guard<nn::os::Mutex> codecHardwareLock(m_NneAudioCodecMutex);
        nne::audio::codec::Finalize();
    }
}

void Codec::SleepImpl() NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        return;
    }

    std::lock_guard<nn::os::Mutex> lock(m_NneAudioCodecMutex);

    nne::audio::codec::Sleep();
}

void Codec::WakeImpl() NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        return;
    }

    std::lock_guard<nn::os::Mutex> lock(m_NneAudioCodecMutex);

    nne::audio::codec::Wake();
}

bool Codec::IsInitialized() NN_NOEXCEPT
{
    return (GetSleepWakeState() == SleepWakePolicy::State::Initialized);
}

void Codec::Abort() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_NneAudioCodecMutex);

    nne::audio::codec::Abort();
}

nn::os::SystemEvent* Codec::RegisterJackEvent() NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::ResultNotSupported());
    }

    std::lock_guard<nn::os::Mutex> lock(m_StateCheckMutex);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::gpio::BindInterrupt(m_HeadphoneMicJackGpioEvent.GetBase(), &m_HeadphoneMicJackGpioSession));
    nn::gpio::ClearInterruptStatus(&m_HeadphoneMicJackGpioSession);
    nn::gpio::SetInterruptEnable(&m_HeadphoneMicJackGpioSession, true);

    m_HeadphoneMicJackGpioEvent.Clear();
    return &m_HeadphoneMicJackGpioEvent;
}

void Codec::UnregisterJackEvent() NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::ResultNotSupported());
    }

    std::lock_guard<nn::os::Mutex> lock(m_StateCheckMutex);
    nn::gpio::SetInterruptEnable(&m_HeadphoneMicJackGpioSession, false);
    nn::gpio::ClearInterruptStatus(&m_HeadphoneMicJackGpioSession);
    nn::gpio::UnbindInterrupt(&m_HeadphoneMicJackGpioSession);
}

void Codec::ClearJackEvent() NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::ResultNotSupported());
    }

    std::lock_guard<nn::os::Mutex> lock(m_StateCheckMutex);
    nn::gpio::ClearInterruptStatus(&m_HeadphoneMicJackGpioSession);
    m_HeadphoneMicJackGpioEvent.Clear();
    nn::gpio::SetInterruptEnable(&m_HeadphoneMicJackGpioSession, true);
}

bool Codec::IsJackPlugged() NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::ResultNotSupported());
    }

    std::lock_guard<nn::os::Mutex> lock(m_StateCheckMutex);
    return (nn::gpio::GetValue(&m_HeadphoneMicJackGpioSession) == nn::gpio::GpioValue_Low);;
}

void Codec::SetSpeakerHeadphoneVolume(int32_t volume) NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::ResultNotSupported());
    }

    std::lock_guard<nn::os::Mutex> lock(m_NneAudioCodecMutex);

    NN_SDK_ASSERT(VolumeLevelMin <= volume && volume <= VolumeLevelMax);
    nne::audio::codec::SetOutVolume(static_cast<uint8_t>(volume));
}

void Codec::SetHeadphoneAmplifierGain(int32_t gain) NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::ResultNotSupported());
    }

    std::lock_guard<nn::os::Mutex> lock(m_NneAudioCodecMutex);
    NN_SDK_ASSERT_MINMAX(gain, HeadphoneAmplifierGainMin, HeadphoneAmplifierGainMax);

    nne::audio::codec::SetHeadphoneAmplifierGain(static_cast<uint8_t>(gain));
}

void Codec::SetSpeakerMute(bool mute) NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::ResultNotSupported());
    }


    if(IsSpeakerMute() != mute)
    {
        std::lock_guard<nn::os::Mutex> lock(m_NneAudioCodecMutex);

        nne::audio::codec::SetSpeakerMute(mute);
    }
}

bool Codec::IsSpeakerMute() NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::ResultNotSupported());
    }

    std::lock_guard<nn::os::Mutex> lock(m_NneAudioCodecMutex);

    return nne::audio::codec::IsSpeakerMute();
}

bool Codec::IsSpeakerOverCurrent() NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::ResultNotSupported());
    }

    std::lock_guard<nn::os::Mutex> lock(m_NneAudioCodecMutex);

    return nne::audio::codec::IsSpeakerOverCurrent();
}

bool Codec::IsOverTemperature() NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::ResultNotSupported());
    }

    std::lock_guard<nn::os::Mutex> lock(m_NneAudioCodecMutex);

    return nne::audio::codec::IsOverTemperature();
}

bool Codec::IsMicBiasOverCurrent() NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::ResultNotSupported());
    }

    std::lock_guard<nn::os::Mutex> lock(m_NneAudioCodecMutex);

    return nne::audio::codec::IsMicBiasOverCurrent();
}

void Codec::DumpFactorySettings() NN_NOEXCEPT
{
    // 備考：工程書き込みされるコーデック IC の設定値について
    // 工程書き込みされるコーデック IC の設定値は不備があるデータが書かれているので使用せず、ファームウェアで抱えている値を現状では利用する。
    //
    // ボードリビジョン毎の書かれている値：
    // SDEV PreMP-1, SDEV PreMP-2, DP1, FinalDP, SDEV MP   空文字
    // EDEV MP, FinalDP2, Prod MP                          不備があるパラメータ
    // (http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=92328451 を参照)
    //
    // もし使用することになった時は、ボードリビジョンと較正値バージョンを基に「ファームウェアが抱えている値を使用」する場合と、
    // 「工程書き込みされた値を使用」する場合で動的に分岐する必要がある。

    nn::settings::factory::SpeakerParameter parameter;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::settings::factory::GetSpeakerParameter(&parameter));
    NN_DETAIL_AUDIO_INFO("version                    0x%04x\n", nn::util::LoadBigEndian(&parameter.version));                    // 較正値バージョン
    NN_DETAIL_AUDIO_INFO("speakerHpf2A1              0x%04x\n", nn::util::LoadBigEndian(&parameter.speakerHpf2A1));              // HPF2 a1
    NN_DETAIL_AUDIO_INFO("speakerHpf2A2              0x%04x\n", nn::util::LoadBigEndian(&parameter.speakerHpf2A2));              // HPF2 a2
    NN_DETAIL_AUDIO_INFO("speakerHpf2H0              0x%04x\n", nn::util::LoadBigEndian(&parameter.speakerHpf2H0));              // HPF2 h0
    NN_DETAIL_AUDIO_INFO("speakerEqInputVolume       0x%04x\n", nn::util::LoadBigEndian(&parameter.speakerEqInputVolume));       // EQ Input Volume
    NN_DETAIL_AUDIO_INFO("speakerEqOutputVolume      0x%04x\n", nn::util::LoadBigEndian(&parameter.speakerEqOutputVolume));      // EQ Output Volume
    NN_DETAIL_AUDIO_INFO("speakerEqCtrl1             0x%04x\n", nn::util::LoadBigEndian(&parameter.speakerEqCtrl1));             // EQ 設定 1（ON/OFF, type）
    NN_DETAIL_AUDIO_INFO("speakerEqCtrl2             0x%04x\n", nn::util::LoadBigEndian(&parameter.speakerEqCtrl2));             // EQ 設定 2（path）
    NN_DETAIL_AUDIO_INFO("speakerDrcAgcCtrl2         0x%04x\n", nn::util::LoadBigEndian(&parameter.speakerDrcAgcCtrl2));         // DRC Pre/Post-Boost Gain
    NN_DETAIL_AUDIO_INFO("speakerDrcAgcCtrl3         0x%04x\n", nn::util::LoadBigEndian(&parameter.speakerDrcAgcCtrl3));         // DRC Limiter Level
    NN_DETAIL_AUDIO_INFO("speakerDrcAgcCtrl1         0x%04x\n", nn::util::LoadBigEndian(&parameter.speakerDrcAgcCtrl1));         // DRC Attack/Recovery Rate
    NN_DETAIL_AUDIO_INFO("speakerAnalogVolume        0x%04x\n", nn::util::LoadBigEndian(&parameter.speakerAnalogVolume));        // スピーカーのアナログボリューム値
    NN_DETAIL_AUDIO_INFO("headphoneAnalogVolume      0x%04x\n", nn::util::LoadBigEndian(&parameter.headphoneAnalogVolume));      // ヘッドホンのアナログボリューム値
    NN_DETAIL_AUDIO_INFO("speakerDigitalVolumeMin    0x%04x\n", nn::util::LoadBigEndian(&parameter.speakerDigitalVolumeMin));    // スピーカー出力時のデジタルボリュームの最小値
    NN_DETAIL_AUDIO_INFO("speakerDigitalVolumeMax    0x%04x\n", nn::util::LoadBigEndian(&parameter.speakerDigitalVolumeMax));    // スピーカー出力時のデジタルボリュームの最大値
    NN_DETAIL_AUDIO_INFO("headphoneDigitalVolumeMin  0x%04x\n", nn::util::LoadBigEndian(&parameter.headphoneDigitalVolumeMin));  // ヘッドホン出力時のデジタルボリュームの最小値
    NN_DETAIL_AUDIO_INFO("headphoneDigitalVolumeMax  0x%04x\n", nn::util::LoadBigEndian(&parameter.headphoneDigitalVolumeMax));  // ヘッドホン出力時のデジタルボリュームの最大値
    NN_DETAIL_AUDIO_INFO("micFixedGain               0x%04x\n", nn::util::LoadBigEndian(&parameter.micFixedGain));               // マイクの固定ゲイン値
    NN_DETAIL_AUDIO_INFO("micVariableVolumeMin       0x%04x\n", nn::util::LoadBigEndian(&parameter.micVariableVolumeMin));       // マイクの可変ボリュームの最小値
    NN_DETAIL_AUDIO_INFO("micVariableVolumeMax       0x%04x\n", nn::util::LoadBigEndian(&parameter.micVariableVolumeMax));       // マイクの可変ボリュームの最大値
}

CodecErrorDetector::CodecErrorDetector(Codec& codec, AhubIn& ahubIn) NN_NOEXCEPT
    : m_Codec(codec)
    , m_AhubIn(ahubIn)
    , m_SpeakerOverCurrentPollingTimerEvent(nn::os::EventClearMode_ManualClear)
    , m_MicBiasDebounceTimerEvent(nn::os::EventClearMode_ManualClear)
    , m_IsInDebouncePeriod(false)
    , m_IsJackStateChanged(false)
    , m_IsMicBiasOverCurrentDetectionEnabled(true)
    , m_RequestSleepEvent(nn::os::EventClearMode_AutoClear)
    , m_RequestWakeEvent(nn::os::EventClearMode_AutoClear)
    , m_RequestExitEvent(nn::os::EventClearMode_AutoClear)
    , m_SleepWakeRequestCompleteEvent(nn::os::EventClearMode_AutoClear)
{
}

void CodecErrorDetector::DetectionTreadFunc() NN_NOEXCEPT
{
    OpenCodecAlertGpioSession();

    // Start SpeakerOverCurrent polling.
    m_SpeakerOverCurrentPollingTimerEvent.StartPeriodic(GetPollingTimerEventInterval(), GetPollingTimerEventInterval());
    m_SpeakerOverCurrentPollingTimerEvent.Clear();
    m_MicBiasDebounceTimerEvent.Clear();

    for (;;)
    {
        // This enum's sequence must be same as the sequence of WaitAny argument.
        enum class EventIndex
        {
            CodecAlert = 0,
            SpeakerOverCurrentPollingTimer,
            MicBiasDebounceTimer,
            RequestSleep,
            RequestExit,
            SystemFatalError,
        };

        auto index = static_cast<EventIndex>(nn::os::WaitAny(
            m_CodecAlertGpioEvent.GetBase(),
            m_SpeakerOverCurrentPollingTimerEvent.GetBase(),
            m_MicBiasDebounceTimerEvent.GetBase(),
            m_RequestSleepEvent.GetBase(),
            m_RequestExitEvent.GetBase(),
            m_SystemFatalErrorEvent.GetBase()));

        if ((index == EventIndex::CodecAlert) ||
            (index == EventIndex::SpeakerOverCurrentPollingTimer) ||
            (index == EventIndex::MicBiasDebounceTimer))
        {
            switch (index)
            {
            case EventIndex::CodecAlert:
            {
                m_CodecAlertGpioEvent.Clear();

                // 割り込み状態をクリア
                nn::gpio::ClearInterruptStatus(&codecAlertGpioSession);

                // 割り込みを再度有効にする (割り込みイベントがシグナル後、割り込み禁止状態になる)
                nn::gpio::SetInterruptEnable(&codecAlertGpioSession, true);
                CodecAlertHandler();
            }
            break;
            case EventIndex::SpeakerOverCurrentPollingTimer:
            {
                m_SpeakerOverCurrentPollingTimerEvent.Clear();
                SpeakerOverCurrentErrorHandler();
            }
            break;
            case EventIndex::MicBiasDebounceTimer:
            {
                m_MicBiasDebounceTimerEvent.Clear();
                MicOverCurrentErrorHandler();
            }
            break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
        else if (index == EventIndex::RequestSleep)
        {
            // NN_DETAIL_AUDIO_INFO("[audio] RequestSleep event.\n");
            m_RequestSleepEvent.Clear();

            m_SleepWakeRequestCompleteEvent.Signal(); // スリープ要求が完了したことを通知
            m_RequestWakeEvent.Wait(); // ウェイク要求が来るまで待機
            m_SleepWakeRequestCompleteEvent.Signal(); // ウェイク要求が完了したことを通知
        }
        else if (index == EventIndex::RequestExit)
        {
            // NN_DETAIL_AUDIO_INFO("[audio] RequestExit event.\n");
            m_RequestExitEvent.Clear();
            break;
        }
        else if (index == EventIndex::SystemFatalError)
        {
            m_SystemFatalErrorEvent.Clear();
            SystemFatalErrorHandler();
        }
        else
        {
            NN_SDK_ASSERT(false, "Unexpected index.");
        }
    }
    CloseCodecAlertGpioSession();
}

void CodecErrorDetector::OpenCodecAlertGpioSession() NN_NOEXCEPT
{
    // GPIO セッションを作成し CODEC_ALERT の割り込みを受け取れるように設定
    nn::gpio::OpenSession(&codecAlertGpioSession, nn::gpio::GpioPadName_CodecAlert);
    nn::gpio::SetDirection(&codecAlertGpioSession, nn::gpio::Direction_Input);
    nn::gpio::SetInterruptMode(&codecAlertGpioSession, nn::gpio::InterruptMode_RisingEdge);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::gpio::BindInterrupt(m_CodecAlertGpioEvent.GetBase(), &codecAlertGpioSession));
    nn::gpio::ClearInterruptStatus(&codecAlertGpioSession);
    nn::gpio::SetInterruptEnable(&codecAlertGpioSession, true);
}

void CodecErrorDetector::CloseCodecAlertGpioSession() NN_NOEXCEPT
{
    // GPIO セッションの終了処理
    nn::gpio::SetInterruptEnable(&codecAlertGpioSession, false);
    nn::gpio::ClearInterruptStatus(&codecAlertGpioSession);
    nn::gpio::UnbindInterrupt(&codecAlertGpioSession);
    nn::gpio::CloseSession(&codecAlertGpioSession);
}

void CodecErrorDetector::EnableJackStateFlag() NN_NOEXCEPT
{
    m_IsJackStateChanged = true;
}

void CodecErrorDetector::TriggerMicCheck() NN_NOEXCEPT
{
    if (m_IsInDebouncePeriod == false)
    {
        m_IsInDebouncePeriod = true;
        m_MicBiasDebounceTimerEvent.StartOneShot(GetMicBiasOverCurrentDetectionDelayTime());
    }
}

void CodecErrorDetector::SetMicBiasOverCurrentDetectionEnabled(bool isEnabled) NN_NOEXCEPT
{
    m_IsMicBiasOverCurrentDetectionEnabled = isEnabled;
}

void CodecErrorDetector::CodecAlertHandler() NN_NOEXCEPT
{
    if (m_Codec.IsOverTemperature())
    {
        NN_DETAIL_AUDIO_FATAL("[audio:CODEC IC] OverTemperature is detected.");
        // 何らかの理由による CODEC IC の異常発熱。電源を落として fatal error。
        m_Codec.Abort();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::ResultCodecOverTemperature());
    }

    if (m_Codec.IsMicBiasOverCurrent())
    {
        if (m_IsInDebouncePeriod || // MicBiasOverCurrent のチャタリング防止タイマーが既に開始済みだったら無視する
            (m_IsJackStateChanged == false)) // Jack 状態に変化が無ければ無視する （Mic Mute 時に短絡するケースへの対策 | SIGLO-34405
        {
            return;
        }

        // MicBiasOverCurrent のチャタリング防止タイマーを開始
        m_IsInDebouncePeriod = true;
        m_MicBiasDebounceTimerEvent.StartOneShot(GetMicBiasOverCurrentDetectionDelayTime());
    }
}

void CodecErrorDetector::SpeakerOverCurrentErrorHandler() NN_NOEXCEPT
{
    if (m_Codec.IsSpeakerOverCurrent())
    {
        NN_DETAIL_AUDIO_FATAL("[audio:CODEC IC] SpeakerOverCurrent is detected.");
        // Speaker 短絡の疑い。電源を落として fatal error。
        m_Codec.Abort();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::ResultCodecSpeakerOverCurrent());
    }
}

void CodecErrorDetector::MicOverCurrentErrorHandler() NN_NOEXCEPT
{
    m_IsInDebouncePeriod = false;
    m_IsJackStateChanged = false;

    if (m_Codec.IsMicBiasOverCurrent() && m_IsMicBiasOverCurrentDetectionEnabled)
    {
        // NN_DETAIL_AUDIO_TRACE("[audio:CODEC IC] MicBiasOverCurrent is detected.\n");
        m_AhubIn.Stop();
    }
    else
    {
        m_AhubIn.Start();
    }
}

NN_NORETURN
void CodecErrorDetector::SystemFatalErrorHandler() NN_NOEXCEPT
{
    for(;;)
    {
        NN_DETAIL_AUDIO_FATAL("[audio] Fatal error has been detected.\n");
        nn::os::SleepThread(nn::TimeSpan::FromMinutes(1));
    }

    // never return from this function.
}

void CodecErrorDetector::InitializeImpl() NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        return;
    }

    // TODO: HDMI の音声出力の停止などを行うために、将来的には別の場所へ移動する (SIGLO-39474)
    // Fatal エラー発生時に通知を受け取るシステムイベントを取得
    nn::fatal::GetFatalEvent(&m_SystemFatalErrorEvent);

    // CODEC IC の異常状態をハンドリングするスレッドを作成
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &m_CodecAlertHandlerThread,
        CodecErrorDetector::HandlerThreadFunc,
        this,
        g_CodecAlertHandlerThreadStack,
        sizeof(g_CodecAlertHandlerThreadStack),
        NN_SYSTEM_THREAD_PRIORITY(audio, CodecAlertHandler)
    ));
    nn::os::SetThreadNamePointer(&m_CodecAlertHandlerThread, NN_SYSTEM_THREAD_NAME(audio, CodecAlertHandler));
    nn::os::StartThread(&m_CodecAlertHandlerThread);
}

void CodecErrorDetector::FinalizeImpl() NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        return;
    }

    // CODEC IC の異常状態をハンドリングするスレッドを終了・破棄
    m_RequestExitEvent.Signal();
    nn::os::WaitThread(&m_CodecAlertHandlerThread);
    nn::os::DestroyThread(&m_CodecAlertHandlerThread);
}

void CodecErrorDetector::SleepImpl() NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        return;
    }

    m_RequestSleepEvent.Signal(); // スリープ要求イベントをシグナル
    m_SleepWakeRequestCompleteEvent.Wait(); // 要求完了を待機
}

void CodecErrorDetector::WakeImpl() NN_NOEXCEPT
{
    if (!m_HasCodecIc)
    {
        return;
    }

    m_RequestWakeEvent.Signal(); // ウェイク要求イベントをシグナル
    m_SleepWakeRequestCompleteEvent.Wait(); // 要求完了を待機
}

// ================================================================================
// AudioDeviceController
// ================================================================================

AudioDeviceController::AudioDeviceController() NN_NOEXCEPT
    : m_HotplugEvent(nn::os::EventClearMode_ManualClear)
    , m_OutputDeviceControlEvent(nn::os::EventClearMode_ManualClear)
    , m_DeviceFadeEvent(nn::os::EventClearMode_ManualClear)
    , m_HeadphoneSelectEvent(nn::os::EventClearMode_ManualClear)
    , m_UacSpeakerSelectEvent(nn::os::EventClearMode_ManualClear)
    , m_ExitRequestEvent(nn::os::EventClearMode_ManualClear)
    , m_CodecErrorDetector(m_CodecDevice, m_AhubInDevice)
    , m_DeviceFadeInTime(0)
    , m_DeviceFadeOutTime(0)
    , m_RequestedDevice(Device::Invalid)
    , m_SelectedDevice(Device::Invalid)
    , m_IsInputDeviceForceEnabled(false)
    , m_OutputModes{
        audioctrl::AudioOutputMode_Invalid, // Invalid
        audioctrl::AudioOutputMode_Invalid, // Speaker
        audioctrl::AudioOutputMode_Invalid, // Headphone
        audioctrl::AudioOutputMode_Invalid, // Tv
        audioctrl::AudioOutputMode_Invalid, // UsbOutputDevice
    }
{
}

void AudioDeviceController::OutputControlThreadFunc() NN_NOEXCEPT
{
    nne::audio::hda::RegisterHotplugEvent(m_HotplugEvent.GetBase());
    nn::os::MultiWaitType           multiWait;
    nn::os::MultiWaitHolderType     requestApiHolder;
    nn::os::MultiWaitHolderType     hotplugHolder;
    nn::os::MultiWaitHolderType     headphoneHolder;
    nn::os::MultiWaitHolderType     uacSpeakerHolder;
    nn::os::MultiWaitHolderType     requestExitHolder;

    nn::os::EventType* events[EventIndex_Count] = { nullptr };
    events[EventIndex_RequestViaApi] = m_OutputDeviceControlEvent.GetBase();
    events[EventIndex_Hotplug] = m_HotplugEvent.GetBase();
    events[EventIndex_HeadphoneSelect] = m_HeadphoneSelectEvent.GetBase();
    events[EventIndex_UacSpeakerSelect] = m_UacSpeakerSelectEvent.GetBase();
    events[EventIndex_ExitRequest] = m_ExitRequestEvent.GetBase();


    // セマフォ用多重待ちホルダー
    nn::os::InitializeMultiWaitHolder( &requestApiHolder, m_OutputDeviceControlEvent.GetBase() );
    nn::os::SetMultiWaitHolderUserData( &requestApiHolder, static_cast<uintptr_t>(EventIndex_RequestViaApi) );

    // イベント用多重待ちホルダー
    nn::os::InitializeMultiWaitHolder( &hotplugHolder, m_HotplugEvent.GetBase() );
    nn::os::SetMultiWaitHolderUserData( &hotplugHolder, static_cast<uintptr_t>(EventIndex_Hotplug) );

    // メッセージキュー用用多重待ちホルダー
    nn::os::InitializeMultiWaitHolder( &headphoneHolder, m_HeadphoneSelectEvent.GetBase() );
    nn::os::SetMultiWaitHolderUserData( &headphoneHolder, static_cast<uintptr_t>(EventIndex_HeadphoneSelect) );
    nn::os::InitializeMultiWaitHolder( &uacSpeakerHolder, m_UacSpeakerSelectEvent.GetBase() );
    nn::os::SetMultiWaitHolderUserData( &uacSpeakerHolder, static_cast<uintptr_t>(EventIndex_UacSpeakerSelect) );

    // スレッド用多重待ちホルダー
    nn::os::InitializeMultiWaitHolder( &requestExitHolder, m_ExitRequestEvent.GetBase() );
    nn::os::SetMultiWaitHolderUserData( &requestExitHolder, static_cast<uintptr_t>(EventIndex_ExitRequest) );

    // 多重待ちオブジェクトの初期化とリンク
    nn::os::InitializeMultiWait( &multiWait );
    nn::os::LinkMultiWaitHolder( &multiWait, &requestApiHolder );
    nn::os::LinkMultiWaitHolder( &multiWait, &hotplugHolder );
    nn::os::LinkMultiWaitHolder( &multiWait, &headphoneHolder );
    nn::os::LinkMultiWaitHolder( &multiWait, &uacSpeakerHolder );
    nn::os::LinkMultiWaitHolder( &multiWait, &requestExitHolder );

    for(;;)
    {
        nn::os::MultiWaitHolderType* holder = nn::os::TimedWaitAny(&multiWait, ThreadTimeOut);
        if (holder == nullptr)
        {
            continue;
        }

        auto idx = nn::os::GetMultiWaitHolderUserData(holder);
        nn::os::ClearEvent(events[idx]);

        if(idx == EventIndex_ExitRequest)
        {
            break;
        }
        m_RequestedDevice = GetRequestedDevice();
        if (m_RequestedDevice == Device::Invalid)
        {
            continue;
        }

        auto fade = IsFadeRequired();
        if (fade)
        {
            nn::audio::AppletVolumeManager::StartDeviceFadeOut(&m_DeviceFadeEvent, m_DeviceFadeOutTime.GetNanoSeconds());
            m_DeviceFadeEvent.Wait();
            m_DeviceFadeEvent.Clear();
        }

        bool okToSkipUpdatingDevices =
            (m_RequestedDevice == m_SelectedDevice) &&
            ((m_SelectedDevice == Device::AhubHeadphone) || (m_SelectedDevice == Device::AhubSpeaker));
        if (okToSkipUpdatingDevices == false)
        {
            m_HdaDevice.UpdateHotplugState();
            SwitchOutputDevice(m_RequestedDevice);
        }
        SwitchDownmixMode(m_RequestedDevice);

        if (fade)
        {
            nn::audio::AppletVolumeManager::StartDeviceFadeIn(&m_DeviceFadeEvent, m_DeviceFadeInTime.GetNanoSeconds());
            m_DeviceFadeEvent.Wait();
            m_DeviceFadeEvent.Clear();
        }
    }
    nn::os::UnlinkMultiWaitHolder(&requestApiHolder);
    nn::os::UnlinkMultiWaitHolder(&hotplugHolder);
    nn::os::UnlinkMultiWaitHolder(&headphoneHolder);
    nn::os::UnlinkMultiWaitHolder(&uacSpeakerHolder);
    nn::os::UnlinkMultiWaitHolder(&requestExitHolder);

    nn::os::FinalizeMultiWaitHolder(&requestApiHolder);
    nn::os::FinalizeMultiWaitHolder(&hotplugHolder);
    nn::os::FinalizeMultiWaitHolder(&headphoneHolder);
    nn::os::FinalizeMultiWaitHolder(&uacSpeakerHolder);
    nn::os::FinalizeMultiWaitHolder(&requestExitHolder);

    nn::os::FinalizeMultiWait( &multiWait );
} // NOLINT(impl/function_size)

nn::audio::server::detail::AudioDeviceController::Device AudioDeviceController::GetRequestedDevice() const NN_NOEXCEPT
{
    return m_ForcedTarget != nn::audioctrl::AudioTarget_Invalid ? ConvertToDevice(m_ForcedTarget) :
           m_IsUacSpeakerSelected ? AudioDeviceController::Device:: UacSpeaker :
           m_IsHeadphoneOutSelected ? AudioDeviceController::Device::AhubHeadphone :
           m_IsHdaConnectionRequested ? AudioDeviceController::Device::Hda : AudioDeviceController::Device::AhubSpeaker;
}

AudioDeviceController::Device AudioDeviceController::ConvertToDevice(nn::audioctrl::AudioTarget target) const NN_NOEXCEPT
{
    switch(target)
    {
    case nn::audioctrl::AudioTarget_Speaker:
        return AudioDeviceController::Device::AhubSpeaker;
    case nn::audioctrl::AudioTarget_Headphone:
        return AudioDeviceController::Device::AhubHeadphone;
    case nn::audioctrl::AudioTarget_Tv:
        return AudioDeviceController::Device::Hda;
    case nn::audioctrl::AudioTarget_UsbOutputDevice:
        return AudioDeviceController::Device::UacSpeaker;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

bool AudioDeviceController::IsHdaRequested() const NN_NOEXCEPT
{
    return m_IsHdaConnectionRequested;
}

bool AudioDeviceController::IsFadeRequired() const NN_NOEXCEPT
{
    switch (m_SelectedDevice)
    {
    case Device::Hda:
        if ((m_RequestedDevice == Device::AhubSpeaker) ||
            (m_RequestedDevice == Device::AhubHeadphone))
        {
            return true;
        }
        break;
    case Device::AhubHeadphone:
    case Device::AhubSpeaker:
        if (m_RequestedDevice == Device::Hda)
        {
            return true;
        }
        break;
    case Device::UacSpeaker:
        return true;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    return false;
}

nn::audio::server::detail::Codec* AudioDeviceController::GetCodecDevice() NN_NOEXCEPT
{
    return (m_CodecDevice.IsInitialized()) ? &m_CodecDevice : nullptr;
}

nn::audio::server::detail::Ahub* AudioDeviceController::GetAhubDevice() NN_NOEXCEPT
{
    return (m_AhubDevice.IsInitialized()) ? &m_AhubDevice : nullptr;
}

nn::audio::server::detail::AhubIn* AudioDeviceController::GetAhubInDevice() NN_NOEXCEPT
{
    return (m_AhubInDevice.IsInitialized()) ? &m_AhubInDevice : nullptr;
}

void AudioDeviceController::SwitchOutputDevice(Device device) NN_NOEXCEPT
{
    switch (device)
    {
    case Device::Hda:
        m_HdaDevice.Finalize();
        m_HdaDevice.SetOutputMode(m_OutputModes[nn::audioctrl::AudioTarget_Tv]);
        m_HdaDevice.Initialize(m_HasCodecIc, m_HasHdmi);
        m_HdaDevice.Start();
        nne::audio::hda::RegisterHotplugEvent(m_HotplugEvent.GetBase());
        nne::audio::gmix::SetDefaultDeviceOut(nne::audio::gmix::Session::Name::Hda);
        m_SelectedDevice = Device::Hda;
        nn::audioctrl::server::detail::NotifyAudioOutputTarget(nn::audioctrl::AudioTarget_Tv);
        ::nn::audio::detail::UacManager::GetInstance().SetSpeakerSource(nullptr, nullptr);
        if(m_HasCodecIc)
        {
            m_CodecDevice.SetSpeakerHeadphoneVolume(0);
            m_CodecDevice.SetSpeakerMute(false);
        }
        break;
    case Device::AhubHeadphone:
        nne::audio::gmix::SetDefaultDeviceOut(nne::audio::gmix::Session::Name::Ahub);
        ::nn::audio::detail::UacManager::GetInstance().SetSpeakerSource(m_AhubDevice.GetGmixSession(), m_AhubDevice.GetProcessSemaphore());
        m_SelectedDevice = Device::AhubHeadphone;
        nn::audioctrl::server::detail::NotifyAudioOutputTarget(nn::audioctrl::AudioTarget_Headphone);
        ::nn::audio::detail::UacManager::GetInstance().SetSpeakerSource(nullptr, nullptr);
        m_HdaDevice.Stop();
        if(m_HasCodecIc)
        {
            m_CodecDevice.SetSpeakerHeadphoneVolume(0);
            m_CodecDevice.SetSpeakerMute(true);
            SetCodecHeadphoneVolume(m_CodecHeadphoneVolume);
        }
        break;
    case Device::AhubSpeaker:
        nne::audio::gmix::SetDefaultDeviceOut(nne::audio::gmix::Session::Name::Ahub);
        ::nn::audio::detail::UacManager::GetInstance().SetSpeakerSource(m_AhubDevice.GetGmixSession(), m_AhubDevice.GetProcessSemaphore());
        m_SelectedDevice = Device::AhubSpeaker;
        nn::audioctrl::server::detail::NotifyAudioOutputTarget(nn::audioctrl::AudioTarget_Speaker);
        ::nn::audio::detail::UacManager::GetInstance().SetSpeakerSource(nullptr, nullptr);
        m_HdaDevice.Stop();
        if(m_HasCodecIc)
        {
            m_CodecDevice.SetSpeakerHeadphoneVolume(0);
            m_CodecDevice.SetSpeakerMute(false);
            SetCodecSpeakerVolume(m_CodecSpeakerVolume);
        }
        break;
    case Device::UacSpeaker:
        m_HdaDevice.Finalize();
        m_HdaDevice.SetOutputMode(m_OutputModes[nn::audioctrl::AudioTarget_UsbOutputDevice]);
        m_HdaDevice.Initialize(m_HasCodecIc, m_HasHdmi);
        m_HdaDevice.Start();
        m_HdaDevice.StopDevice();
        nne::audio::hda::RegisterHotplugEvent(m_HotplugEvent.GetBase());
        m_SelectedDevice = Device::UacSpeaker;
        nn::audioctrl::server::detail::NotifyAudioOutputTarget(nn::audioctrl::AudioTarget_UsbOutputDevice);
        nne::audio::gmix::SetDefaultDeviceOut(nne::audio::gmix::Session::Name::Hda);
        ::nn::audio::detail::UacManager::GetInstance().SetSpeakerSource(m_HdaDevice.GetGmixSession(), m_HdaDevice.GetProcessSemaphore());
        if(m_HasCodecIc)
        {
            m_CodecDevice.SetSpeakerHeadphoneVolume(0);
            m_CodecDevice.SetSpeakerMute(false);
        }
        break;
    case Device::Invalid:
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    // Set OutputMaterVolume to Ahub gmix session and HDA gmix session
    SetOutputMasterVolume(m_OutputMasterVolume, 0);
}

void AudioDeviceController::SwitchDownmixMode(Device device) NN_NOEXCEPT
{
    int channelCount = 0;
    auto deviceType = nn::audio::AppletVolumeManager::DeviceType_Count;
    auto target = nn::audioctrl::AudioTarget_Invalid;
    switch (device)
    {
    case Device::Hda:
        channelCount = m_HdaDevice.ShouldSurroundEnable() ? 6 : 2;
        deviceType = nn::audio::AppletVolumeManager::DeviceType_Hda;
        target = nn::audioctrl::AudioTarget_Tv;
        break;
    case Device::AhubHeadphone:
        channelCount = 2;
        deviceType = nn::audio::AppletVolumeManager::DeviceType_AhubHeadphone;
        target = nn::audioctrl::AudioTarget_Speaker; // TOOD: prepare headphone type : SIGLO-28907
        break;
    case Device::AhubSpeaker:
        channelCount = 2;
        deviceType = nn::audio::AppletVolumeManager::DeviceType_AhubSpeaker;
        target = nn::audioctrl::AudioTarget_Speaker;
        break;
    case Device::UacSpeaker:
        channelCount = 2;
        deviceType = nn::audio::AppletVolumeManager::DeviceType_UsbOutputDevice;
        target = nn::audioctrl::AudioTarget_UsbOutputDevice;
        break;
    case Device::Invalid:
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    NN_SDK_ASSERT_NOT_EQUAL(deviceType, nn::audio::AppletVolumeManager::DeviceType_Count);
    NN_SDK_ASSERT_NOT_EQUAL(target, nn::audioctrl::AudioTarget_Invalid);
    if ((deviceType == nn::audio::AppletVolumeManager::DeviceType_Count) ||
        (target == nn::audioctrl::AudioTarget_Invalid))
    {
        return;
    }

    if(m_OutputModes[target] == nn::audioctrl::AudioOutputMode_Pcm1ch)
    {
        m_GmixDevice.UpdateDownmixMode(Gmix::DownmixMode::Mono);
    }
    else
    {
        m_GmixDevice.UpdateDownmixMode(Gmix::DownmixMode::Default);
    }

    nn::audio::AppletVolumeManager::SetDevice(deviceType, channelCount);
}

void AudioDeviceController::Initialize(bool hasCodecIc, bool hasHdmi) NN_NOEXCEPT
{
    m_HasCodecIc = hasCodecIc;
    m_HasHdmi = hasHdmi;

    nne::audio::iova::Initialize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(ConvertAdspErrorTypeToNnResult(
        nne::audio::adsp::Initialize(
            reinterpret_cast<uintptr_t>(adsp_os_bin_begin), adsp_os_bin_end - adsp_os_bin_begin,
            reinterpret_cast<uintptr_t>(vector_bin_begin), vector_bin_end - vector_bin_begin,
            nn::audio::server::IsAdspLogEnabled(), AdspExceptionHandler,
            AdspFrequency, ApeFrequency)
    ));

    // Set default volume to OutputMaterVolume
    m_OutputMasterVolume = 1.0f;

    // Set default volume to Codec IC ouptut volumes
    m_CodecSpeakerVolume = 0;
    m_CodecHeadphoneVolume = 0;

    m_IsInputDeviceForceEnabled = false;

    // hwopus
    nn::codec::server::HardwareOpusDecoder::Hardware::Initialize();

    m_GmixDevice.Initialize(hasCodecIc, hasHdmi);
    m_HdaDevice.Initialize(hasCodecIc, hasHdmi);
    m_AhubDevice.Initialize(hasCodecIc, hasHdmi);
    m_AhubInDevice.Initialize(hasCodecIc, hasHdmi);
    m_CodecDevice.Initialize(hasCodecIc, hasHdmi);
    ::nn::audio::detail::UacManager::GetInstance().Initialize();
    m_CodecErrorDetector.Initialize(hasCodecIc, hasHdmi);

    m_HdaDevice.UpdateHotplugState();
    auto device = m_HdaDevice.IsConnected() ? AudioDeviceController::Device::Hda : AudioDeviceController::Device::AhubSpeaker;
    m_IsHdaConnectionRequested = m_HdaDevice.IsConnected();
    if (!hasCodecIc && hasHdmi)
    {
        device = AudioDeviceController::Device::Hda;
        m_IsHdaConnectionRequested = true;
    }
    else if (hasCodecIc && hasHdmi)
    {
        device = AudioDeviceController::Device::AhubSpeaker;
        m_IsHdaConnectionRequested = false;
    }
    m_ForcedTarget = nn::audioctrl::AudioTarget_Invalid;
    SwitchOutputDevice(device);

    InitializeOutputControlThread(); // after all devices initialized.

}

void AudioDeviceController::Finalize() NN_NOEXCEPT
{
    FinalizeOutputControlThread();// before all devices finalized.

    m_CodecErrorDetector.Finalize();
    m_CodecDevice.Finalize();
    ::nn::audio::detail::UacManager::GetInstance().Finalize();
    m_AhubInDevice.Finalize();
    m_AhubDevice.Finalize();
    m_HdaDevice.Finalize();
    m_GmixDevice.Finalize();

    nn::codec::server::HardwareOpusDecoder::Hardware::Finalize();
    nne::audio::adsp::Finalize();
    nne::audio::iova::Finalize();
}

void AudioDeviceController::Shutdown() NN_NOEXCEPT
{
    m_CodecErrorDetector.Sleep();
    m_CodecDevice.Sleep();
}

void AudioDeviceController::Sleep() NN_NOEXCEPT
{
    if (!m_IsAwake)
    {
        return;
    }
    m_IsAwake = false;

    FinalizeOutputControlThread(); // before all devices finalized.

    m_CodecErrorDetector.Sleep();
    m_CodecDevice.Sleep();
    m_AhubDevice.Sleep();
    m_AhubInDevice.Sleep();
    ::nn::audio::detail::UacManager::GetInstance().Sleep();
    m_HdaDevice.Sleep();
    m_GmixDevice.Sleep();

    nn::codec::server::HardwareOpusDecoder::Hardware::Finalize();
    nne::audio::adsp::Finalize();
}

void AudioDeviceController::Wake() NN_NOEXCEPT
{
    if (m_IsAwake)
    {
        return;
    }
    m_IsAwake = true;

    NN_ABORT_UNLESS_RESULT_SUCCESS(ConvertAdspErrorTypeToNnResult(
        nne::audio::adsp::Initialize(
            reinterpret_cast<uintptr_t>(adsp_os_bin_begin), adsp_os_bin_end - adsp_os_bin_begin,
            reinterpret_cast<uintptr_t>(vector_bin_begin), vector_bin_end - vector_bin_begin,
            nn::audio::server::IsAdspLogEnabled(), AdspExceptionHandler,
            AdspFrequency, ApeFrequency)
    ));

    // hwopus
    nn::codec::server::HardwareOpusDecoder::Hardware::Initialize();

    m_GmixDevice.Wake();
    m_HdaDevice.Wake();
    m_AhubDevice.Wake();
    ::nn::audio::detail::UacManager::GetInstance().Wake();
    m_AhubInDevice.Wake();
    m_CodecDevice.Wake();
    m_CodecErrorDetector.Wake();

    m_HdaDevice.UpdateHotplugState();
    auto device = GetRequestedDevice();
    SwitchOutputDevice(device);

    if(m_CodecDevice.IsJackPlugged())
    {
        m_CodecErrorDetector.EnableJackStateFlag();
        m_AhubInDevice.Prepare();
        m_CodecErrorDetector.TriggerMicCheck();
    }

    InitializeOutputControlThread(); // after all devices initialized.
}

void AudioDeviceController::NotifyDeviceSwitch() NN_NOEXCEPT
{
    m_OutputDeviceControlEvent.Signal();
}

void AudioDeviceController::NotifyDeviceSwitch(bool isHda, nn::TimeSpan fadeOutTime, nn::TimeSpan fadeInTime) NN_NOEXCEPT
{
    m_DeviceFadeOutTime = std::max(fadeOutTime, MinimumFadeTime);
    m_DeviceFadeInTime = std::max(fadeInTime, MinimumFadeTime);
    m_IsHdaConnectionRequested = isHda;
    m_OutputDeviceControlEvent.Signal();
}

void AudioDeviceController::NotifyHeadphoneOutSwitch(bool isEnable) NN_NOEXCEPT
{
    // NN_DETAIL_AUDIO_TRACE("[audio] Headphone selected: %s\n", isEnable ? "t" : "f");
    m_IsHeadphoneOutSelected = isEnable;
    m_HeadphoneSelectEvent.Signal();
}

void AudioDeviceController::NotifyHeadphoneInEnabled(bool isEnable) NN_NOEXCEPT
{
    // NN_DETAIL_AUDIO_TRACE("[audio] NotifyHeadphoneInEnabled: %s\n", isEnable ? "t" : "f");
    if (isEnable)
    {
        m_CodecErrorDetector.EnableJackStateFlag();
        m_AhubInDevice.Prepare();
        if (IsAdditionalMicCheckTriggerSuppressed() == false)
        {
            m_CodecErrorDetector.TriggerMicCheck();
        }
    }
    else
    {
        m_AhubInDevice.Stop();
    }
}

void AudioDeviceController::NotifyUacSpeakerSwitch(bool isEnable) NN_NOEXCEPT
{
    // NN_DETAIL_AUDIO_TRACE("[audio] UacSpeaker selected: %s\n", isEnable ? "t" : "f");
    m_IsUacSpeakerSelected = isEnable;

    m_UacSpeakerSelectEvent.Signal();
}

void AudioDeviceController::SetInputDeviceForceEnabled(bool isEnabled) NN_NOEXCEPT
{
    m_IsInputDeviceForceEnabled = isEnabled;
    m_CodecErrorDetector.SetMicBiasOverCurrentDetectionEnabled(isEnabled == false);
}

bool AudioDeviceController::IsInputDeviceForceEnabled() const NN_NOEXCEPT
{
    return m_IsInputDeviceForceEnabled;
}

void AudioDeviceController::NotifyForceDeviceSwitch(nn::audioctrl::AudioTarget target) NN_NOEXCEPT
{
    m_ForcedTarget = target;
    m_OutputDeviceControlEvent.Signal();
}

void AudioDeviceController::SetOutputMasterVolume(float volume, uint32_t curveDurationInMicroseconds) NN_NOEXCEPT
{
    // TODO: Add USB Speaker/Headset support.
    m_OutputMasterVolume = volume;

    m_AhubDevice.SetVolume(volume, curveDurationInMicroseconds);
    m_HdaDevice.SetVolume(volume, curveDurationInMicroseconds);
}

void AudioDeviceController::SetCodecSpeakerVolume(int32_t volume) NN_NOEXCEPT
{
    m_CodecSpeakerVolume = volume;
    if(m_CodecDevice.IsSpeakerMute())
    {
        return;
    }

    m_CodecDevice.SetSpeakerHeadphoneVolume(volume);

    // コーデック IC 出力のボリュームを 0 に絞った時でも音が聞こえる問題への対策のため、
    // ソフトウェアでもミュートする
    m_AhubDevice.SetMute(volume == 0);
}

void AudioDeviceController::SetCodecHeadphoneVolume(int32_t volume) NN_NOEXCEPT
{
    m_CodecHeadphoneVolume = volume;
    if(!m_CodecDevice.IsSpeakerMute())
    {
        return;
    }

    m_CodecDevice.SetSpeakerHeadphoneVolume(volume);

    // コーデック IC 出力のボリュームを 0 に絞った時でも音が聞こえる問題への対策のため、
    // ソフトウェアでもミュートする
    m_AhubDevice.SetMute(volume == 0);
}

void AudioDeviceController::InitializeOutputControlThread() NN_NOEXCEPT
{
    if (m_IsHdaHotplugDetectThreadActive == true)
    {
        return;
    }

    m_IsHdaHotplugDetectThreadActive = true;
    auto rt = nn::os::CreateThread(&m_HdaHotplugDetectThread, AudioDeviceController::HotPlugFunc, this, g_HdaHotplugDetectThreadStack, sizeof(g_HdaHotplugDetectThreadStack), NN_SYSTEM_THREAD_PRIORITY(audio, HdaHotplugDetector));
    NN_ABORT_UNLESS_RESULT_SUCCESS(rt);
    nn::os::SetThreadNamePointer(&m_HdaHotplugDetectThread, NN_SYSTEM_THREAD_NAME(audio, HdaHotplugDetector));
    nn::os::StartThread(&m_HdaHotplugDetectThread);
}

void AudioDeviceController::FinalizeOutputControlThread() NN_NOEXCEPT
{
    if (m_IsHdaHotplugDetectThreadActive == false)
    {
        return;
    }
    m_IsHdaHotplugDetectThreadActive = false;
    m_ExitRequestEvent.Signal();
    nn::os::WaitThread(&m_HdaHotplugDetectThread);
    nn::os::DestroyThread(&m_HdaHotplugDetectThread);
}

void AudioDeviceController::SetOutputMode(nn::audioctrl::AudioTarget target, nn::audioctrl::AudioOutputMode mode) NN_NOEXCEPT
{
    m_OutputModes[target] = mode;
}


}}}} // namespace nn::audio::server::detail

// TODO: move to "nn::audio::server", upper folder.
namespace nn { namespace audio {

//////////////////////////////////////////////////////////////////////////
// For Output device control
//////////////////////////////////////////////////////////////////////////
void NotifyOutputDeviceEvent() NN_NOEXCEPT
{
    server::detail::g_DeviceController.NotifyDeviceSwitch();
}

void NotifyOutputDeviceEvent(bool isHda, nn::TimeSpanType fadeOutTime, nn::TimeSpanType fadeInTime) NN_NOEXCEPT
{
    server::detail::g_DeviceController.NotifyDeviceSwitch(isHda, fadeOutTime, fadeInTime);
}

void NotifyHeadphoneOutSelect(bool isEnable) NN_NOEXCEPT
{
    server::detail::g_DeviceController.NotifyHeadphoneOutSwitch(isEnable);
}

void NotifyHeadphoneInEnabled(bool isEnable) NN_NOEXCEPT
{
    server::detail::g_DeviceController.NotifyHeadphoneInEnabled(isEnable);
}

void NotifyInputDeviceForceEnable(bool isEnable) NN_NOEXCEPT
{
    server::detail::g_DeviceController.SetInputDeviceForceEnabled(isEnable);
    server::detail::g_DeviceController.NotifyHeadphoneInEnabled(isEnable);
    server::detail::g_DeviceController.NotifyHeadphoneOutSwitch(isEnable);
}

void NotifyOutputDeviceForceSwitchEvent(nn::audioctrl::AudioTarget target) NN_NOEXCEPT
{
    server::detail::g_DeviceController.NotifyForceDeviceSwitch(target);
}

void NotifyUacSpeakerSwitch(bool isEnable) NN_NOEXCEPT
{
    server::detail::g_DeviceController.NotifyUacSpeakerSwitch(isEnable);
}

void SetOutputMasterVolume(float volume, uint32_t curveDurationInMicroseconds) NN_NOEXCEPT
{
    server::detail::g_DeviceController.SetOutputMasterVolume(volume, curveDurationInMicroseconds);
}

nn::os::EventType* GetUacSpeakerAttachEvent() NN_NOEXCEPT
{
    return ::nn::audio::detail::UacManager::GetInstance().GetSpeakerAttachEvent();
}

nn::os::EventType* GetUacSpeakerDetachEvent() NN_NOEXCEPT
{
    return ::nn::audio::detail::UacManager::GetInstance().GetSpeakerDetachEvent();
}

int GetNumUacSpeakers() NN_NOEXCEPT
{
    return ::nn::audio::detail::UacManager::GetInstance().GetNumSpeakers();
}

nn::os::EventType* GetUacUnsupportedSpeakerAttachEvent() NN_NOEXCEPT
{
    return ::nn::audio::detail::UacManager::GetInstance().GetUnsupportedSpeakerAttachEvent();
}

//////////////////////////////////////////////////////////////////////////
// For Start up routine.
//////////////////////////////////////////////////////////////////////////
void InitializeAudioHardware(bool hasCodecIc, bool hasHda) NN_NOEXCEPT
{
    server::detail::g_DeviceController.Initialize(hasCodecIc, hasHda);
}

void FinalizeAudioHardware() NN_NOEXCEPT
{
    server::detail::g_DeviceController.Finalize();
}

void ShutdownAudioHardware() NN_NOEXCEPT
{
    server::detail::g_DeviceController.Shutdown();
}

void WakeAudioHardware() NN_NOEXCEPT
{
    server::detail::g_DeviceController.Wake();
}

void SleepAudioHardware() NN_NOEXCEPT
{
    server::detail::g_DeviceController.Sleep();
}

//////////////////////////////////////////////////////////////////////////
// For Audio hardware state dump
//////////////////////////////////////////////////////////////////////////
void DumpAudioHardwareState() NN_NOEXCEPT
{
#if !defined(NN_SDK_BUILD_RELEASE)
    // Dump clock and power states
    nn::pcv::Initialize();
    const nn::pcv::Module modules[] = {
        nn::pcv::Module_Hda2hdmicodec,
        nn::pcv::Module_Hda,
        nn::pcv::Module_Hda2codec2x,
        nn::pcv::Module_AudioDsp,
        nn::pcv::Module_Ape,
        nn::pcv::Module_AudioUart,
        nn::pcv::Module_Disp1,
        nn::pcv::Module_Disp2,
        nn::pcv::Module_Sor1,
    };

    const char* moduleNames[] = {
        "Module_Hda2hdmicodec",
        "Module_Hda",
        "Module_Hda2codec2x",
        "Module_AudioDsp",
        "Module_Ape",
        "Module_AudioUart",
        "Module_Disp1",
        "Module_Disp2",
        "Module_Sor1",
    };

    for(int i = 0; i < sizeof(modules) / sizeof(modules[0]); ++i)
    {
        nn::pcv::ModuleState modState = {0};
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::GetState(&modState, modules[i]));
        NN_DETAIL_AUDIO_INFO("[audio][ClockInfo] module       : %s (%d)\n", moduleNames[i], static_cast<int>(modules[i]));
        NN_DETAIL_AUDIO_INFO("[audio][ClockInfo] Reset        : %s\n", modState.resetAsserted ? "TRUE" : "FALSE");
        NN_DETAIL_AUDIO_INFO("[audio][ClockInfo] Power enabled: %s\n", modState.powerEnabled ? "TRUE" : "FALSE");
        NN_DETAIL_AUDIO_INFO("[audio][ClockInfo] Clock enabled: %s\n", modState.clockEnabled ? "TRUE" : "FALSE");
        NN_DETAIL_AUDIO_INFO("[audio][ClockInfo] Clock frequency: %.2f MHz\n", modState.clockFrequency / (1000.0 * 1000.0));
    }
    NN_DETAIL_AUDIO_INFO("[audio]{ClockInfo] ------------------------------------------------------\n");
    nn::pcv::Finalize();
#endif
}


//////////////////////////////////////////////////////////////////////////
// For AudioController APIs
//////////////////////////////////////////////////////////////////////////
nn::os::SystemEvent* RegisterJackEvent() NN_NOEXCEPT
{
    auto codec = server::detail::g_DeviceController.GetCodecDevice();
    return (codec) ? codec->RegisterJackEvent() : nullptr;
}

void UnregisterJackEvent() NN_NOEXCEPT
{
    auto codec = server::detail::g_DeviceController.GetCodecDevice();
    if (codec)
    {
        codec->UnregisterJackEvent();
    }
}

bool IsJackPlugged() NN_NOEXCEPT
{
    auto codec = server::detail::g_DeviceController.GetCodecDevice();
    return (codec) ? codec->IsJackPlugged() : false;
}

bool IsAnalogMicrophonePlugged() NN_NOEXCEPT
{
    auto isInputDeviceForceEnabled = server::detail::g_DeviceController.IsInputDeviceForceEnabled();
    auto ahubIn = server::detail::g_DeviceController.GetAhubInDevice();

    return  (isInputDeviceForceEnabled) ? true :
                               (ahubIn) ? ahubIn->IsStarted() : false;
}

void ClearJackEvent() NN_NOEXCEPT
{
    auto codec = server::detail::g_DeviceController.GetCodecDevice();
    if (codec)
    {
        codec->ClearJackEvent();
    }
}

void SetSpeakerVolume(int32_t volume) NN_NOEXCEPT
{
    server::detail::g_DeviceController.SetCodecSpeakerVolume(volume);
}

void SetHeadphoneVolume(int32_t volume) NN_NOEXCEPT
{
    server::detail::g_DeviceController.SetCodecHeadphoneVolume(volume);
}

void SetHeadphoneAmplifierGain(int32_t gain) NN_NOEXCEPT
{
    auto codec = server::detail::g_DeviceController.GetCodecDevice();
    if (codec)
    {
        codec->SetHeadphoneAmplifierGain(gain);
    }
}

void SetSpeakerMute(bool mute) NN_NOEXCEPT
{
    auto codec = server::detail::g_DeviceController.GetCodecDevice();
    if (codec)
    {
        codec->SetSpeakerMute(mute);
    }
}

bool IsSpeakerMute() NN_NOEXCEPT
{
    auto codec = server::detail::g_DeviceController.GetCodecDevice();
    return (codec) ? codec->IsSpeakerMute() : false;
}

bool IsCodecRequested() NN_NOEXCEPT
{
    return !server::detail::g_DeviceController.IsHdaRequested();
}

void SetUacOutputDeviceVolume(float volume) NN_NOEXCEPT
{
    // NN_DETAIL_AUDIO_INFO("[%s:%s:%4d] volume(%.2f)\n", __FILE__, NN_CURRENT_FUNCTION_NAME,  __LINE__, volume);
    ::nn::audio::detail::UacManager::GetInstance().SetSpeakerVolume(volume);
}

void SetOutputMode(nn::audioctrl::AudioTarget target, nn::audioctrl::AudioOutputMode mode) NN_NOEXCEPT
{
    server::detail::g_DeviceController.SetOutputMode(target, mode);
}

}} // namespace nn::audio
