﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <array>
#include <nn/util/util_IntrusiveList.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Mutex.h>
#include <nn/nn_Abort.h>
#include "../common/audio_BuildDefinition.h"

namespace nn { namespace audio { namespace server {

class SystemEventHolder : public nn::util::IntrusiveListBaseNode<SystemEventHolder>
{
private:
    nn::os::SystemEvent m_SystemEvent;

#if defined(NN_AUDIO_SYSTEM_PROCESS)
static const bool IsSystemProcess = true;
#else
static const bool IsSystemProcess = false;
#endif

public:
    SystemEventHolder() NN_NOEXCEPT
        : m_SystemEvent(nn::os::EventClearMode_AutoClear, IsSystemProcess)
    {

    }

    nn::os::SystemEvent& GetSystemEvent() NN_NOEXCEPT
    {
        return m_SystemEvent;
    }
};

template<int N>
class SystemEventHolderPool
{
private:
    nn::os::Mutex m_Mutex;

    typedef nn::util::IntrusiveList<SystemEventHolder, nn::util::IntrusiveListBaseNodeTraits<SystemEventHolder>> SystemEventHolderList;

    SystemEventHolderList m_UnusedHolderList;
    std::array<SystemEventHolder, N> m_SystemEventHolders;

public:
    SystemEventHolderPool() NN_NOEXCEPT
        : m_Mutex(false)
    {
        for(auto& holder : m_SystemEventHolders)
        {
            m_UnusedHolderList.push_back(holder);
        }
    }

    SystemEventHolder& Acquire() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_ABORT_UNLESS(!m_UnusedHolderList.empty());

        auto& holder = m_UnusedHolderList.front();
        m_UnusedHolderList.pop_front();

        holder.GetSystemEvent().Clear();

        return holder;
    }

    void Release(SystemEventHolder& holder) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        m_UnusedHolderList.push_back(holder);
    }
};

}}} // namespace nn::audio::server
