﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/audio/audio_VoiceTypes.h>
#include <nn/audio/audio_SampleFormat.h>
#include <nn/util/util_IntrusiveList.h>

#include "../audio_MixManager.h"
#include "../audio_VoiceInfoManager.h"
#include "audio_VoiceState.h"

#include "audio_ServiceBehaviorInfo.h"
#include "audio_ServiceMemoryPoolInfo.h"
#include "../common/audio_BehaviorParameters.h"
#include "../common/audio_SplitterParameters.h"

#include "../common/audio_Util.h"
#include "../common/audio_VoiceCommon.h"

namespace nn {
namespace audio {
namespace server {

typedef int32_t VoiceChannelResourceId;
class VoiceContext;

class VoiceChannelResource
{
protected:
    float m_MixVolume[MixBufferCountMax];
    float m_MixVolumePrev[MixBufferCountMax];
    VoiceChannelResourceId m_Id;
    bool m_IsInUse;
    int8_t _padding[11];

public:
    NN_FORCEINLINE VoiceChannelResource(int index) NN_NOEXCEPT
        : m_Id(index), m_IsInUse(false)
    {
        memset(m_MixVolume, 0, sizeof(m_MixVolume));
        memset(m_MixVolumePrev, 0, sizeof(m_MixVolumePrev));
    }

    NN_FORCEINLINE VoiceChannelResourceId GetId() const NN_NOEXCEPT
    {
        return m_Id;
    };

    NN_FORCEINLINE bool IsInUse() const NN_NOEXCEPT
    {
        return m_IsInUse;
    };

    NN_FORCEINLINE float GetMixVolume(int index) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_RANGE(index, 0, MixBufferCountMax);
        return m_MixVolume[index];
    };

    NN_FORCEINLINE float GetMixVolumePrev(int index) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_RANGE(index, 0, MixBufferCountMax);
        return m_MixVolumePrev[index];
    };

    NN_FORCEINLINE const float* GetMixVolume() const NN_NOEXCEPT
    {
        return m_MixVolume;
    };

    NN_FORCEINLINE const float* GetMixVolumePrev() const NN_NOEXCEPT
    {
        return m_MixVolumePrev;
    };

    NN_FORCEINLINE void Update(const nn::audio::VoiceChannelResource::InParameter* pInParameter) NN_NOEXCEPT
    {
        m_IsInUse = pInParameter->_isInUse;

        if(pInParameter->_isInUse)
        {
            NN_SDK_ASSERT_EQUAL(m_Id, pInParameter->_id);

            NN_STATIC_ASSERT(sizeof(m_MixVolume) == sizeof(pInParameter->_mixVolume));
            memcpy(m_MixVolume, pInParameter->_mixVolume, sizeof(pInParameter->_mixVolume));
        }
    };

    NN_FORCEINLINE void UpdateInternalState() NN_NOEXCEPT
    {
        std::memcpy(m_MixVolumePrev, m_MixVolume, sizeof(m_MixVolume));
    }
};
#if defined(NN_BUILD_CONFIG_SPEC_NX) && !defined(NN_BUILD_CONFIG_OS_WIN)
NN_STATIC_ASSERT(sizeof(VoiceChannelResource) == 208);
#endif

enum class PlayState
{
    Play,               // 再生中
    Stop,               // 停止中
    StopRequested,      // 停止要求
    Pause,              // 一時停止中
};

class WaveBuffer
{
public:
    WaveBuffer() NN_NOEXCEPT;
    void Initialize() NN_NOEXCEPT;

    AddressInfo buffer;
    int32_t startSampleOffset;
    int32_t endSampleOffset;
    bool loop;
    bool isEndOfStream;

    AddressInfo pContextInfo;
    bool _isSentToDsp;
};
static const size_t wbs = sizeof(WaveBuffer);
#if defined(NN_BUILD_CONFIG_SPEC_NX) && !defined(NN_BUILD_CONFIG_OS_WIN)
NN_AUDIO_INFOTYPE_CHECK(WaveBuffer, NN_AUDIO_ADDRESS_SELECT(72, 88));
#endif


struct VoiceInfo
{
    struct InParameter
    {
        bool _isInUse;
        bool _isNew; // TODO: isNew の処理が分散しているので、まとめる
        bool _shouldDepop;
        uint8_t _sampleFormat;
        int32_t _sampleRate;
        int32_t _channelCount;
        int32_t _id;
        NodeId _nodeId;
        MixId _destinationMixId;
        PlayState _playState;
        PlayState _playStatePrev;
        int32_t _priority;
        int32_t _sortingOrder;

        float _pitch;
        float _volume;
        float _volumePrev;

        BiquadFilterParameter _biquadFilterParameter[VoiceType::BiquadFilterCountMax];
        int32_t _waveBufferCount;
        int16_t _waveBufferHead;
        int8_t _padding1[2];
        VoiceBehaviorFlagSet _behaviorFlags;

        AddressInfo _pAdditionalParameterInfo;
        WaveBuffer _waveBuffer[VoiceType::WaveBufferCountMax];
        VoiceChannelResourceId _voiceChannelResourceIds[VoiceType::ChannelCountMax];
        common::SplitterInfoId _splitterInfoId;
        int8_t _padding2[11];
    };

    struct OutStatus
    {
        int64_t _playedSampleCount;
        int32_t _waveBufferConsumed;
        int8_t _padding[4];
    };

    InParameter _inParameter;
    OutStatus _outStatus;

    int nodeIdBase;
    int nodeIdVariation;
    bool voiceDroppedFlag;
    bool noMappedParameter;
    bool noMappedWaveBuffer;
    bool isBiquadFilterEnabledPrev[VoiceType::BiquadFilterCountMax];
    uint8_t waveBufferFlushRequestedCount;

    int8_t _padding[2];

public:
    void Initialize() NN_NOEXCEPT;
    VoiceInfo() NN_NOEXCEPT;

    bool UpdateForCommandGeneration(VoiceContext* data) NN_NOEXCEPT;
    void UpdateParameters(common::BehaviorParameter::ErrorInfo* pOutErrorInfo, const nn::audio::VoiceInfo::InParameter* pInParameter, server::PoolMapper& poolMapper, const server::BehaviorInfo& behaviorInfo) NN_NOEXCEPT;
    void UpdateWaveBuffers(
        common::BehaviorParameter::ErrorInfo* pOutErrorInfo,
        int errorInfoCount,
        const nn::audio::VoiceInfo::InParameter* pInParameter,
        VoiceState* pStates[],
        server::PoolMapper& poolMapper,
        const BehaviorInfo& behaviorInfo) NN_NOEXCEPT;
    void WriteOutStatus(nn::audio::VoiceInfo::OutStatus * outStatus, const nn::audio::VoiceInfo::InParameter* pInParameter, VoiceState* pStates[]) NN_NOEXCEPT;
    bool ShouldSkip() const NN_NOEXCEPT;
    bool HasAnyConnection() const NN_NOEXCEPT;

    MixId GetDestinationMixId() const   { return _inParameter._destinationMixId; }
    void SetDestinationMixId(MixId val) { _inParameter._destinationMixId = val; }
    common::SplitterInfoId GetSplitterInfoId() const { return _inParameter._splitterInfoId; }
    void SetSplitterInfoId(common::SplitterInfoId val) { _inParameter._splitterInfoId = val; }

private:
    void ResetResources(VoiceContext* data) NN_NOEXCEPT;
    bool UpdateParametersForCommandGeneration(VoiceState* pStates[]) NN_NOEXCEPT;
    void UpdateWaveBuffer(
        common::BehaviorParameter::ErrorInfo* pOutErrorInfo,
        server::WaveBuffer* pOutWaveBuffer,
        const ::nn::audio::VoiceInfo::WaveBufferInternal* pInWaveBuffer,
        uint8_t format,
        bool validState,
        server::PoolMapper& poolMapper,
        const BehaviorInfo& behaviorInfo) NN_NOEXCEPT;
    void UpdatePlayState(uint8_t state) NN_NOEXCEPT;
    bool ShouldUpdateParameters(const nn::audio::VoiceInfo::InParameter* pInParameter) const NN_NOEXCEPT;
    bool ShouldUpdateWaveBuffer(const ::nn::audio::VoiceInfo::WaveBufferInternal* pInWaveBuffer) const NN_NOEXCEPT;
    void FlushWaveBuffers(int count, VoiceState* pStates[], int channelCount) NN_NOEXCEPT;

};
#if defined(NN_BUILD_CONFIG_SPEC_NX) && !defined(NN_BUILD_CONFIG_OS_WIN)
NN_AUDIO_INFOTYPE_CHECK(VoiceInfo, NN_AUDIO_ADDRESS_SELECT(472, 544));
NN_STATIC_ASSERT(sizeof(VoiceInfo::InParameter) == NN_AUDIO_ADDRESS_SELECT(440, 512));
NN_STATIC_ASSERT(sizeof(VoiceInfo::OutStatus) == NN_AUDIO_ADDRESS_SELECT(16, 16));
#endif


// Data container & accessor for service side voice related infos.
class VoiceContext
{
private:
    server::VoiceInfo** m_SortedInfos;
    server::VoiceInfo* m_Infos;
    VoiceChannelResource* m_ChannelResources;
    VoiceState* m_States;
    VoiceState* m_DspSharedStates;
    int m_InfoCount;
    int m_ActiveCount;

public:
    VoiceContext() NN_NOEXCEPT;
    void Initialize(
        server::VoiceInfo** sortedInfos,
        server::VoiceInfo* infos,
        VoiceChannelResource* channelResources,
        VoiceState* states,
        VoiceState* dspSharedStates,
        int infoCount) NN_NOEXCEPT;
    server::VoiceInfo& GetSortedInfo(int id) const NN_NOEXCEPT;
    server::VoiceInfo& GetInfo(int id) const NN_NOEXCEPT;
    VoiceChannelResource& GetChannelResource(int id) const NN_NOEXCEPT;
    VoiceState& GetState(int id) const NN_NOEXCEPT;
    VoiceState& GetDspSharedState(int id) const NN_NOEXCEPT;
    int GetCount() const NN_NOEXCEPT;
    int GetActiveCount() const NN_NOEXCEPT;
    void SetActiveCount(int activeCount) NN_NOEXCEPT;

    void SortInfo() NN_NOEXCEPT;
    void UpdateStateByDspShared() NN_NOEXCEPT;
};

}  // namespace server
}  // namespace audio
}  // namespace nn
