﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
File:     audio_Downsampler.cpp
Block SRC
*---------------------------------------------------------------------------*/
#include <cmath>
#include <cstring>
#include <limits>

#include <nn/result/result_HandlingUtility.h>
#include <nn/audio/audio_Result.h>
#include <nn/nn_SdkAssert.h>

#include <nn/audio/audio_Resampler.h>
#include "audio_Resampler.private.h"
#include "audio_ResamplerFilter.h"
#include "audio_ResamplerFilter_4ZeroCross.h"
#include <nn/audio/audio_Common.h>

namespace nn {
namespace audio {

inline int32_t Multiply(int32_t x, int32_t y, int q)
{
    return static_cast<int32_t>(static_cast<int64_t>(x) * y >> q);
}


template <typename T, typename U>
T Clamp(U x)
{
    const U max = static_cast<U>(std::numeric_limits<T>::max());
    const U min = static_cast<U>(std::numeric_limits<T>::min());
    x = x < max ? x : max;
    x = x > min ? x : min;
    return static_cast<T>(x);
}

#ifndef MAX
#define MAX(x, y) ((x) > (y) ? (x) : (y))
#endif
#ifndef MIN
#define MIN(x, y) ((x) < (y) ? (x) : (y))
#endif

/*! \fn    uint16_t checkBoundsUp(ResamplerState *pState, uint16_t arrayIndex)
*  \brief
*/
inline static uint16_t checkBoundsUp(ResamplerState *pState, uint16_t arrayIndex)NN_NOEXCEPT
{
    if (arrayIndex > pState->input_cb_upperBound) /* CHECK ON BOUNDS */
    {
        arrayIndex -= 2 * pState->nz_input; //20; /* buffer size 2*pState->nz_input */
    }
    return arrayIndex;
}

/*! \fn    int32_t FilterUpLeft(uint16_t Xi, uint32_t Ph, ResamplerState *pState)
*  \brief
*/
static int32_t FilterUpLeft(uint16_t Xi, uint32_t Ph, ResamplerState *pState) NN_NOEXCEPT
{
    const int16_t *Hp = NULL;
    const int16_t *Imp;
    Imp = pState->h;
    Hp = &Imp[Ph >> NN_AUDIO_SRC_BL_DN_NNU];
    int64_t  v = 0;
    uint16_t  i;
    uint16_t  jj = Xi - pState->input_cb_lowerBound + 1;
    uint16_t  loop1 = (jj > pState->nz_input) ? pState->nz_input : jj;
    for (i = 0; i < loop1; i++)
    {
        v += static_cast<int64_t>(*Hp) * pState->input_samples[Xi]; /* Mult coeff by input sample */
        Hp += NN_AUDIO_SRC_BL_DN_L;                /* Filter coeff step */
        Xi--;
    }
    Xi = pState->input_cb_upperBound;
    for (; i < pState->nz_input; i++)
    {
        v += static_cast<int64_t>(*Hp) * pState->input_samples[Xi]; /* Mult coeff by input sample */
        Hp += NN_AUDIO_SRC_BL_DN_L;                /* Filter coeff step */
        Xi--;                              /* Input signal step. */
    }
    v >>= Nhxn;
    return static_cast<int32_t>(v);
}

static int32_t FilterUpLeftLinear(int32_t numChannels, const int16_t* pIn, uint32_t Ph, ResamplerState *pState) NN_NOEXCEPT
{
    const int16_t *Hp = NULL;
    const int16_t *Imp;
    Imp = pState->h;
    Hp = &Imp[Ph >> NN_AUDIO_SRC_BL_DN_NNU];
    auto pInBuffer = pIn;
    int64_t  v = 0;

    for (auto i = 0; i < pState->nz_input; i++)
    {
        v += static_cast<int64_t>(*Hp) * (static_cast<int32_t>(*pInBuffer) << 8); /* Mult coeff by input sample */
        Hp += NN_AUDIO_SRC_BL_DN_L;                /* Filter coeff step */
        pInBuffer -= numChannels;
    }

    v >>= Nhxn;
    return static_cast<int32_t>(v);
}

/*! \fn    int32_t FilterUpRight(uint16_t Xi, uint32_t Ph, ResamplerState *pState)
*  \brief
*/
static int32_t FilterUpRight(uint16_t Xi, uint32_t Ph, ResamplerState *pState) NN_NOEXCEPT
{
    const int16_t *Hp = NULL;
    const int16_t *Imp;
    Imp = pState->h;
    Hp = &Imp[Ph >> NN_AUDIO_SRC_BL_DN_NNU];
    int64_t  v = 0;
    uint16_t  i;
    uint16_t  jj = pState->input_cb_upperBound - Xi + 1;
    uint16_t  loop1 = (jj > pState->nz_input) ? pState->nz_input : jj;


    for (i = 0; i < loop1; i++)
    {
        v += static_cast<int64_t>(*Hp) * pState->input_samples[Xi]; /* Mult coeff by input sample */
        Hp += NN_AUDIO_SRC_BL_DN_L;                /* Filter coeff step */
        Xi++;
    }

    Xi = pState->input_cb_lowerBound;

    for (; i < pState->nz_input; i++)
    {
        v += static_cast<int64_t>(*Hp) * pState->input_samples[Xi]; /* Mult coeff by input sample */
        Hp += NN_AUDIO_SRC_BL_DN_L;                /* Filter coeff step */
        Xi++;
    }

    v >>= Nhxn;

    return static_cast<int32_t>(v);
}

static int32_t FilterUpRightLinear(int32_t numChannels, const int16_t* pIn, uint32_t Ph, ResamplerState *pState) NN_NOEXCEPT
{
    const int16_t *Hp = NULL;
    const int16_t *Imp;
    Imp = pState->h;
    Hp = &Imp[Ph >> NN_AUDIO_SRC_BL_DN_NNU];
    auto pInBuffer = pIn;
    int64_t  v = 0;

    for (auto i = 0; i < pState->nz_input; i++)
    {
        v += static_cast<int64_t>(*Hp) * (static_cast<int32_t> (* pInBuffer) << 8); /* Mult coeff by input sample */
        Hp += NN_AUDIO_SRC_BL_DN_L;                /* Filter coeff step */
        pInBuffer += numChannels;
    }

    v >>= Nhxn;
    return static_cast<int32_t>(v);
}

/*! \fn    int32_t FilterDownLeft(uint16_t Xi, uint32_t Ph, uint32_t L_step, ResamplerState *pState)
*  \brief
*/
static int32_t FilterDownLeft(uint16_t Xi, uint32_t Ph, uint32_t L_step, ResamplerState *pState) NN_NOEXCEPT
{
    uint32_t  l;
    uint32_t ratio;
    uint32_t temp;

    const int16_t *Imp = pState->h;
    ratio = pState->ratioRho;

    temp = (Ph * ratio);
    if (temp & (1 << (NQRATIO - 1)))            /* Round, if needed */
    {
        temp += (1 << (NQRATIO - 1));
    }
    Ph = temp >> NQRATIO;

    int64_t  v = 0;
    uint16_t  i;
    uint16_t  jj = Xi - pState->input_cb_lowerBound + 1;
    uint16_t  loop1 = (jj > pState->nz_input) ? pState->nz_input : jj;

    for (i = 0; i < loop1; i++)
    {
        l = Ph >> NN_AUDIO_SRC_BL_DN_NNU;
        v += static_cast<int64_t>(*(Imp + l)) * pState->input_samples[Xi];
        Ph += L_step;
        Xi--;
    }
    Xi = pState->input_cb_upperBound;
    for (; i < pState->nz_input; i++)
    {
        l = Ph >> NN_AUDIO_SRC_BL_DN_NNU;
        v += static_cast<int64_t>(*(Imp + l)) * pState->input_samples[Xi];
        Ph += L_step;
        Xi--;                              /* Input signal step. */
    }
    v >>= Nhxn;
    return static_cast<int32_t>(v);
}

static int32_t FilterDownLeftLinear(int32_t numChannels, const int16_t* pIn, uint32_t Ph, uint32_t L_step, ResamplerState *pState) NN_NOEXCEPT
{
    uint32_t  l;
    uint32_t ratio;
    uint32_t temp;
    auto pInBuffer = pIn;

    const int16_t *Imp = pState->h;
    ratio = pState->ratioRho;

    temp = (Ph * ratio);
    if (temp & (1 << (NQRATIO - 1)))            /* Round, if needed */
    {
        temp += (1 << (NQRATIO - 1));
    }
    Ph = temp >> NQRATIO;

    int64_t  v = 0;

    for (auto i = 0; i < pState->nz_input; i++)
    {
        l = Ph >> NN_AUDIO_SRC_BL_DN_NNU;
        v += static_cast<int64_t>(*(Imp + l)) * (static_cast<int32_t> (*pInBuffer) << 8);
        Ph += L_step;
        pInBuffer -= numChannels;
    }

    v >>= Nhxn;
    return static_cast<int32_t>(v);
}

/*! \fn    int32_t FilterDownRight(uint16_t Xi, uint32_t Ph, uint32_t L_step, ResamplerState *pState)
*  \brief
*/
static int32_t FilterDownRight(uint16_t Xi, uint32_t Ph, uint32_t L_step, ResamplerState *pState) NN_NOEXCEPT
{
    uint32_t  l;
    uint32_t ratio;
    uint32_t temp;

    const int16_t *Imp = pState->h;
    ratio = pState->ratioRho;

    temp = (Ph * ratio);
    if (temp & (1 << (NQRATIO - 1)))            /* Round, if needed */
    {
        temp += (1 << (NQRATIO - 1));
    }
    Ph = temp >> NQRATIO;

    int64_t  v = 0;
    uint16_t  i;
    uint16_t  jj = pState->input_cb_upperBound - Xi + 1;
    uint16_t  loop1 = (jj > pState->nz_input) ? pState->nz_input : jj;

    for (i = 0; i < loop1; i++)
    {
        l = Ph >> NN_AUDIO_SRC_BL_DN_NNU;
        v += static_cast<int64_t>(*(Imp + l)) * pState->input_samples[Xi];
        Ph += L_step;
        Xi++;
    }
    Xi = pState->input_cb_lowerBound;
    for (; i < pState->nz_input; i++)
    {
        l = Ph >> NN_AUDIO_SRC_BL_DN_NNU;
        v += static_cast<int64_t>(*(Imp + l)) * pState->input_samples[Xi];
        Ph += L_step;
        Xi++;
    }
    v >>= Nhxn;
    return static_cast<int32_t>(v);
}

static int32_t FilterDownRightLinear(int32_t numChannels, const int16_t* pIn, uint32_t Ph, uint32_t L_step, ResamplerState *pState) NN_NOEXCEPT
{
    uint32_t  l;
    uint32_t ratio;
    uint32_t temp;
    auto pInBuffer = pIn;

    const int16_t *Imp = pState->h;
    ratio = pState->ratioRho;

    temp = (Ph * ratio);
    if (temp & (1 << (NQRATIO - 1)))            /* Round, if needed */
    {
        temp += (1 << (NQRATIO - 1));
    }
    Ph = temp >> NQRATIO;

    int64_t  v = 0;

    for (auto i = 0; i < pState->nz_input; i++)
    {
        l = Ph >> NN_AUDIO_SRC_BL_DN_NNU;
        v += static_cast<int64_t>(*(Imp + l)) * (static_cast<int32_t> (*pInBuffer) << 8);
        Ph += L_step;
        pInBuffer += numChannels;
    }

    v >>= Nhxn;
    return static_cast<int32_t>(v);
}

/*! \fn   SrcInitBandlimitedSrc(DownSamplerState *pState, AudioBuffer *pSrc, uint32_t ratio, int32_t *input)
*  \brief initialization for Bandlimited SRC.
*
*  Initialize block SRC configuration buffer, read input audio samples and save to the structure
*
*  \param[in]  DownSamplerState *pState: The SRC configuration
*  \param[in]  AudioBuffer    *pSrc: pointer to the input audio buffer
*  \param[in]  uint32_t ratio:       SRC ratio
*  \param[out] Return number of samples read
*/

uint32_t InitializeResampler(ResamplerState *pState, uint32_t ratio, bool highQuality) NN_NOEXCEPT
{
    uint32_t inSamples;

    /* Clear input sample buffer before write pointer */
    memset(pState->input_samples, 0, sizeof(pState->input_samples));

    /* limit the conversion ratio */
    ratio = MIN(ratio, static_cast<uint32_t>(NN_AUDIO_SRC_BL_DN_RATIO_MAX));
    ratio = MAX(ratio, static_cast<uint32_t>(NN_AUDIO_SRC_BL_DN_RATIO_MIN));

    if (ratio >= (static_cast<uint32_t>(1) << NQRATIO))  // 1.0, Q"NQRATIO" format
    {
        if (true == highQuality)
        {
            pState->nz_input = NN_AUDIO_SRC_BL_DN_NZ;
        }
        else
        {
            pState->nz_input = NN_AUDIO_SRC_BL_DN_NZ4;
        }
    }
    else
    {
        if (true == highQuality)
        {
            pState->nz_input = static_cast<uint16_t>((static_cast<uint32_t>(NN_AUDIO_SRC_BL_DN_NZ) << NQRATIO) / ratio);
        }
        else
        {
            pState->nz_input = static_cast<uint16_t>((static_cast<uint32_t>(NN_AUDIO_SRC_BL_DN_NZ4) << NQRATIO) / ratio);
        }
    }

    /* Set pState */
    pState->timeRegister = 0;
    pState->time_inc = (static_cast<uint32_t>(1) << (NN_AUDIO_SRC_BL_DN_NP + NQRATIO)) / ratio;
    pState->ratioRho = ratio;

    pState->input_current = (NN_AUDIO_SRC_BL_DN_CB_MID - 1);
    pState->input_write = NN_AUDIO_SRC_BL_DN_CB_MID;
    pState->input_cb_lowerBound = (NN_AUDIO_SRC_BL_DN_CB_MID - pState->nz_input);
    pState->input_cb_upperBound = (NN_AUDIO_SRC_BL_DN_CB_MID - 1 + pState->nz_input);
    pState->numLeftovers = 0;
    memset(pState->leftoverSamples, 0, sizeof(pState->leftoverSamples));

    if (true == highQuality)
    {
        pState->h = h;
    }
    else
    {
        pState->h = h4;
    }
    /*  initialize input_buffer with silence */
    for (inSamples = 0; inSamples < pState->nz_input; inSamples++)
    {
        pState->input_samples[pState->input_write++] = 0;   //Q8 format
    }
    pState->input_write = pState->input_cb_lowerBound;

    pState->isInitialized = ResamplerState::ResamplerStateMagic;
    return inSamples;
}

/*! \fn  ResampleUpConverting(AudioBuffer *pSrc, int32_t *output, DownSamplerState *pState)
*
*  \brief  run SRC for rho >= 1
*
*  \param[in]  AudioBuffer *pSrc   Audio context buffer
*  \param[in]  DownSamplerState *pState
*  \param[out]
*  \return 0
*/
static int32_t ResampleUpConverting(
    int32_t numChannels, int32_t channel,
    const int16_t* pSrc, int16_t* pDest,
    int32_t inSamples, int32_t outSamples,
    ResamplerState *pState, float volume) NN_NOEXCEPT
{
    int32_t index;
    int32_t gain = static_cast<int32_t>(volume * (1 << 15));
    int16_t* pOut = &pDest[channel];
    const int16_t* pIn = &pSrc[channel];
    int16_t samplesConsumed = 0;
    auto nz = pState->nz_input;

    /*
        now generate a frames worth of output samples
        */
    for (index = 0; index < outSamples; index++)
    {
        uint32_t P;
        int32_t v_L;

        /* n- how many more new input samples needed */
        uint16_t n = static_cast<uint16_t>(pState->timeRegister >> NN_AUDIO_SRC_BL_DN_NP);
        pState->timeRegister += pState->time_inc;
        n = static_cast<uint16_t>(pState->timeRegister >> NN_AUDIO_SRC_BL_DN_NP) - n;

        if (samplesConsumed <= nz * 2)
        {
            /* if we need any originals get them now, calling function need to make sure enough samples in the input buffer
               if not, remaining locations should be set to 0 by calling function
               */
            if (0 != n)
            {
                for (auto i = 0; i < n; i++)
                {
                    /* Assume input from *pIn */
                    int32_t input = static_cast<int32_t>(*pIn);
                    pIn += numChannels;
                    pState->input_samples[pState->input_write++] = (input << 8); /* Q8 format */
                    pState->input_write = checkBoundsUp(pState, pState->input_write);
                    inSamples++;
                }
                pState->input_current += n;
                pState->input_current = checkBoundsUp(pState, pState->input_current);
                samplesConsumed += n;
            }

            /* apply left wing of the filter */
            P = (pState->timeRegister << NN_AUDIO_SRC_BL_DN_NN) >> NN_AUDIO_SRC_BL_DN_NN;
            v_L = FilterUpLeft(pState->input_current, P, pState);

            /* apply right wing of the filter */
            P = (1 << NN_AUDIO_SRC_BL_DN_NP) - P;
            v_L += FilterUpRight(checkBoundsUp(pState, pState->input_current + 1), P, pState);

            /* write output  note channel interleave */
            v_L = Multiply(v_L, gain, 15);
            *pOut = Clamp<int16_t>(v_L >> 8);
            pOut = &pOut[numChannels];
        }
        else
        {
            if (0 != n)
            {
                pIn += numChannels * n;
                samplesConsumed += n;
            }
            auto pInLeft = (pIn - numChannels) - numChannels * nz;

            /* apply left wing of the filter */
            P = (pState->timeRegister << NN_AUDIO_SRC_BL_DN_NN) >> NN_AUDIO_SRC_BL_DN_NN;
            v_L = FilterUpLeftLinear(numChannels, pInLeft, P, pState);

            /* apply right wing of the filter */
            P = (1 << NN_AUDIO_SRC_BL_DN_NP) - P;
            v_L += FilterUpRightLinear(numChannels, pInLeft + numChannels, P, pState);

            /* write output  note channel interleave */
            v_L = Multiply(v_L, gain, 15);
            *pOut = Clamp<int16_t>(v_L >> 8);
            pOut = &pOut[numChannels];
        }
    }

    pState->input_current = (NN_AUDIO_SRC_BL_DN_CB_MID - 1);
    pState->input_write = pState->input_cb_lowerBound;
    pIn = (pIn - numChannels * 2 * nz);
    for (auto j = 0; j < 2 * nz; j++)
    {
        pState->input_samples[pState->input_write++] = static_cast<int32_t>(*pIn) << 8; /* Q8 format */
        pIn += numChannels;
    }
    pState->input_write = pState->input_cb_lowerBound;

    return 0;
}

/*! \fn  ResampleDownConverting(AudioBuffer *pSrc, int32_t *output, DownSamplerState *pState)
*
*  \brief  run SRC for rho < 1
*
*  \param[in]  AudioBuffer *pSrc   Audio context buffer
*  \param[in]  DownSamplerState *pState
*  \param[out] AudioBuffer *pOut  Audio context buffer updated
*  \return 0
*/
static int32_t ResampleDownConverting(
    int32_t numChannels, int32_t channel,
    const int16_t* pSrc, int16_t* pDest,
    int32_t inSamples, int32_t outSamples,
    ResamplerState *pState, float volume) NN_NOEXCEPT
{
    int32_t gain = static_cast<int32_t>(volume * (1 << 15));
    int16_t* pOut = &pDest[channel];
    const int16_t* pIn = &pSrc[channel];
    int16_t samplesConsumed = 0;
    int16_t* pLeftover = &(pState->leftoverSamples[0]);
    int32_t input;

    /*
    now generate a frames worth of output samples
    NB: modified for specific 48K - 44K1 downsampling inline
    must consume precisely inSamples each frame time
    let outSamples float appropriately
    */
    for (auto index = 0; index < outSamples; index++)
    {
        uint32_t P;
        uint32_t L_step;
        int32_t v_L;
        int64_t temp;

        /* n- how many more new input samples needed */
        uint16_t n = static_cast<uint16_t>(pState->timeRegister >> NN_AUDIO_SRC_BL_DN_NP);
        pState->timeRegister += pState->time_inc;
        n = static_cast<uint16_t>(pState->timeRegister >> NN_AUDIO_SRC_BL_DN_NP) - n;

        if (samplesConsumed <= pState->nz_input * 2)
        {
            /* if we need any originals get them now, calling function need to make sure enough samples in the input buffer
            if not, remaining locations should be set to 0 by calling function
            */
            if (0 != n)
            {
                for (auto i = 0; i < n ; i++)
                {
                    if (pState->numLeftovers > 0)
                    {
                        input = static_cast<int32_t>(*pLeftover++);
                        (pState->numLeftovers)--;
                    }
                    else
                    {
                        input = static_cast<int32_t>(*pIn);
                        pIn += numChannels;
                        samplesConsumed++;
                    }
                    pState->input_samples[pState->input_write++] = (input << 8); /* Q8 format */
                    pState->input_write = checkBoundsUp(pState, pState->input_write);
                }
                pState->input_current += n;
                pState->input_current = checkBoundsUp(pState, pState->input_current);
            }

            /* how far to step through filter table, expressed in same format as P */
            P = (pState->timeRegister << NN_AUDIO_SRC_BL_DN_NN) >> NN_AUDIO_SRC_BL_DN_NN;
            L_step = static_cast<uint32_t>(NN_AUDIO_SRC_BL_DN_L);
            L_step = L_step << NN_AUDIO_SRC_BL_DN_NNU;
            L_step = (L_step * pState->ratioRho) >> NQRATIO;

            v_L = FilterDownLeft(pState->input_current, P, L_step, pState);

            /* apply right wing of the filter  */
            P = (static_cast<uint32_t>(1) << NN_AUDIO_SRC_BL_DN_NP) - P;

            v_L += FilterDownRight(checkBoundsUp(pState, pState->input_current + 1), P, L_step, pState);

            /* write output, update t and y index, output samples in Q23.8 format */
            temp = static_cast<int64_t>(pState->ratioRho) * v_L;
            if (temp & (1 << (NQRATIO - 1)))            // Round, if needed
            {
                temp += (1 << (NQRATIO - 1));
            }

            /* write output  note channel interleave */
            int32_t temp1 = static_cast<int32_t>(temp >> (NQRATIO + 8));
            // temp = Multiply(temp1, gain, 15);
            *pOut = Clamp<int16_t>(Multiply(temp1, gain, 15));
            pOut = &pOut[numChannels];
        }
        else
        {
            if (0 != n)
            {
                pIn += numChannels * n;
                samplesConsumed += n;
            }
            auto pInLeft = (pIn - numChannels) - numChannels * pState->nz_input;

            /* how far to step through filter table, expressed in same format as P */
            P = (pState->timeRegister << NN_AUDIO_SRC_BL_DN_NN) >> NN_AUDIO_SRC_BL_DN_NN;
            L_step = static_cast<uint32_t>(NN_AUDIO_SRC_BL_DN_L);
            L_step = L_step << NN_AUDIO_SRC_BL_DN_NNU;
            L_step = (L_step * pState->ratioRho) >> NQRATIO;

            v_L = FilterDownLeftLinear(numChannels, pInLeft, P, L_step, pState);

            /* apply right wing of the filter  */
            P = (static_cast<uint32_t>(1) << NN_AUDIO_SRC_BL_DN_NP) - P;
            v_L += FilterDownRightLinear(numChannels, pInLeft + numChannels, P, L_step, pState);

            /* write output, update t and y index, output samples in Q23.8 format */
            temp = (static_cast<int64_t>(pState->ratioRho) * v_L);
            if (temp & (1 << (NQRATIO - 1)))            // Round, if needed
            {
                temp += (1 << (NQRATIO - 1));
            }

            /* write output  note channel interleave */
            int32_t temp1 = static_cast<int32_t>(temp >> (NQRATIO + 8));
            // temp = Multiply(temp1, gain, 15);
            *pOut = Clamp<int16_t>(Multiply(temp1, gain, 15));
            pOut = &pOut[numChannels];
        }
    }

    pState->input_current = (NN_AUDIO_SRC_BL_DN_CB_MID - 1);
    pState->input_write = pState->input_cb_lowerBound;
    pIn = (pIn - numChannels * 2 * pState->nz_input);
    for (auto j = 0; j < 2 * pState->nz_input; j++)
    {
        pState->input_samples[pState->input_write++] = static_cast<int32_t>(*pIn) << 8; /* Q8 format */
        pIn += numChannels;
    }
    pState->input_write = pState->input_cb_lowerBound;

    pState->numLeftovers = static_cast<int16_t>(inSamples) - samplesConsumed;
    if(pState->numLeftovers > sizeof(pState->leftoverSamples) / sizeof(pState->leftoverSamples[0]))
    {
        return -1;
    }

    for (auto k = 0; k < pState->numLeftovers; k++)
    {
        pState->leftoverSamples[k] = *pIn;
        pIn += numChannels;
    }

    return 0;
}

/*! \fn  int32_t SrcProcessFrame(
*    DownSamplerState* pState,
*    int32_t numChannels,
*    int32_t channel,
*    int32_t* pIn,
*    int32_t* pOut,
*    int32_t inSamples,
*    int32_t outSamples,
*    int32_t* nextSamples)
*
*  \brief  run Resampler for rho < 1
*
*  \param[in]  DownSamplerState *pState
*  \param[in]  numChannels - number of interleaved channels in *pOut
*  \param[in]  channel - which channel position this resample is to occupy in the interleave
*  \param[in]  *pIn - pointer to single channel buffer of input samples
*  \param[in]  *pOut - pointer to interleaved output buffer base (ch 0)
*  \param[in]  inSamples - number of samples that must be consumed from *pIn
*  \param[in]  outSamples - number of samples that must be generated into *pOut
*  \param[out] *nextSamples - number of samples that must be generated on the next frame of inSamples size.
*              If pOut and pIn are both nullptr, only compute *nextSamples (first time up)
*  \return 0
*/
int32_t ResampleOutputFrame(
    ResamplerState* pState,
    int32_t numChannels, int32_t channel,
    const int16_t* pIn, int16_t* pOut,
    int32_t inSamples, int32_t outSamples,
    float volume) NN_NOEXCEPT
{
    int32_t status = 0;

    NN_SDK_ASSERT(ResamplerState::ResamplerStateMagic == pState->isInitialized);

    if (nullptr != pIn && nullptr != pOut)
    {
        if (pState->ratioRho >= (static_cast<uint32_t>(1) << NQRATIO)) /* up sampling*/
        {
            status  = ResampleUpConverting(numChannels, channel, pIn, pOut, inSamples, outSamples, pState, volume);
        }
        else
        {
            status  = ResampleDownConverting(numChannels, channel, pIn, pOut, inSamples, outSamples, pState, volume);
        }
    }
    return status;
}

int ResampleOutputGetNextFrameSize(const ResamplerState* pState, int numStates, int inSamples) NN_NOEXCEPT
{
    auto outSamples = static_cast<int>((inSamples * static_cast<int64_t>(pState->ratioRho)) >> NQRATIO) + 1;  // initial estimation

    inSamples = inSamples + pState->numLeftovers;

    for (auto i = 0; i < numStates; i++)
    {
        if (ResamplerState::ResamplerStateMagic == pState->isInitialized)
        {
            int32_t deltaN  = (((static_cast<int64_t>(pState->timeRegister) + static_cast<int64_t>(pState->time_inc) * outSamples) >> NN_AUDIO_SRC_BL_DN_NP) - (pState->timeRegister >> NN_AUDIO_SRC_BL_DN_NP)); // & 0x000000ff;
            while (deltaN < inSamples)
            {
                ++outSamples;
                deltaN  = (((static_cast<int64_t>(pState->timeRegister) + static_cast<int64_t>(pState->time_inc) * outSamples) >> NN_AUDIO_SRC_BL_DN_NP) - (pState->timeRegister >> NN_AUDIO_SRC_BL_DN_NP)); // & 0x000000ff;
            }
            while (deltaN > inSamples)
            {
                --outSamples;
                deltaN  = (((static_cast<int64_t>(pState->timeRegister) + static_cast<int64_t>(pState->time_inc) * outSamples) >> NN_AUDIO_SRC_BL_DN_NP) - (pState->timeRegister >> NN_AUDIO_SRC_BL_DN_NP)); // & 0x000000ff;
            }
            break;
        }
        ++pState;
    }
    return outSamples;
}

size_t GetRequiredBufferSizeForResampler(int channelCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(channelCount > 0, "Channel count must be greater than 0");
    NN_SDK_REQUIRES(channelCount <= ResamplerType::ChannelCountMax, "Channel count must be less than or equal to ChannelCountMax");
    size_t resamplerWorkBufferSize = sizeof(ResamplerState) * channelCount;
    resamplerWorkBufferSize = nn::util::align_up(resamplerWorkBufferSize, nn::audio::BufferAlignSize);

    return resamplerWorkBufferSize;
}

size_t GetResampledOutBufferSize(const ResamplerType* pResampler, size_t inputBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pResampler);
    const auto ratio = static_cast<float>(pResampler->_outSampleRate) / pResampler->_inSampleRate;
    const auto inputSampleCount = static_cast<int>(inputBufferSize / sizeof(int16_t) / pResampler->_channelCount);
    const auto extraSampleCount = 2;  // extra sample space to be safe
    const auto estimatedOutputSampleCount = static_cast<int>(std::ceil(inputSampleCount * ratio)) + extraSampleCount;
    return estimatedOutputSampleCount * sizeof(int16_t) * pResampler->_channelCount;
}

int GetResamplerOutputSampleCount(const ResamplerType* pResampler, int inputSampleCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pResampler);
    if (inputSampleCount <= 0)
    {
        return 0;
    }
    const auto ratio = static_cast<float>(pResampler->_outSampleRate) / pResampler->_inSampleRate;
    const auto extraSampleCount = 2;  // extra sample space to be safe
    return static_cast<int>(std::ceil(inputSampleCount * ratio)) + extraSampleCount;
}

Result InitializeResampler(ResamplerType* pResampler, void* buffer, size_t bufferSize, int inputSampleRate, int outputSampleRate, int channelCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pResampler);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(inputSampleRate > 0, "Input sample rate must be greater than 0");
    NN_SDK_REQUIRES(outputSampleRate > 0, "Output sample rate must be greater than 0");
    auto requiredBufferSize = GetRequiredBufferSizeForResampler(channelCount);
    NN_SDK_ASSERT_GREATER_EQUAL(bufferSize, requiredBufferSize);
    NN_UNUSED(requiredBufferSize);
    auto ratio = static_cast<uint32_t>((static_cast<uint64_t>(1 << nn::audio::NQRATIO) * outputSampleRate) / inputSampleRate);
    float ratioInFloat = static_cast<float>(outputSampleRate) / inputSampleRate;
    NN_SDK_ASSERT_LESS_EQUAL(ratioInFloat, ResamplerType::GetConvertRatioMax());
    NN_SDK_ASSERT_GREATER_EQUAL(ratioInFloat, ResamplerType::GetConvertRatioMin());
    NN_UNUSED(ratioInFloat);

    pResampler->_workBuffer = buffer;
    pResampler->_bufferSize = bufferSize;
    pResampler->_inSampleRate = inputSampleRate;
    pResampler->_outSampleRate = outputSampleRate;
    pResampler->_channelCount = channelCount;
    auto pResamplerStates = reinterpret_cast<ResamplerState*> (buffer);
    memset(pResamplerStates, 0, bufferSize);
    for (int i = 0; i < channelCount; ++i)
    {
        InitializeResampler(&pResamplerStates[i], ratio);
    }

    NN_RESULT_SUCCESS;
}

Result ProcessResamplerBuffer(ResamplerType* pResampler, int* pOutputSampleCount, int16_t* pOutputBuffer, size_t outputBufferSize, const int16_t* pInputBuffer, int inputSampleCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pResampler);
    NN_SDK_REQUIRES_NOT_NULL(pOutputBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pInputBuffer);
    auto pResamplerStates = reinterpret_cast<ResamplerState*> (pResampler->_workBuffer);
    NN_SDK_REQUIRES(pResamplerStates->isInitialized == ResamplerState::ResamplerStateMagic, "ResamplerType must be initialized");

    const auto channelCount = pResampler->_channelCount;

    const auto outputSampleCount = ResampleOutputGetNextFrameSize(&pResamplerStates[0], 1, inputSampleCount);
    for (auto channel = 1; channel < channelCount; channel++)
    {
        NN_SDK_ASSERT(outputSampleCount == ResampleOutputGetNextFrameSize(&pResamplerStates[channel], 1, inputSampleCount));
    }
    if (pOutputSampleCount)
    {
        *pOutputSampleCount = outputSampleCount;
    }
    NN_RESULT_THROW_UNLESS(outputBufferSize >= outputSampleCount * channelCount * sizeof(int16_t), ResultInsufficientBuffer());
    for (auto channel = 0; channel < channelCount; channel++)
    {
        auto status = ResampleOutputFrame(&pResamplerStates[channel], channelCount, channel, pInputBuffer, pOutputBuffer, inputSampleCount, outputSampleCount, 1.0f);
        NN_SDK_ASSERT(status == 0);
        NN_UNUSED(status);
    }
    NN_RESULT_SUCCESS;
}

}  // namespace audio
}  // namespace nn
