﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring> // std::memcpy
#include <limits>

#include <nn/nn_SdkAssert.h>
#include <nn/util/util_BytePtr.h>
#include <nn/util/util_PlacementArray.h>

#include <nn/audio/audio_PerformanceMetrics.h>
#include <nn/audio/detail/audio_AudioRendererTypesInternal.h>

#include "../common/audio_Command.h"
#include "../common/audio_Util.h"
#include "../common/audio_PerformanceMetricsCommon.h"
#include "audio_PerformanceMetricsManager.h"

namespace nn { namespace audio { namespace server {

NN_STATIC_ASSERT(sizeof(PerformanceEntry) % 4 == 0);
NN_STATIC_ASSERT(sizeof(PerformanceDetail) % 4 == 0);


//================================================================================
// PerformanceManager
//================================================================================
size_t PerformanceManager::GetRequiredBufferSizeForPerformanceMetricsPerFrame(const BehaviorInfo& behavior, const detail::AudioRendererParameterInternal& parameter) NN_NOEXCEPT
{
    switch(behavior.GetPerformanceMetricsDataFormat())
    {
    case PerformanceMetricsDataFormat::Version1:
        return PerformanceManagerImplVersion1::GetRequiredBufferSizeForPerformanceMetricsPerFrame(parameter);
    case PerformanceMetricsDataFormat::Version2:
        return PerformanceManagerImplVersion2::GetRequiredBufferSizeForPerformanceMetricsPerFrame(parameter);
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

IPerformanceManagerImpl* PerformanceManager::CreateImpl(PerformanceMetricsDataFormat format) NN_NOEXCEPT
{
    switch(format)
    {
    case PerformanceMetricsDataFormat::Version1:
        return new(m_WorkBuffer) PerformanceManagerImplVersion1();
    case PerformanceMetricsDataFormat::Version2:
        return new(m_WorkBuffer) PerformanceManagerImplVersion2();
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void PerformanceManager::Initialize(void* buffer, size_t bufferSize,
                                    const detail::AudioRendererParameterInternal& parameter,
                                    const BehaviorInfo& behavior,
                                    const server::MemoryPoolInfo& servicePool
                                    ) NN_NOEXCEPT
{
    m_Impl = CreateImpl(behavior.GetPerformanceMetricsDataFormat());
    return m_Impl->Initialize(buffer, bufferSize, parameter, behavior, servicePool);
}

bool PerformanceManager::IsInitialized() NN_NOEXCEPT
{
    return (m_Impl != nullptr) && m_Impl->IsInitialized();
}

size_t PerformanceManager::CopyHistories(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Impl);

    return m_Impl->CopyHistories(buffer, bufferSize);
}

bool PerformanceManager::GetNextEntry(PerformanceEntryAddresses* block, PerformanceEntryType targetType, NodeId targetId) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Impl);

    return m_Impl->GetNextEntry(block, targetType, targetId);
}

bool PerformanceManager::GetNextEntry(PerformanceEntryAddresses* block, PerformanceDetailType targetType, PerformanceEntryType parentType, NodeId parentId) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Impl);

    return m_Impl->GetNextEntry(block, targetType, parentType, parentId);
}

bool PerformanceManager::GetNextEntry(PerformanceEntryAddresses* block, uint32_t** ppEstimatedProcessingTime, PerformanceSysDetailType target, NodeId parentId) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Impl);

    return m_Impl->GetNextEntry(block, ppEstimatedProcessingTime, target, parentId);
}

void PerformanceManager::TapFrame(bool isRenderingTimeLimitExceeded, int voiceDropCount, int64_t startTime) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Impl);

    return m_Impl->TapFrame(isRenderingTimeLimitExceeded, voiceDropCount, startTime);
}

bool PerformanceManager::IsDetailTarget(NodeId id) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Impl);

    return m_Impl->IsDetailTarget(id);
}

void PerformanceManager::SetDetailTarget(NodeId id) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Impl);

    return m_Impl->SetDetailTarget(id);
}

template<>
void PerformanceManagerImplVersion1::UpdateHeader(common::PerformanceFrameHeaderVersion1* pDestHeader, const common::PerformanceFrameHeaderVersion1* pSrcHeader, int totalProcessingTime, size_t size, int entryCount, int detailCount) NN_NOEXCEPT
{
    pDestHeader->frameMagic = common::CreateMagic(common::PerformanceFrameHeaderMagic);
    pDestHeader->totalProcessingTime = totalProcessingTime;
    pDestHeader->offsetToNextHeader = static_cast<int32_t>(size);
    pDestHeader->entryCount = entryCount;
    pDestHeader->detailCount= detailCount;
    pDestHeader->frameIndex = pSrcHeader->frameIndex;
}

template<>
void PerformanceManagerImplVersion2::UpdateHeader(common::PerformanceFrameHeaderVersion2* pDestHeader, const common::PerformanceFrameHeaderVersion2* pSrcHeader, int totalProcessingTime, size_t size, int entryCount, int detailCount) NN_NOEXCEPT
{
    pDestHeader->frameMagic = common::CreateMagic(common::PerformanceFrameHeaderMagic);
    pDestHeader->totalProcessingTime = totalProcessingTime;
    pDestHeader->offsetToNextHeader = static_cast<int32_t>(size);
    pDestHeader->entryCount = entryCount;
    pDestHeader->detailCount= detailCount;
    pDestHeader->frameIndex = pSrcHeader->frameIndex;
    pDestHeader->isRenderingTimeLimitExceeded = pSrcHeader->isRenderingTimeLimitExceeded;
    pDestHeader->voiceDropCount = pSrcHeader->voiceDropCount;
    pDestHeader->startTime = pSrcHeader->startTime;
}

template<>
void PerformanceManagerImplVersion1::CopyFrame(common::PerformanceFrameHeaderVersion1* destFrame, const common::PerformanceFrameHeaderVersion1* srcFrame, bool isRenderingTimeLimitExceeded, int voiceDropCount, int64_t startTime) NN_NOEXCEPT
{
    NN_UNUSED(isRenderingTimeLimitExceeded);
    NN_UNUSED(voiceDropCount);
    NN_UNUSED(startTime);

    std::memcpy(destFrame, srcFrame, m_OneFrameSize);
    auto pHeader = destFrame;
    pHeader->frameIndex = GetFrameIndex();
}

template<>
void PerformanceManagerImplVersion2::CopyFrame(common::PerformanceFrameHeaderVersion2* destFrame, const common::PerformanceFrameHeaderVersion2* srcFrame, bool isRenderingTimeLimitExceeded, int voiceDropCount, int64_t startTime) NN_NOEXCEPT
{
    std::memcpy(destFrame, srcFrame, m_OneFrameSize);

    auto pHeader = destFrame;
    pHeader->isRenderingTimeLimitExceeded = isRenderingTimeLimitExceeded;
    pHeader->voiceDropCount = voiceDropCount;
    pHeader->startTime = startTime;
    pHeader->frameIndex = GetFrameIndex();
}

template<>
bool PerformanceManagerImplVersion1::GetNextEntry(PerformanceEntryAddresses* block, uint32_t** ppEstimatedProcessingTime, PerformanceSysDetailType target, NodeId parentId) NN_NOEXCEPT
{
    NN_UNUSED(block);
    NN_UNUSED(ppEstimatedProcessingTime);
    NN_UNUSED(target);
    NN_UNUSED(parentId);

    // Nothing to do.
    return false;
}

template<>
bool PerformanceManagerImplVersion2::GetNextEntry(PerformanceEntryAddresses* block, uint32_t** ppEstimatedProcessingTime, PerformanceSysDetailType target, NodeId parentId) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(block);

    if (m_Initialized == false || m_DetailOffset > nn::audio::common::PerformanceDetailCountMax)
    {
        return false;
    }

    block->dspFrameBase = m_DspMappedBuffer;

    auto bufferBase = reinterpret_cast<uintptr_t>(m_DspBuffer);
    block->offsetToCount = static_cast<int32_t>(reinterpret_cast<uintptr_t>(&m_CurrentFrame.header->detailCount) - bufferBase);

    auto sysDetail = reinterpret_cast<PerformanceSysDetail*>(m_CurrentFrame.detail + m_DetailOffset);
    block->offsetToStartTime = static_cast<int32_t>(reinterpret_cast<uintptr_t>(&sysDetail->startTime) - bufferBase);
    block->offsetToProcessingTime = static_cast<int32_t>(reinterpret_cast<uintptr_t>(&sysDetail->processingTime) - bufferBase);

    ++m_DetailOffset;
    memset(sysDetail, 0, sizeof(PerformanceSysDetail));

    // Set type & id in advance.
    sysDetail->detailType = static_cast<int8_t>(target);
    sysDetail->parentEntryType = static_cast<int8_t>(PerformanceEntryType_Unknown);
    sysDetail->parentId = parentId;

    if(ppEstimatedProcessingTime != nullptr)
    {
        *ppEstimatedProcessingTime = &sysDetail->estimatedProcessingTime;
    }

    return true;
}

}}} // namespace nn::audio::server
