﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdint>
#include <nn/nn_SdkAssert.h>
#include "audio_PerformanceMetricsManager.h"
#include "audio_ServiceVoiceInfo.h"
#include "audio_ServiceMemoryPoolInfo.h"
#include "audio_ServiceMixInfo.h"
#include "audio_ServiceSinkInfo.h"
#include "audio_CommandProcessingTimeEstimator.h"
#include "audio_CommandBuffer.h"

namespace nn { namespace audio { namespace server {

namespace {

// 1 audio frame あたりの目標減衰量を -80 dB とする
// 20 * math.log10(0.0001) = -80dB より
// depop ratio は 1 audio frame 分の乗算の結果、 0.0001 以下となればよい
//
// 1 audio frame あたりの乗算回数は
// 48k : 240 回
// 32k : 160 回
// であるので、これよりそれぞれの係数は以下のように計算できる。
// 48k : format(int(pow(0.0001, 1./(240 - 1)) * (1 << Q)), "04x") = 0x7b29
// 32k : format(int(pow(0.0001, 1./(160 - 1)) * (1 << Q)), "04x") = 0x78cb
const int32_t DepopDecayRatio48k = 0x7b29;
const int32_t DepopDecayRatio32k = 0x78cb;

}

CommandBuffer::CommandBuffer(void* commandBuffer, size_t bufferSize, size_t initialOffset, const server::MemoryPoolInfo& servicePool, ICommandProcessingTimeEstimator* pEstimator) NN_NOEXCEPT
    : m_BufferBase(reinterpret_cast<uintptr_t>(commandBuffer))
    , m_BufferSize(bufferSize)
    , m_Offset(initialOffset)
    , m_Count(0)
    , m_EstimatedProcessingTime(0u)
    , m_ServicePool(servicePool)
    , m_pEstimator(pEstimator)
{

}

void* CommandBuffer::GetBuffer() const NN_NOEXCEPT
{
    return reinterpret_cast<void*>(m_BufferBase);
}

size_t CommandBuffer::GetCommandTotalSize() const NN_NOEXCEPT
{
    return m_Offset;
}

int CommandBuffer::GetCount() const NN_NOEXCEPT
{
    return m_Count;
}

uint32_t CommandBuffer::GetEstimatedProcessingTime() const NN_NOEXCEPT
{
    return m_EstimatedProcessingTime;
}

void* CommandBuffer::GetCurrentBufferBase() const NN_NOEXCEPT
{
    return reinterpret_cast<void*>(m_BufferBase + m_Offset);
}

CommandBuffer::CommandBufferState CommandBuffer::GetState() const NN_NOEXCEPT
{
    CommandBufferState state = {m_Offset, m_Count, m_EstimatedProcessingTime};
    return state;
}

void CommandBuffer::SetState(const CommandBufferState& state) NN_NOEXCEPT
{
    m_Offset = state.offset;
    m_Count = state.commandCount;
    m_EstimatedProcessingTime = state.estimatedProcessingTime;
}

PcmInt16DataSourceCommand* CommandBuffer::GeneratePcmInt16DataSourceCommand(const server::VoiceInfo* pVoice, VoiceState* state, int mixBufferCount, int32_t channel, NodeId nodeId) NN_NOEXCEPT
{
    const auto channelCount = pVoice->_inParameter._channelCount;

    auto c = CreateCommand<PcmInt16DataSourceCommand>(nodeId);
    c->outputBufferIndex = static_cast<int16_t>(mixBufferCount + channel);
    c->sampleRate = pVoice->_inParameter._sampleRate;
    c->pitch = static_cast<int32_t>(pVoice->_inParameter._pitch * (1 << 15));
    c->channel = channel;
    c->channelCount = channelCount;
    for(int i = 0; i < VoiceType::WaveBufferCountMax; ++i)
    {
        auto pDest = &(c->waveBuffer[i]);
        auto pSrc = &(pVoice->_inParameter._waveBuffer[i]);
        pDest->buffer =            pSrc->buffer.GetReference();
        pDest->size =              static_cast<uint32_t>(pSrc->buffer.GetSize());
        pDest->startSampleOffset = pSrc->startSampleOffset;
        pDest->endSampleOffset =   pSrc->endSampleOffset;
        pDest->loop =              pSrc->loop;
        pDest->isEndOfStream =     pSrc->isEndOfStream;
        pDest->pContext =          0;
        pDest->contextSize =       0;
    }
    NN_SDK_ASSERT(m_ServicePool.Contains(state, sizeof(VoiceState)));
    c->state = m_ServicePool.Translate(state, sizeof(VoiceState));

    const auto isPlayedSampleCountResetAtLoopPoint = pVoice->_inParameter._behaviorFlags.Get<VoiceBehaviorFlag::IsPlayedSampleCountResetAtLoopPoint>();
    const auto isPitchAndSrcSkipped = pVoice->_inParameter._behaviorFlags.Get<VoiceBehaviorFlag::IsPitchAndSrcSkipped>();
    uint16_t behaviorFlags = 0u;

    if(isPlayedSampleCountResetAtLoopPoint)
    {
        behaviorFlags |= DataSourceBehaviorFlag_IsPlayedSampleCountResetAtLoopPoint;
    }

    if(isPitchAndSrcSkipped)
    {
        behaviorFlags |= DataSourceBehaviorFlag_IsPitchAndSrcSkipped;
    }
    c->behaviorFlags = behaviorFlags;

    EstimateProcessTime(c);

    return c;
}

AdpcmDataSourceCommand* CommandBuffer::GenerateAdpcmDataSourceCommand(const server::VoiceInfo* pVoice, VoiceState* state, int mixBufferCount, NodeId nodeId) NN_NOEXCEPT
{
    auto c = CreateCommand<AdpcmDataSourceCommand>(nodeId);
    c->outputBufferIndex = static_cast<int16_t>(mixBufferCount);
    c->sampleRate = pVoice->_inParameter._sampleRate;
    c->pitch = static_cast<int32_t>(pVoice->_inParameter._pitch * (1 << 15));
    for(int i = 0; i < VoiceType::WaveBufferCountMax; ++i)
    {
        auto pDest = &(c->waveBuffer[i]);
        auto pSrc = &(pVoice->_inParameter._waveBuffer[i]);
        pDest->buffer =            pSrc->buffer.GetReference();
        pDest->size =              static_cast<uint32_t>(pSrc->buffer.GetSize());
        pDest->startSampleOffset = pSrc->startSampleOffset;
        pDest->endSampleOffset =   pSrc->endSampleOffset;
        pDest->loop =              pSrc->loop;
        pDest->isEndOfStream =     pSrc->isEndOfStream;
        pDest->pContext =          pSrc->pContextInfo.GetReference();
        pDest->contextSize =       static_cast<uint32_t>(pSrc->pContextInfo.GetSize());
    }
    NN_SDK_ASSERT(m_ServicePool.Contains(state, sizeof(VoiceState)));
    c->state = m_ServicePool.Translate(state, sizeof(VoiceState));
    c->adpcmParameter = pVoice->_inParameter._pAdditionalParameterInfo.GetReference();

    const auto isPlayedSampleCountResetAtLoopPoint = pVoice->_inParameter._behaviorFlags.Get<VoiceBehaviorFlag::IsPlayedSampleCountResetAtLoopPoint>();
    const auto isPitchAndSrcSkipped = pVoice->_inParameter._behaviorFlags.Get<VoiceBehaviorFlag::IsPitchAndSrcSkipped>();
    uint16_t behaviorFlags = 0u;

    if(isPlayedSampleCountResetAtLoopPoint)
    {
        behaviorFlags |= DataSourceBehaviorFlag_IsPlayedSampleCountResetAtLoopPoint;
    }

    if(isPitchAndSrcSkipped)
    {
        behaviorFlags |= DataSourceBehaviorFlag_IsPitchAndSrcSkipped;
    }
    c->behaviorFlags = behaviorFlags;

    EstimateProcessTime(c);

    return c;
}

OpusDataSourceCommand* CommandBuffer::GenerateOpusDataSourceCommand(const server::VoiceInfo* pVoice, VoiceState* state, int mixBufferCount, NodeId nodeId) NN_NOEXCEPT
{
    auto c = CreateCommand<OpusDataSourceCommand>(nodeId);
    c->outputBufferIndex = static_cast<int16_t>(mixBufferCount);
    c->sampleRate = pVoice->_inParameter._sampleRate;
    c->pitch = static_cast<int32_t>(pVoice->_inParameter._pitch * (1 << 15));
    for(int i = 0; i < VoiceType::WaveBufferCountMax; ++i)
    {
        auto pDest = &(c->waveBuffer[i]);
        auto pSrc = &(pVoice->_inParameter._waveBuffer[i]);
        pDest->buffer =            pSrc->buffer.GetReference();
        pDest->size =              static_cast<uint32_t>(pSrc->buffer.GetSize());
        pDest->startSampleOffset = pSrc->startSampleOffset;
        pDest->endSampleOffset =   pSrc->endSampleOffset;
        pDest->loop =              pSrc->loop;
        pDest->isEndOfStream =     pSrc->isEndOfStream;
        pDest->pContext =          0;
        pDest->contextSize =       0;
    }
    NN_SDK_ASSERT(m_ServicePool.Contains(state, sizeof(VoiceState)));
    c->state = m_ServicePool.Translate(state, sizeof(VoiceState));
    c->opusParameter = pVoice->_inParameter._pAdditionalParameterInfo.GetReference();

    const auto isPlayedSampleCountResetAtLoopPoint = pVoice->_inParameter._behaviorFlags.Get<VoiceBehaviorFlag::IsPlayedSampleCountResetAtLoopPoint>();
    const auto isPitchAndSrcSkipped = pVoice->_inParameter._behaviorFlags.Get<VoiceBehaviorFlag::IsPitchAndSrcSkipped>();
    uint16_t behaviorFlags = 0u;

    if(isPlayedSampleCountResetAtLoopPoint)
    {
        behaviorFlags |= DataSourceBehaviorFlag_IsPlayedSampleCountResetAtLoopPoint;
    }

    if(isPitchAndSrcSkipped)
    {
        behaviorFlags |= DataSourceBehaviorFlag_IsPitchAndSrcSkipped;
    }
    c->behaviorFlags = behaviorFlags;

    EstimateProcessTime(c);

    return c;
}

VolumeCommand* CommandBuffer::GenerateVolumeCommand(float volume, int mixBufferCount, NodeId nodeId) NN_NOEXCEPT
{
    auto c = CreateCommand<VolumeCommand>(nodeId);
    c->inputBufferIndex = static_cast<int16_t>(mixBufferCount);
    c->outputBufferIndex = static_cast<int16_t>(mixBufferCount);
    c->volume = static_cast<int32_t>(volume * (1 << 15));
    EstimateProcessTime(c);

    return c;
}

VolumeRampCommand* CommandBuffer::GenerateVolumeRampCommand(float volume0, float volume1, int mixBufferCount, NodeId nodeId) NN_NOEXCEPT
{
    auto c = CreateCommand<VolumeRampCommand>(nodeId);
    c->inputBufferIndex = static_cast<int16_t>(mixBufferCount);
    c->outputBufferIndex = static_cast<int16_t>(mixBufferCount);
    c->volume0 = static_cast<int32_t>(volume0 * (1 << 15));
    c->volume1 = volume0 == volume1 ? c->volume0 : static_cast<int32_t>(volume1 * (1 << 15));
    EstimateProcessTime(c);

    return c;
}

BiquadFilterCommand* CommandBuffer::GenerateBiquadFilterCommand(int mixBufferOffset, const BiquadFilterParameter* pFilterParameter, BiquadFilterState* pState, int8_t input, int8_t output, bool needInitialization, NodeId nodeId) NN_NOEXCEPT
{
    auto c = CreateCommand<BiquadFilterCommand>(nodeId);
    c->inputBufferIndex = static_cast<int16_t>(input + mixBufferOffset);
    c->outputBufferIndex = static_cast<int16_t>(output + mixBufferOffset);
    memcpy(c->numerator, pFilterParameter->numerator, sizeof(pFilterParameter->numerator));
    memcpy(c->denominator, pFilterParameter->denominator, sizeof(pFilterParameter->denominator));
    NN_SDK_ASSERT(m_ServicePool.Contains(pState, sizeof(BiquadFilterState)));
    c->state = m_ServicePool.Translate(pState, sizeof(BiquadFilterState));

    c->needInitialization = needInitialization;

    EstimateProcessTime(c);

    return c;
}

DepopPrepareCommand* CommandBuffer::GenerateDepopPrepareCommand(VoiceState* state, int32_t* depopBuffer, int mixBufferCount, int mixBufferOffset, NodeId nodeId, bool commandEnabled) NN_NOEXCEPT
{
    auto c = CreateCommand<DepopPrepareCommand>(nodeId);
    c->header.isEnabled = commandEnabled;
    for (auto i = 0; i < MixBufferCountMax; ++i)
    {
        c->outputBufferIndices[i] = static_cast<int16_t>(i + mixBufferOffset);
    }
    NN_SDK_ASSERT(m_ServicePool.Contains(state->lastSamples, sizeof(int32_t) * MixBufferCountMax));
    c->state = m_ServicePool.Translate(state->lastSamples, sizeof(int32_t) * MixBufferCountMax);
    c->mixBufferCount = mixBufferCount;
    NN_SDK_ASSERT(m_ServicePool.Contains(depopBuffer, sizeof(int32_t) * mixBufferCount));
    c->depopBuffer = m_ServicePool.Translate(depopBuffer, sizeof(int32_t) * mixBufferCount);
    EstimateProcessTime(c);

    return c;
}

DepopForMixBuffersCommand* CommandBuffer::GenerateDepopForMixBuffersCommand(int32_t* depopBuffer, int mixBufferOffset, int mixBufferCount, NodeId nodeId, int32_t sampleRate) NN_NOEXCEPT
{
    auto c = CreateCommand<DepopForMixBuffersCommand>(nodeId);
    c->mixBufferOffset = mixBufferOffset;
    c->mixBufferCount = mixBufferCount;
    switch (sampleRate)
    {
    case 48000:
        c->decay = DepopDecayRatio48k;
        break;
    case 32000:
    default: // より急峻に収束する方を default にしておく
        c->decay = DepopDecayRatio32k;
        break;
    }
    NN_SDK_ASSERT(m_ServicePool.Contains(depopBuffer, sizeof(int32_t) * mixBufferCount));
    c->depopBuffer = m_ServicePool.Translate(depopBuffer, sizeof(int32_t) * mixBufferCount);
    EstimateProcessTime(c);

    return c;
}

AuxCommand* CommandBuffer::GenerateAuxCommand(int mixBufferOffset, int8_t input, int8_t output, const AuxBufferAddresses* pAddresses, bool enabled, int32_t sampleCount, DspAddr sendBuffer, DspAddr returnBuffer, int updateCount, int offset, NodeId nodeId) NN_NOEXCEPT
{
    if (pAddresses->_returnBufferBase == NULL ||
        pAddresses->_sendBufferBase == NULL)
    {
        return nullptr;
    }

    auto c = CreateCommand<AuxCommand>(nodeId);
    c->inputBufferIndex = static_cast<int16_t>(input + mixBufferOffset);
    c->outputBufferIndex = static_cast<int16_t>(output + mixBufferOffset);
    c->inputAuxInfo = pAddresses->_sendDspInfo;
    c->inputBuffer = sendBuffer;
    c->outputAuxInfo = pAddresses->_returnDspInfo;
    c->outputBuffer = returnBuffer;
    c->countMax = sampleCount;
    c->updateCount = updateCount;
    c->writeOffset = offset;
    c->enabled = enabled;
    EstimateProcessTime(c);

    return c;
}

DelayCommand* CommandBuffer::GenerateDelayEffectCommand(int mixBufferOffset, const DelayParameter* pParameter, void* pState, bool enabled, DspAddr workBuffer, NodeId nodeId) NN_NOEXCEPT
{
    if (IsSupportedDelayChannelCount(pParameter->_numChannels) == false)
    {
        return nullptr;
    }

    auto stateAddr = m_ServicePool.Translate(pState, sizeof(DelayState));
    if (stateAddr == NULL)
    {
        return nullptr;
    }

    auto c = CreateCommand<DelayCommand>(nodeId);
    for (int i = 0; i < pParameter->_numChannels; i++)
    {
        c->inputBufferIndex[i] = static_cast<int16_t>(pParameter->_input[i] + mixBufferOffset);
        c->outputBufferIndex[i] = static_cast<int16_t>(pParameter->_output[i] + mixBufferOffset);
    }
    c->delayConfig = *pParameter;
    c->delayState = stateAddr;
    c->enabled = enabled;
    c->workBuffer = workBuffer;
    EstimateProcessTime(c);

    return c;
}

ReverbCommand* CommandBuffer::GenerateReverbEffectCommand(int mixBufferOffset, const ReverbParameter* pParameter, void* pState, bool enabled, bool longSizePreDelaySupported, DspAddr workBuffer, NodeId nodeId) NN_NOEXCEPT
{
    if (IsSupportedReverbChannelCount(pParameter->_numChannels) == false)
    {
        return nullptr;
    }

    auto stateAddr = m_ServicePool.Translate(pState, sizeof(ReverbState));
    if (stateAddr == NULL)
    {
        return nullptr;
    }

    auto c = CreateCommand<ReverbCommand>(nodeId);
    for (int i = 0; i < pParameter->_numChannels; i++)
    {
        c->inputBufferIndex[i] = static_cast<int16_t>(pParameter->_input[i] + mixBufferOffset);
        c->outputBufferIndex[i] = static_cast<int16_t>(pParameter->_output[i] + mixBufferOffset);
    }
    c->reverbConfig = *pParameter;
    c->reverbState = stateAddr;
    c->enabled = enabled;
    c->longSizePreDelaySupported = longSizePreDelaySupported;
    c->workBuffer = workBuffer;
    EstimateProcessTime(c);

    return c;
}

I3dl2ReverbCommand* CommandBuffer::GenerateI3dl2ReverbEffectCommand(int mixBufferOffset, const I3dl2ReverbParameter* pParameter, void* pState, bool enabled, DspAddr workBuffer, NodeId nodeId) NN_NOEXCEPT
{
    if (IsSupportedI3dl2ReverbChannelCount(pParameter->_numChannels) == false)
    {
        return nullptr;
    }

    auto stateAddr = m_ServicePool.Translate(pState, sizeof(I3dl2ReverbState));
    if (stateAddr == NULL)
    {
        return nullptr;
    }

    auto c = CreateCommand<I3dl2ReverbCommand>(nodeId);
    for (int i = 0; i < pParameter->_numChannels; i++)
    {
        c->inputBufferIndex[i] = static_cast<int16_t>(pParameter->_input[i] + mixBufferOffset);
        c->outputBufferIndex[i] = static_cast<int16_t>(pParameter->_output[i] + mixBufferOffset);
    }
    c->i3dl2ReverbConfig = *pParameter;
    c->i3dl2ReverbState = stateAddr;
    c->enabled = enabled;
    c->workBuffer = workBuffer;
    EstimateProcessTime(c);

    return c;
}

CopyMixBufferCommand* CommandBuffer::GenerateCopyMixBufferCommand(int inputBufferIndex, int outputBufferIndex, NodeId nodeId) NN_NOEXCEPT
{
    auto c = CreateCommand<CopyMixBufferCommand>(nodeId);
    c->inputBufferIndex = static_cast<int16_t>(inputBufferIndex);
    c->outputBufferIndex = static_cast<int16_t>(outputBufferIndex);
    EstimateProcessTime(c);

    return c;
}

MixCommand* CommandBuffer::GenerateMixCommand(const int inputBufferIndex, const int outputBufferIndex, const float mixVolume, NodeId nodeId) NN_NOEXCEPT
{
    auto c = CreateCommand<MixCommand>(nodeId);
    c->inputBufferIndex = static_cast<int16_t>(inputBufferIndex);
    c->outputBufferIndex = static_cast<int16_t>(outputBufferIndex);
    c->volume = static_cast<int32_t>(mixVolume * (1 << 15));
    EstimateProcessTime(c);

    return c;
}

MixRampCommand* CommandBuffer::GenerateMixRampCommand(const int inputBufferIndex, const int outputBufferIndex, const float mixVolume0, const float mixVolume1, DspAddr lastSampleAddress, NodeId nodeId) NN_NOEXCEPT
{
    auto c = CreateCommand<MixRampCommand>(nodeId);
    c->inputBufferIndex = static_cast<int16_t>(inputBufferIndex);
    c->outputBufferIndex = static_cast<int16_t>(outputBufferIndex);
    c->volume0 = static_cast<int32_t>(mixVolume0 * (1 << 15));
    c->volume1 = mixVolume0 == mixVolume1 ? c->volume0 : static_cast<int32_t>(mixVolume1 * (1 << 15));
    c->state = lastSampleAddress;
    EstimateProcessTime(c);

    return c;
}

MixRampGroupedCommand* CommandBuffer::GenerateMixRampGroupedCommand(int mixBufferCount, int inputBufferIndex, int outputBufferIndexStart, const float* pMixVolume, const float* pMixVolumePrev, DspAddr lastSampleAddress, NodeId nodeId) NN_NOEXCEPT
{
    auto c = CreateCommand<MixRampGroupedCommand>(nodeId);
    c->mixBufferCount = mixBufferCount;
    for (auto i = 0; i < mixBufferCount; ++i)
    {
        c->inputBufferIndices[i] = static_cast<int16_t>(inputBufferIndex);
        c->outputBufferIndices[i] = static_cast<int16_t>(outputBufferIndexStart + i);
        const auto mixVolume0 = pMixVolumePrev[i];
        const auto mixVolume1 = pMixVolume[i];
        c->volume0[i] = static_cast<int32_t>(mixVolume0 * (1 << 15));
        c->volume1[i] = mixVolume0 == mixVolume1 ? c->volume0[i] : static_cast<int32_t>(mixVolume1 * (1 << 15));
    }
    c->state = lastSampleAddress;
    EstimateProcessTime(c);

    return c;
}

ClearMixBufferCommand* CommandBuffer::GenerateClearMixBufferCommand(NodeId nodeId) NN_NOEXCEPT
{
    auto c = CreateCommand<ClearMixBufferCommand>(nodeId);
    EstimateProcessTime(c);

    return c;
}

DeviceSinkCommand* CommandBuffer::GenerateDeviceSinkCommand(int mixBufferOffset, const server::SinkInfoBase* pSinkInfo, uint32_t sessionId, DspAddr mixBufferAddr, NodeId nodeId) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(pSinkInfo->GetType(), common::SinkType_Device);

    auto param = reinterpret_cast<const common::DeviceParameter*>(pSinkInfo->GetParameter());
    auto state = reinterpret_cast<const DeviceSinkState*>(pSinkInfo->GetState());

    auto c = CreateCommand<DeviceSinkCommand>(nodeId);
    std::strncpy(c->name, param->id, sizeof(c->name) - 1);
    c->sessionId = sessionId;
    c->inputCount = param->inputCount;
    for (auto i = 0; i < param->inputCount; ++i)
    {
        c->input[i] = param->input[i] + static_cast<int16_t>(mixBufferOffset);
    }

    if (auto pUpsamplerInfo = state->pUpsamplerInfo)
    {
        NN_SDK_ASSERT(m_ServicePool.Contains(pUpsamplerInfo->pOutBuffers, sizeof(int32_t) * pUpsamplerInfo->sampleCount * param->inputCount));
        c->pMixBuffers = m_ServicePool.Translate(pUpsamplerInfo->pOutBuffers, sizeof(int32_t) * pUpsamplerInfo->sampleCount * param->inputCount);
    }
    else
    {
        // The list header's buffer pointer is already mapped, so it doesn't need to be mapped here.
        c->pMixBuffers = mixBufferAddr;
    }
    EstimateProcessTime(c);

    return c;
}

CircularBufferSinkCommand* CommandBuffer::GenerateCircularBufferSinkCommand(int mixBufferOffset, const server::SinkInfoBase* pSinkInfo, NodeId nodeId) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(pSinkInfo->GetType(), common::SinkType_CircularBuffer);

    auto param = reinterpret_cast<const common::CircularBufferParameter*>(pSinkInfo->GetParameter());
    auto state = reinterpret_cast<const CircularBufferSinkState*>(pSinkInfo->GetState());

    NN_SDK_ASSERT( param->_size % (param->_inputCount * param->_sampleCount * sizeof(int16_t)) == 0 );

    auto c = CreateCommand<CircularBufferSinkCommand>(nodeId);

    c->inputCount = param->_inputCount;
    for(int i = 0; i < param->_inputCount; ++i)
    {
        c->input[i] = param->_input[i] + static_cast<int16_t>(mixBufferOffset);
    }

    // destination buffer
    NN_SDK_REQUIRES(state->_circularBuffer.GetReference() != NULL);
    c->circularBuffer = state->_circularBuffer.GetReference();
    c->circularBufferSize = param->_size;
    c->currentOffset = state->_commandPos;
    EstimateProcessTime(c);

    return c;
}

DownMix6chTo2chCommand* CommandBuffer::GenerateDownMixCommand(int mixBufferOffset, const int8_t* input, const int8_t* output, const int32_t* downMixParam, NodeId nodeId) NN_NOEXCEPT
{
    auto c = CreateCommand<DownMix6chTo2chCommand>(nodeId);

    c->inputBufferIndices[nn::audio::ChannelMapping_FrontLeft] = input[nn::audio::ChannelMapping_FrontLeft] + static_cast<int16_t>(mixBufferOffset);
    c->inputBufferIndices[nn::audio::ChannelMapping_FrontRight] = input[nn::audio::ChannelMapping_FrontRight] + static_cast<int16_t>(mixBufferOffset);
    c->inputBufferIndices[nn::audio::ChannelMapping_RearLeft] = input[nn::audio::ChannelMapping_RearLeft] + static_cast<int16_t>(mixBufferOffset);
    c->inputBufferIndices[nn::audio::ChannelMapping_RearRight] = input[nn::audio::ChannelMapping_RearRight] + static_cast<int16_t>(mixBufferOffset);
    c->inputBufferIndices[nn::audio::ChannelMapping_FrontCenter] = input[nn::audio::ChannelMapping_FrontCenter] + static_cast<int16_t>(mixBufferOffset);
    c->inputBufferIndices[nn::audio::ChannelMapping_LowFrequency] = input[nn::audio::ChannelMapping_LowFrequency] + static_cast<int16_t>(mixBufferOffset);

    // use same place for downmix result, RearLR, center, Lfe should be cleared
    c->outputBufferIndices[nn::audio::ChannelMapping_FrontLeft] = output[nn::audio::ChannelMapping_FrontLeft] + static_cast<int16_t>(mixBufferOffset);
    c->outputBufferIndices[nn::audio::ChannelMapping_FrontRight] = output[nn::audio::ChannelMapping_FrontRight] + static_cast<int16_t>(mixBufferOffset);
    c->outputBufferIndices[nn::audio::ChannelMapping_RearLeft] = output[nn::audio::ChannelMapping_RearLeft] + static_cast<int16_t>(mixBufferOffset);
    c->outputBufferIndices[nn::audio::ChannelMapping_RearRight] = output[nn::audio::ChannelMapping_RearRight] + static_cast<int16_t>(mixBufferOffset);
    c->outputBufferIndices[nn::audio::ChannelMapping_FrontCenter] = output[nn::audio::ChannelMapping_FrontCenter] + static_cast<int16_t>(mixBufferOffset);
    c->outputBufferIndices[nn::audio::ChannelMapping_LowFrequency] = output[nn::audio::ChannelMapping_LowFrequency] + static_cast<int16_t>(mixBufferOffset);

    for (auto i = 0; i < DownMix6chTo2chCommand::CoeffCount; ++i)
    {
        c->DownMixMatrixCoeff[i] = downMixParam[i];
    }

    EstimateProcessTime(c);

    return c;
}

UpsampleCommand* CommandBuffer::GenerateUpsampleCommand(int mixBufferOffset, UpsamplerInfo* pInfo, int inputCount, const int8_t* input, int32_t bufferCount, int32_t sampleCount, int32_t sampleRate, NodeId nodeId) NN_NOEXCEPT
{
    auto c = CreateCommand<UpsampleCommand>(nodeId);
    c->inputBufferIndex = 0;
    NN_SDK_ASSERT(m_ServicePool.Contains(pInfo->pOutBuffers, pInfo->sampleCount * sizeof(int32_t)));
    c->pDestBuffers = m_ServicePool.Translate(pInfo->pOutBuffers, pInfo->sampleCount * sizeof(int32_t));
    NN_SDK_ASSERT(m_ServicePool.Contains(pInfo->input, sizeof(pInfo->input)));
    c->input = m_ServicePool.Translate(pInfo->input, sizeof(pInfo->input));
    c->mixBufferCount = bufferCount;
    c->inputSampleCount = sampleCount;
    c->inputSampleRate = sampleRate;

    pInfo->inputCount = inputCount;
    std::memcpy(pInfo->input, input, sizeof(pInfo->input));
    for(int i = 0; i < inputCount; ++i)
    {
        pInfo->input[i] += static_cast<int8_t>(mixBufferOffset);
    }
    c->pInfo = m_ServicePool.Translate(pInfo, sizeof(*pInfo));
    NN_SDK_ASSERT(m_ServicePool.Contains(pInfo, sizeof(*pInfo)));
    EstimateProcessTime(c);

    return c;
}

PerformanceCommand* CommandBuffer::GeneratePerformanceCommand(const PerformanceEntryAddresses* entry, const PerformanceCommandType type, NodeId nodeId) NN_NOEXCEPT
{
    if (entry == nullptr)
    {
        return nullptr;
    }

    auto c = CreateCommand<PerformanceCommand>(nodeId);
    c->baseAddress = entry->dspFrameBase;
    c->offsetToEntryCount = entry->offsetToCount;
    c->offsetToStartTime = entry->offsetToStartTime;
    c->offsetToProcessingTime = entry->offsetToProcessingTime;
    c->type = type;
    EstimateProcessTime(c);

    return c;
}

}}}  // namespace nn::audio::server
