﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_SdkThreadCommon.h>
#include "audio_AudioRenderSystemManager.h"
#include "../dsp/audio_Dsp.h"
#include "../dsp/audio_DspExceptionNotifier.h"

namespace nn { namespace audio {  namespace server {

namespace {

NN_OS_ALIGNAS_THREAD_STACK uint8_t g_Stack[12 * 1024];

void DspExceptionObserver(void* args) NN_NOEXCEPT
{
    NN_UNUSED(args);
    AudioRenderSystemManager::GetInstance().Dump();
}

} // anonymous namespace

AudioRenderSystemManager::AudioRenderSystemManager() NN_NOEXCEPT
    : m_Mutex(false)
    , m_MutexForInitializeFinalize(false)
    , m_Thread()
    , m_IsActive(false)
{
    m_InstanceList.clear();
}

AudioRenderSystemManager::~AudioRenderSystemManager() NN_NOEXCEPT
{

}

void AudioRenderSystemManager::Initialize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_MutexForInitializeFinalize);

    if( m_IsActive )
    {
        return;
    }

    dsp::DspExceptionNotifier::AddObserver(DspExceptionObserver);
    m_IsActive = true;
    NN_ABORT_UNLESS_RESULT_SUCCESS(dsp::Start());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_Thread, ThreadFunction, this, g_Stack, sizeof(g_Stack), NN_SYSTEM_THREAD_PRIORITY(audio, AudioRenderSystemManager)));
    nn::os::SetThreadNamePointer(&m_Thread, NN_SYSTEM_THREAD_NAME(audio, AudioRenderSystemManager));
    nn::os::StartThread(&m_Thread);
}

void AudioRenderSystemManager::Finalize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_MutexForInitializeFinalize);

    if( m_IsActive == false )
    {
        return;
    }

    m_IsActive = false;
    nn::os::WaitThread(&m_Thread);
    nn::os::DestroyThread(&m_Thread);
    dsp::Stop();
    dsp::DspExceptionNotifier::RemoveObserver(DspExceptionObserver);
}

AudioRenderSystemManager* AudioRenderSystemManager::sManagerInstance = nullptr;

void AudioRenderSystemManager::CreateInstance(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_UNUSED(bufferSize);

    if (sManagerInstance == nullptr)
    {
        NN_SDK_ASSERT_ALIGNED(buffer, alignof(AudioRenderSystemManager));
        NN_SDK_ASSERT(bufferSize >= sizeof(AudioRenderSystemManager));

        sManagerInstance = new(buffer) AudioRenderSystemManager();
    }
}

void AudioRenderSystemManager::DeleteInstance() NN_NOEXCEPT
{
    if (sManagerInstance)
    {
        sManagerInstance->~AudioRenderSystemManager();
        sManagerInstance = nullptr;
    }
}

AudioRenderSystemManager& AudioRenderSystemManager::GetInstance() NN_NOEXCEPT
{
    return *sManagerInstance;
}

bool AudioRenderSystemManager::Add(AudioRenderSystem& instance) NN_NOEXCEPT
{
    if(m_InstanceList.empty())
    {
        Initialize();
    }

    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        const auto isRemained = (m_InstanceList.size() < common::AudioRendererSessionCountMax);

        if(isRemained)
        {
            m_InstanceList.push_back(instance);
        }

        return isRemained;
    }

}

void AudioRenderSystemManager::Remove(AudioRenderSystem& instance) NN_NOEXCEPT
{
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        m_InstanceList.erase(m_InstanceList.iterator_to(instance));
    }

    if(m_InstanceList.empty())
    {
        Finalize();
    }
}

void AudioRenderSystemManager::ThreadFunction(void* arg) NN_NOEXCEPT
{
    reinterpret_cast<AudioRenderSystemManager*>(arg)->ThreadFunctionImpl();
}

void AudioRenderSystemManager::ThreadFunctionImpl() NN_NOEXCEPT
{
    for(;;)
    {
        dsp::Wait();
        {
            if(!m_IsActive)
            {
                break;
            }
            else
            {
                std::lock_guard<nn::os::Mutex> lock(m_Mutex);

                for (auto& entry : m_InstanceList)
                {
                    entry.SendCommandToDsp();
                }
            }
        }
        dsp::Signal();
    }
}

void AudioRenderSystemManager::Dump() NN_NOEXCEPT
{
    for (auto& entry : m_InstanceList)
    {
        entry.Dump();
    }
}

void AudioRenderSystemManager::Sleep() NN_NOEXCEPT
{
    dsp::Sleep();
}

void AudioRenderSystemManager::Wake() NN_NOEXCEPT
{
    dsp::Wake();
}

}}}  // namespace nn::audio::server
