﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdint>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_TickTypes.h>
#include <nn/dd/dd_DeviceAddressSpace.h>
#include <nn/applet/applet_Types.h>
#include <nn/audio/audio_AudioRendererTypes.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/audio/detail/audio_AudioRendererTypesInternal.h>
#include "audio_VoiceState.h"
#include "audio_PerformanceMetricsManager.h"
#include "audio_ServiceBehaviorInfo.h"
#include "audio_ServiceEffectInfo.h"
#include "audio_ServiceVoiceInfo.h"
#include "audio_ServiceMemoryPoolInfo.h"
#include "audio_ServiceMixInfo.h"
#include "audio_ServiceSinkInfo.h"
#include "audio_ServiceSplitterInfo.h"
#include "audio_UpsamplerManager.h"
#include "audio_CommandBuffer.h"
#include "audio_PerformanceMeasuredCommandBuffer.h"
#include "audio_CommandProcessingTimeEstimator.h"

namespace nn { namespace audio { namespace server {

class AudioRenderSystem : public nn::util::IntrusiveListBaseNode<AudioRenderSystem>
{
public:
    enum RunState : uint8_t
    {
        RunState_Running,
        RunState_Suspended,
        RunState_Stopped,
    };

private:
    bool m_IsActive;
    RunState m_RunState;
    int32_t m_SampleRate;
    int32_t m_SampleCount;
    int32_t m_MixBufferCount;
    int32_t* m_MixBuffer;
    int32_t* m_DepopBuffer;

    // MemoryPool fields
    int32_t m_MemoryPoolCount;
    server::MemoryPoolInfo* m_pMemoryPoolInfos;
    server::MemoryPoolInfo m_ServicePoolInfo;

    int8_t* m_CommandBuffer;
    size_t m_CommandBufferSize;
    size_t m_CurrentCommandBufferSize;

    UpsamplerManager* m_UpsamplerManager; // here because config is public and cannot be changed yet
    UpsamplerInfo* m_UpsamplerInfos;
    int32_t m_UpsamplerCount;

    VoiceContext m_VoiceContext;
    MixContext m_MixContext;
    EffectContext m_EffectContext;
    SinkContext m_SinkContext;
    SplitterContext m_SplitterContext;
    ICommandProcessingTimeEstimator* m_pCommandProcessingTimeEstimator;

    uint32_t m_SessionId;
    int32_t m_VoiceChannelCountMax;

    nn::os::SystemEvent& m_SystemEvent;
    nn::os::Event m_ReleasedFromDspEvent;
    nn::os::Mutex m_Mutex;
    nn::dd::ProcessHandle m_ClientProcessHandle;
    nn::applet::AppletResourceUserId m_AppletResourceUserId;

    PerformanceManager m_PerformanceManager;
    int8_t* m_PerformanceBuffer;

    void* m_WorkBuffer;
    size_t m_WorkBufferSize;

    void* m_SharedBuffer;
    size_t m_SharedBufferSize;

    int m_RenderingTimeLimitPercent;
    bool m_IsVoiceDropEnabled;
    bool m_IsNewForDspRenderer;

    AudioRendererExecutionMode m_ExecutionMode;
    AudioRendererRenderingDevice m_RenderingDevice;

    CommandProcessingTimeEstimatorVersion1 m_CommandProcessingTimeEstimatorVersion1;
    CommandProcessingTimeEstimatorVersion2 m_CommandProcessingTimeEstimatorVersion2;

    server::BehaviorInfo m_BehaviorInfo;

    struct Statistics
    {
        nn::os::Tick generateCommandsTotalTick;
        nn::os::Tick updateTotalTick;
        uint64_t generateCommandsCount;
        uint64_t updateCount;
    };

    Statistics m_Statistics;
    int64_t m_ElapsedFrameCount;
    bool m_IsRenderingTimeLimitExceededOnPreviousFrame;
    int m_VoiceDropCountOnPreviousFrame;
    int64_t m_RenderingStartTimeOnPreviousFrame;

    void* m_AudioCodecBuffer;
    size_t m_AudioCodecBufferSize;

    void CalculateDistancesFromFinalMix() NN_NOEXCEPT;
    uint32_t GetDspProcessingTimeBudget() const NN_NOEXCEPT;

    static size_t CalculateCommandBufferSize(const nn::audio::detail::AudioRendererParameterInternal& parameter, const BehaviorInfo& behaviorInfo) NN_NOEXCEPT;

public:
    explicit AudioRenderSystem(nn::os::SystemEvent& systemEvent) NN_NOEXCEPT;
    Result Initialize(const detail::AudioRendererParameterInternal& parameter, nn::dd::ProcessHandle clientProcessHandle, void* buffer, size_t size, const nn::applet::AppletResourceUserId& appletResourceUserId, int sessionId) NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

    Result Start() NN_NOEXCEPT;
    void Stop() NN_NOEXCEPT;

    bool IsActive() const NN_NOEXCEPT
    {
        return m_IsActive;
    }

    Result Update(void* out, size_t outSize, void* performance, size_t performanceSize, const void* in, size_t inSize) NN_NOEXCEPT;
    size_t GenerateCommand(void* pCommandBuffer, size_t size) NN_NOEXCEPT;
    void SendCommandToDsp() NN_NOEXCEPT;

    void* GetCommandBufferWorkBuffer() const NN_NOEXCEPT;
    size_t GetCommandBufferWorkBufferSize() const NN_NOEXCEPT;

    int GetSampleRate() const NN_NOEXCEPT;
    int GetSampleCount() const NN_NOEXCEPT;
    int GetMixBufferCount() const NN_NOEXCEPT;
    int GetVoiceCount() const NN_NOEXCEPT;
    int GetRenderingTimeLimit() const NN_NOEXCEPT;
    void SetRenderingTimeLimit(int limitPercent) NN_NOEXCEPT;
    AudioRendererExecutionMode GetExecutionMode() const NN_NOEXCEPT;
    AudioRendererRenderingDevice GetRenderingDevice() const NN_NOEXCEPT;


    uint32_t GetSessionId() const NN_NOEXCEPT;

    static size_t GetWorkBufferSize(const detail::AudioRendererParameterInternal& parameter) NN_NOEXCEPT;
    static size_t GetRequiredBufferSizeForPerformanceMetricsPerFrame(const BehaviorInfo& behaviorInfo, const detail::AudioRendererParameterInternal& parameter) NN_NOEXCEPT;

    int DropVoices(CommandBuffer& commandBuffer, uint32_t voiceProcessingTime, uint32_t voiceProcessingTimeBudget) NN_NOEXCEPT;

    Result QuerySystemEvent(nn::os::NativeHandle* outHandle) NN_NOEXCEPT;
    void SignalSystemEvent() NN_NOEXCEPT;
    void Dump() NN_NOEXCEPT;
};

typedef nn::util::IntrusiveList<AudioRenderSystem, nn::util::IntrusiveListBaseNodeTraits<AudioRenderSystem>> AudioRenderSystemList;

}}}  // namespace nn::audio::server
