﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <cstring>
#include <algorithm>

#include <nn/nn_Abort.h>
#include <nn/nn_TimeSpan.h>
#include <nn/util/util_BytePtr.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/dd/dd_Types.h>
#include <nn/audio/audio_Result.h>
#include <nn/audio/audio_Result.private.h>
#include <nn/audio/detail/audio_Log.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/audio/audio_AudioRendererApi-os.win32.h>
#endif

#include "../common/audio_BuildDefinition.h"

#if defined(NN_AUDIO_ENABLE_PROFILER) && defined(NN_AUDIO_SYSTEM_PROCESS)
#include <nn/profiler.h>
#endif

#include "../common/audio_AudioRendererResourceControlParameter.h"
#include "../common/audio_Command.h"
#include "../common/audio_CommandDumper.h"
#include "../common/audio_MemoryPoolPolicy.h"
#include "../common/audio_NodeIdManager.h"
#include "../common/audio_PerformanceMetricsCommon.h"
#include "../common/audio_SortLogic.h"
#include "../common/audio_Util.h"
#include "../common/audio_WorkBufferAllocator.h"
#include "../dsp/audio_Cache.h"
#include "../dsp/audio_Dsp.h"
#include "../dsp/audio_DspUtility.h"
#include "audio_AppletVolumeManager.h"
#include "audio_AudioRenderSystem.h"
#include "audio_CommandBuffer.h"
#include "audio_PerformanceMeasuredCommandBuffer.h"
#include "audio_CommandGenerator.h"
#include "audio_InfoUpdater.h"
#include "audio_PerformanceMetricsManager.h"
#include "audio_ServiceMemoryPoolInfo.h"
#include "audio_AudioHardwareConfiguration.h"
#include "audio_ServiceDiagnostics.h"
#if defined(NN_BUILD_CONFIG_OS_WIN)
#include "../dsp/audio_DeviceMixerManager.h"
#endif


namespace
{
int g_VoiceChannelCountMax = 6; // TODO: クライアント側の定義との比較

int GetCurrentChannelCount(uint32_t sessionId)
{
#if defined(NN_AUDIO_ENABLE_APPLET_VOLUME_MANAGER)
    NN_UNUSED(sessionId);
    return nn::audio::AppletVolumeManager::GetDeviceChannelCount();
#elif defined(NN_BUILD_CONFIG_OS_WIN)
    return nn::audio::GetDeviceChannelCount(nn::audio::GetDevice("MainAudioOut", sessionId));
#elif defined(NN_BUILD_CONFIG_OS_HORIZON) && defined(NN_AUDIO_ENABLE_CPU_RENDERER)
    NN_SDK_ASSERT(false, "Unsupported platform.");
    return 2;
#else
#error "unsuppoerted target"
#endif
}

}

namespace nn { namespace audio { namespace server {

AudioRenderSystem::AudioRenderSystem(nn::os::SystemEvent& systemEvent) NN_NOEXCEPT
    : m_IsActive(false)
    , m_RunState(RunState_Stopped)
    , m_ServicePoolInfo(server::MemoryPoolInfo::Location_Service)
    , m_pCommandProcessingTimeEstimator(nullptr)
    , m_SystemEvent(systemEvent)
    , m_ReleasedFromDspEvent(nn::os::EventClearMode_AutoClear)
    , m_Mutex(false)
    , m_ClientProcessHandle(nn::os::InvalidNativeHandle)
    , m_ElapsedFrameCount(0)
    , m_IsRenderingTimeLimitExceededOnPreviousFrame(false)
    , m_VoiceDropCountOnPreviousFrame(0)
    , m_RenderingStartTimeOnPreviousFrame(0)
{
}

size_t AudioRenderSystem::CalculateCommandBufferSize(const nn::audio::detail::AudioRendererParameterInternal& parameter, const BehaviorInfo& behaviorInfo) NN_NOEXCEPT
{
    if (behaviorInfo.IsVariadicCommandBufferSizeSupported())
    {
        return CommandGenerator::CalculateCommandBufferSize(parameter);
    }
    else
    {
        // Command buffer size is fixed before NXAddon 6.0.0.
        const size_t s_OldCommandListSize = 96 * 1024;
        return s_OldCommandListSize;
    }
}

size_t AudioRenderSystem::GetWorkBufferSize(const ::nn::audio::detail::AudioRendererParameterInternal& parameter) NN_NOEXCEPT
{
    size_t size = 0;
    server::BehaviorInfo behavior;
    behavior.SetUserLibRevision(parameter._magic);

    // ----------------------------------------------------------------
    // ADSP only begin
    // ----------------------------------------------------------------
    // mix buffers
    size += sizeof(int32_t) * parameter.sampleCount * (parameter.mixBufferCount + g_VoiceChannelCountMax);

    size += sizeof(int32_t) * 240 * (parameter.mixBufferCount + g_VoiceChannelCountMax) * (parameter.subMixCount + parameter.sinkCount);

    // depop array for mix buffers
    size = nn::util::align_up(size, nn::audio::BufferAlignSize);
    size += nn::util::align_up(sizeof(int32_t) * parameter.mixBufferCount, nn::audio::BufferAlignSize);

    size += nn::util::align_up(parameter.bufferSizeForAudioCodec, nn::audio::BufferAlignSize);
    // ----------------------------------------------------------------
    // ADSP only end
    // ----------------------------------------------------------------

    // ----------------------------------------------------------------
    // Audio process only begin
    // ----------------------------------------------------------------
    // voices
    size += nn::util::align_up(sizeof(server::VoiceInfo) * parameter.voiceCount, ::nn::audio::common::InfoTypeFieldAlignSize);
    size += nn::util::align_up(sizeof(server::VoiceInfo*) * parameter.voiceCount, ::nn::audio::common::InfoTypeFieldAlignSize);
    size += nn::util::align_up(sizeof(server::VoiceChannelResource) * parameter.voiceCount, ::nn::audio::common::InfoTypeFieldAlignSize);
    size += nn::util::align_up(sizeof(VoiceState) * parameter.voiceCount, ::nn::audio::common::InfoTypeFieldAlignSize);
    // Mix (データそのもの + ソート用)
    size += nn::util::align_up(sizeof(server::MixInfo) * (parameter.subMixCount + 1), ::nn::audio::common::InfoTypeFieldAlignSize); // SubMixCount + FinalMixCount(1)
    size += nn::util::align_up(sizeof(server::MixInfo*) * (parameter.subMixCount + 1), ::nn::audio::common::InfoTypeFieldAlignSize); // SubMixCount + FinalMixCount(1)
    size += nn::util::align_up(sizeof(int32_t) * AudioRendererParameter::EffectCountMax * parameter.subMixCount, ::nn::audio::common::InfoTypeFieldAlignSize); // for EffectProcessingOrder array
    if (behavior.IsSplitterSupported())
    {
        size += nn::util::align_up(common::NodeStates
            ::GetWorkBufferSize(parameter.subMixCount + 1)
            + common::EdgeMatrix::GetWorkBufferSize(parameter.subMixCount + 1), ::nn::audio::common::InfoTypeFieldAlignSize);
    }
    // Upsampler Manager
    size += nn::util::align_up(sizeof(UpsamplerManager), ::nn::audio::common::InfoTypeFieldAlignSize);
    // MemoryPool
    size += nn::util::align_up(sizeof(server::MemoryPoolInfo) * common::GetMemoryPoolCount(parameter), ::nn::audio::common::InfoTypeFieldAlignSize);
    // SplitterInfo
    size += server::SplitterContext::CalcWorkBufferSize(behavior, parameter);

    // ----------------------------------------------------------------
    // Audio process only end
    // ----------------------------------------------------------------

    size = nn::util::align_up(size, nn::audio::BufferAlignSize);

    // ----------------------------------------------------------------
    // Shared begin
    // ----------------------------------------------------------------
    // upsampler
    size += nn::util::align_up(sizeof(UpsamplerInfo) * (parameter.subMixCount + parameter.sinkCount), nn::audio::BufferAlignSize);

    // effect
    size += nn::util::align_up(sizeof(server::EffectInfoBase) * parameter.effectCount, ::nn::audio::common::InfoTypeFieldAlignSize);

    // sink
    size += nn::util::align_up(sizeof(server::SinkInfoBase) * parameter.sinkCount, ::nn::audio::common::InfoTypeFieldAlignSize);

    // voices
    size += nn::util::align_up(sizeof(VoiceState) * parameter.voiceCount + BufferAlignSize - 1, ::nn::audio::common::InfoTypeFieldAlignSize); // for align up

    // performanceMetrics
    if (parameter.performanceFrameCount > 0)
    {
        // Memory for PerformanceManager is allocated from audio process but an old FW allocated it from workbuffer passed from an application.
        // To save compatibility, AudioRenderSystem resevers 128 bytes here. (sizeof(PerformanceManager) in an old FW.)
        const auto reservedSize = nn::util::align_up(128, nn::audio::BufferAlignSize);
        size += nn::util::align_up(GetRequiredBufferSizeForPerformanceMetricsPerFrame(behavior, parameter) * (parameter.performanceFrameCount + 1), nn::audio::BufferAlignSize) // "+1" is for dsp use buffer.
              + nn::util::align_up(sizeof(PerformanceStatistics), nn::audio::BufferAlignSize)
              + reservedSize;
    }

    // ----------------------------------------------------------------
    // CommandBuffer (Audio process write only, ADSP read only) begin
    // ----------------------------------------------------------------
    // tmp - used for CommandList (CalculateCommandBufferSize()+ buffer for align up address and size)
    size += CalculateCommandBufferSize(parameter, behavior) + (nn::audio::BufferAlignSize - 1) * 2;
    // ----------------------------------------------------------------
    // CommandBuffer (Audio process write only, ADSP read only) end
    // ----------------------------------------------------------------

    size = nn::util::align_up(size, nn::os::MemoryPageSize);

    // ----------------------------------------------------------------
    // Shared begin
    // ----------------------------------------------------------------

    return size;
}

size_t AudioRenderSystem::GetRequiredBufferSizeForPerformanceMetricsPerFrame(const BehaviorInfo& behaviorInfo, const ::nn::audio::detail::AudioRendererParameterInternal& parameter) NN_NOEXCEPT
{
    return PerformanceManager::GetRequiredBufferSizeForPerformanceMetricsPerFrame(behaviorInfo, parameter);
}

Result AudioRenderSystem::Initialize(const ::nn::audio::detail::AudioRendererParameterInternal& parameter, nn::dd::ProcessHandle clientProcessHandle, void* buffer, size_t size, const nn::applet::AppletResourceUserId& appletResourceUserId, int sessionId) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(common::CheckValidRevision(parameter._magic), ResultOperationFailed());
    NN_RESULT_THROW_UNLESS(GetWorkBufferSize(parameter) <= size, ResultInsufficientBuffer());
#if defined(NN_AUDIO_SYSTEM_PROCESS) && !defined(NN_BUILD_CONFIG_OS_WIN)
    NN_RESULT_THROW_UNLESS(clientProcessHandle != nn::os::InvalidNativeHandle, ResultInvalidClientProcessHandle());
#endif
    m_BehaviorInfo.SetUserLibRevision(parameter._magic);
    m_SampleRate = parameter.sampleRate;
    m_SampleCount = parameter.sampleCount;
    m_MixBufferCount = parameter.mixBufferCount;
    m_VoiceChannelCountMax = g_VoiceChannelCountMax;
    m_UpsamplerCount = parameter.subMixCount + parameter.sinkCount;
    m_AppletResourceUserId = appletResourceUserId;
    m_MemoryPoolCount = common::GetMemoryPoolCount(parameter);
    m_RenderingDevice = static_cast<AudioRendererRenderingDevice>(parameter.renderingDevice);
    m_ExecutionMode = static_cast<AudioRendererExecutionMode>(parameter.executionMode);

    m_SessionId = sessionId;

    common::WorkBufferAllocator allocator(buffer, size);
    memset(buffer, 0, size);

    server::PoolMapper(clientProcessHandle).InitializeSystemPool(&m_ServicePoolInfo, buffer, size);
    m_WorkBuffer = buffer;
    m_WorkBufferSize = size;

    // ----------------------------------------------------------------
    // ADSP only begin
    // ----------------------------------------------------------------
    m_MixBuffer = reinterpret_cast<int32_t*>(allocator.Allocate((sizeof(int32_t) * m_SampleCount * (m_MixBufferCount + m_VoiceChannelCountMax))));
    NN_RESULT_THROW_UNLESS(m_MixBuffer != nullptr, ResultInsufficientBuffer());
    auto pAdspOnlyBuffer = m_MixBuffer;

    int32_t UpsamplerBuffersSize = sizeof(int32_t) * 240 * (m_MixBufferCount + m_VoiceChannelCountMax) * m_UpsamplerCount;
    int8_t* pUpsamplerBuffers = reinterpret_cast<int8_t*>(allocator.Allocate(UpsamplerBuffersSize));
    NN_RESULT_THROW_UNLESS(pUpsamplerBuffers != nullptr, ResultInsufficientBuffer());

    // depop array for mix buffers
    m_DepopBuffer = reinterpret_cast<int32_t*>(allocator.Allocate(nn::util::align_up(sizeof(int32_t) * m_MixBufferCount, nn::audio::BufferAlignSize), nn::audio::BufferAlignSize));
    NN_RESULT_THROW_UNLESS(m_DepopBuffer != nullptr, ResultInsufficientBuffer());

    if (m_BehaviorInfo.IsAudioCodecSupported())
    {
        m_AudioCodecBuffer = allocator.Allocate(parameter.bufferSizeForAudioCodec, nn::audio::BufferAlignSize);
        m_AudioCodecBufferSize = nn::util::align_up(parameter.bufferSizeForAudioCodec, nn::audio::BufferAlignSize);
    }
    else
    {
        m_AudioCodecBuffer = nullptr;
        m_AudioCodecBufferSize = 0;
    }

    // TODO: a workaround for SIGLO-64358
    //       Invalidate ADSP cache here because DepopBuffer depends to be cleared from CPU side.
    //       (Invalidate m_MixBuffer and pUpsamplerBuffers for just in case.)
    const auto AdspOnlyBufferSize = allocator.GetUsedSize();
    nn::audio::dsp::InvalidateDspCache(m_ServicePoolInfo.Translate(pAdspOnlyBuffer, AdspOnlyBufferSize), AdspOnlyBufferSize);

    // ----------------------------------------------------------------
    // ADSP only end
    // ----------------------------------------------------------------

    NN_SDK_ASSERT(allocator.GetUsedSize() % nn::audio::BufferAlignSize == 0);

    // ----------------------------------------------------------------
    // Audio process only begin
    // ----------------------------------------------------------------
    auto voices = common::PrepareArray<server::VoiceInfo>(allocator.Allocate(sizeof(server::VoiceInfo) * parameter.voiceCount), parameter.voiceCount);
    NN_RESULT_THROW_UNLESS(voices != nullptr, ResultInsufficientBuffer());
    auto voicesSortWork = reinterpret_cast<server::VoiceInfo**>(allocator.Allocate(sizeof(server::VoiceInfo*) * parameter.voiceCount));
    NN_RESULT_THROW_UNLESS(voicesSortWork != nullptr, ResultInsufficientBuffer());
    memset(voicesSortWork, 0, sizeof(server::VoiceInfo*) * parameter.voiceCount);

    auto voiceChannelResources = common::PrepareArrayWithIndex<server::VoiceChannelResource>(allocator.Allocate(sizeof(server::VoiceChannelResource) * parameter.voiceCount), parameter.voiceCount);
    NN_RESULT_THROW_UNLESS(voiceChannelResources != nullptr, ResultInsufficientBuffer());

    // leave VoiceState as POD type, since to use memset in GenerateVoiceCommand()
    auto voiceStates = reinterpret_cast<VoiceState*>(allocator.Allocate(sizeof(VoiceState) * parameter.voiceCount));
    NN_RESULT_THROW_UNLESS(voiceStates != nullptr, ResultInsufficientBuffer());
    memset(voiceStates, 0, sizeof(VoiceState) * parameter.voiceCount);

    auto mixInfosCount = parameter.subMixCount + 1; // SubMixCount + FinalMixCount + 1
    auto mixInfos = reinterpret_cast<server::MixInfo*>(allocator.Allocate(sizeof(server::MixInfo) * mixInfosCount));
    NN_RESULT_THROW_UNLESS(mixInfos != nullptr, ResultInsufficientBuffer());
    size_t effectOrderBuffer = 0;
    int32_t* effectOederBufferSize = nullptr;
    if (parameter.effectCount > 0)
    {
        effectOrderBuffer = mixInfosCount * parameter.effectCount * sizeof(int32_t);
        effectOederBufferSize = reinterpret_cast<int32_t*>(allocator.Allocate(effectOrderBuffer));
        NN_RESULT_THROW_UNLESS(effectOederBufferSize != nullptr, ResultInsufficientBuffer());
        auto pMixInfo = nn::util::BytePtr(mixInfos);
        auto pEffectOrder = nn::util::BytePtr(effectOederBufferSize);
        for (auto i = 0; i < mixInfosCount; ++i)
        {
            new(pMixInfo.Get()) server::MixInfo(reinterpret_cast<int32_t*>(pEffectOrder.Get()), parameter.effectCount, m_BehaviorInfo);
            pMixInfo += sizeof(server::MixInfo);
            pEffectOrder += parameter.effectCount * sizeof(int32_t);
        }
    }
    else
    {
        auto pMixInfo = nn::util::BytePtr(mixInfos);
        for (auto i = 0; i < mixInfosCount; ++i)
        {
            new(pMixInfo.Get()) server::MixInfo(nullptr, 0, m_BehaviorInfo);
            pMixInfo += sizeof(server::MixInfo);
        }
    }
    mixInfos[MixInfo::FinalMixId].mixId = MixInfo::FinalMixId;
    auto mixInfosSortWork = reinterpret_cast<server::MixInfo**>(allocator.Allocate(sizeof(server::MixInfo*) * mixInfosCount));
    NN_RESULT_THROW_UNLESS(mixInfosSortWork != nullptr, ResultInsufficientBuffer());
    memset(mixInfosSortWork, 0, sizeof(server::MixInfo*) * mixInfosCount);
    if (m_BehaviorInfo.IsSplitterSupported())
    {
        auto stateBufferSize = common::NodeStates::GetWorkBufferSize(mixInfosCount);
        auto stateBuffer = allocator.Allocate(stateBufferSize, 1);
        auto edgeBufferSize = common::EdgeMatrix::GetWorkBufferSize(mixInfosCount);
        auto edgeBuffer = allocator.Allocate(edgeBufferSize, 1);
        NN_RESULT_THROW_UNLESS(stateBuffer != nullptr, ResultInsufficientBuffer());
        NN_RESULT_THROW_UNLESS(edgeBuffer != nullptr, ResultInsufficientBuffer());
        m_MixContext.Initialize(
            mixInfosSortWork, mixInfos, mixInfosCount,
            effectOederBufferSize, effectOrderBuffer,
            stateBuffer, stateBufferSize,
            edgeBuffer, edgeBufferSize);
    }
    else
    {
        m_MixContext.Initialize(mixInfosSortWork, mixInfos, mixInfosCount, effectOederBufferSize, effectOrderBuffer, nullptr, 0, nullptr, 0);
    }

    m_UpsamplerManager = reinterpret_cast<UpsamplerManager*>(allocator.Allocate(sizeof(UpsamplerManager)));
    NN_RESULT_THROW_UNLESS(m_UpsamplerManager != nullptr, ResultInsufficientBuffer());

    m_pMemoryPoolInfos = common::PrepareArray<server::MemoryPoolInfo>(allocator.Allocate(sizeof(server::MemoryPoolInfo) * m_MemoryPoolCount), m_MemoryPoolCount);
    NN_RESULT_THROW_UNLESS(m_MemoryPoolCount == 0 || m_pMemoryPoolInfos != nullptr, ResultInsufficientBuffer());

    // SplitterInfo
    NN_RESULT_THROW_UNLESS(
        m_SplitterContext.Initialize(m_BehaviorInfo, parameter, allocator),
        ResultInsufficientBuffer());

    m_ClientProcessHandle = clientProcessHandle;

    // ----------------------------------------------------------------
    // Audio process only end
    // ----------------------------------------------------------------

    // This is for alignment of m_SharedBuffer which is initialized below
    allocator.Allocate(nn::audio::BufferAlignSize - allocator.GetUsedSize() % nn::audio::BufferAlignSize, 1);
    NN_SDK_ASSERT(allocator.GetUsedSize() % nn::audio::BufferAlignSize == 0);

    // ----------------------------------------------------------------
    // Shared begin
    // ----------------------------------------------------------------
    m_SharedBuffer = reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(m_WorkBuffer) + allocator.GetUsedSize());
    m_SharedBufferSize = allocator.GetFreeSize();

    m_UpsamplerInfos = common::PrepareArray<UpsamplerInfo>(allocator.Allocate(sizeof(UpsamplerInfo) * m_UpsamplerCount, nn::audio::BufferAlignSize), m_UpsamplerCount);
    new(m_UpsamplerManager)UpsamplerManager(m_UpsamplerCount, m_UpsamplerInfos, pUpsamplerBuffers);
    NN_RESULT_THROW_UNLESS(m_UpsamplerInfos != nullptr, ResultInsufficientBuffer());
    NN_SDK_ASSERT(reinterpret_cast<uintptr_t>(m_UpsamplerInfos) % nn::audio::BufferAlignSize == 0);

    auto effectInfos = common::PrepareArray<server::EffectInfoBase>(allocator.Allocate(sizeof(server::EffectInfoBase) * parameter.effectCount, BufferAlignSize), parameter.effectCount);
    NN_RESULT_THROW_UNLESS(parameter.effectCount == 0 || effectInfos != nullptr, ResultInsufficientBuffer());
    m_EffectContext.Initialize(effectInfos, parameter.effectCount);

    auto sinkInfos = common::PrepareArray<server::SinkInfoBase>(allocator.Allocate(sizeof(server::SinkInfoBase) * parameter.sinkCount), parameter.sinkCount);
    NN_RESULT_THROW_UNLESS(sinkInfos != nullptr, ResultInsufficientBuffer());
    m_SinkContext.Initialize(sinkInfos, parameter.sinkCount);

    // leave VoiceState as POD type, since to use memset in GenerateVoiceCommand()
    auto voiceStatesDspShared = reinterpret_cast<VoiceState*>(allocator.Allocate(sizeof(VoiceState) * parameter.voiceCount, BufferAlignSize));
    NN_RESULT_THROW_UNLESS(voiceStatesDspShared != nullptr, ResultInsufficientBuffer());
    memset(voiceStatesDspShared, 0, sizeof(VoiceState) * parameter.voiceCount);
    m_VoiceContext.Initialize(voicesSortWork, voices, voiceChannelResources, voiceStates, voiceStatesDspShared, parameter.voiceCount);

    size_t performanceMetricsSize = 0;
    if (parameter.performanceFrameCount > 0)
    {
        performanceMetricsSize = GetRequiredBufferSizeForPerformanceMetricsPerFrame(m_BehaviorInfo, parameter) * (parameter.performanceFrameCount + 1) // "+1" is for dsp use buffer.
                                 + sizeof(PerformanceStatistics);
        m_PerformanceBuffer = reinterpret_cast<int8_t*>(allocator.Allocate(performanceMetricsSize, nn::audio::BufferAlignSize));
        NN_RESULT_THROW_UNLESS(m_PerformanceBuffer != nullptr, ResultInsufficientBuffer());
        memset(m_PerformanceBuffer, 0, performanceMetricsSize);
        m_PerformanceManager.Initialize(m_PerformanceBuffer, performanceMetricsSize, parameter,m_BehaviorInfo, m_ServicePoolInfo);
    }
    else
    {
        m_PerformanceBuffer = nullptr;
    }

    memset(&m_Statistics, 0, sizeof(m_Statistics));

    m_RenderingTimeLimitPercent = 100;

    // VoiceDrop function can run on only TegraX1's ADSP for now.
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
    m_IsVoiceDropEnabled = parameter.isVoiceDropEnabled && m_ExecutionMode == AudioRendererExecutionMode_AutoExecution;
#else
    m_IsVoiceDropEnabled = false;
#endif

    // ----------------------------------------------------------------
    // CommandBuffer (Audio process write only, ADSP read only) begin
    // ----------------------------------------------------------------
    m_CommandBufferSize = nn::util::align_down(allocator.GetFreeSize(), nn::audio::BufferAlignSize);
    m_CommandBuffer = reinterpret_cast<int8_t*>(allocator.Allocate(m_CommandBufferSize, nn::audio::BufferAlignSize));  // TODO
    NN_SDK_ASSERT(allocator.GetFreeSize() == 0);
    NN_RESULT_THROW_UNLESS(m_CommandBuffer != nullptr, ResultInsufficientBuffer());
    m_CurrentCommandBufferSize = 0;
    // ----------------------------------------------------------------
    // CommandBuffer (Audio process write only, ADSP read only) end
    // ----------------------------------------------------------------

    // ----------------------------------------------------------------
    // Shared end
    // ----------------------------------------------------------------

    m_IsNewForDspRenderer = true;

    nn::audio::dsp::FlushDataCache(buffer, size);

    m_ClientProcessHandle = clientProcessHandle;
    m_ElapsedFrameCount = 0;
    m_IsRenderingTimeLimitExceededOnPreviousFrame = false;
    m_VoiceDropCountOnPreviousFrame = 0;
    m_RenderingStartTimeOnPreviousFrame = 0;

    // Setup CommandProcessingTimeEstimator
    if (m_BehaviorInfo.IsCommandProcessingTimeEstimatorVersion2Supported())
    {
        m_CommandProcessingTimeEstimatorVersion2.Initialize(m_SampleCount, m_MixBufferCount);
        m_pCommandProcessingTimeEstimator = &m_CommandProcessingTimeEstimatorVersion2;
    }
    else
    {
        m_CommandProcessingTimeEstimatorVersion1.Initialize(m_SampleCount, m_MixBufferCount);
        m_pCommandProcessingTimeEstimator = &m_CommandProcessingTimeEstimatorVersion1;
    }

    NN_AUDIO_SERVICE_LOG("[AudioRenderSystem] Initialize Done. DspProcessingTimeBudget(%u) UserLibRevision(%u)\n",
            GetDspProcessingTimeBudget(), common::GetRevisionNumber(m_BehaviorInfo.GetUserLibRevision()));

    NN_RESULT_SUCCESS;
} // NOLINT(readability/fn_size) // TODO : Cleanup at SIGLO-34459

void AudioRenderSystem::Finalize() NN_NOEXCEPT
{
    if(IsActive())
    {
        Stop();
    }
    m_AppletResourceUserId = nn::applet::AppletResourceUserId::GetInvalidId();
    server::PoolMapper(m_ClientProcessHandle).Unmap(&m_ServicePoolInfo);

    if(nn::os::InvalidNativeHandle != m_ClientProcessHandle)
    {
        server::PoolMapper::ClearUseState(m_pMemoryPoolInfos, m_MemoryPoolCount);
        for (auto i = 0; i < m_MemoryPoolCount; ++i)
        {
            if (m_pMemoryPoolInfos[i].IsMapped())
            {
                server::PoolMapper(m_ClientProcessHandle).Unmap(&m_pMemoryPoolInfos[i]);
            }
        }
        dsp::ProcessCleanup(m_ClientProcessHandle); // TODO: can be removed after MemoryPool check enabled.

#if defined(NN_AUDIO_SYSTEM_PROCESS) && !defined(NN_BUILD_CONFIG_OS_WIN)
        nn::dd::CloseProcessHandle(m_ClientProcessHandle);
#endif
        m_ClientProcessHandle = nn::os::InvalidNativeHandle;
    }
}

int AudioRenderSystem::GetRenderingTimeLimit() const NN_NOEXCEPT
{
    return m_RenderingTimeLimitPercent;
}

void AudioRenderSystem::SetRenderingTimeLimit(int limitPercent) NN_NOEXCEPT
{
    NN_SDK_ASSERT_MINMAX(limitPercent, 0, 100);
    m_RenderingTimeLimitPercent = limitPercent;
}

Result AudioRenderSystem::Start() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    m_IsActive = true;
    m_RunState = RunState_Running;
    m_ElapsedFrameCount = 0;
#if defined(NN_AUDIO_ENABLE_APPLET_VOLUME_MANAGER)
    AppletVolumeManager::RegisterAppletResourceUserId(m_AppletResourceUserId);
    AppletVolumeManager::OpenSession(AppletVolumeManager::SessionType_AudioRenderer, m_SessionId, m_AppletResourceUserId, true);
#endif // defined(NN_AUDIO_ENABLE_APPLET_VOLUME_MANAGER)

    NN_RESULT_SUCCESS;
}

void AudioRenderSystem::Stop() NN_NOEXCEPT
{
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

#if defined(NN_AUDIO_ENABLE_APPLET_VOLUME_MANAGER)
        AppletVolumeManager::CloseSession(AppletVolumeManager::SessionType_AudioRenderer, m_SessionId);
        AppletVolumeManager::UnregisterAppletResourceUserId(m_AppletResourceUserId);
#endif // defined(NN_AUDIO_ENABLE_APPLET_VOLUME_MANAGER)
        m_IsActive = false;
        m_RunState = RunState_Stopped;
    }

#if defined(NN_BUILD_CONFIG_OS_WIN)
    if (nn::audio::GetFastRenderingMode())
    {
        nn::audio::SetFastRenderingMode(false, nullptr);
    }
#endif

    if(m_ExecutionMode == AudioRendererExecutionMode_AutoExecution)
    {
        m_ReleasedFromDspEvent.Wait();
    }
}

#define NN_AUDIO_RESULT_HANLDE(r) \
{                           \
    auto tmpResult = (r);   \
    if (result.IsSuccess()) \
    {                       \
        result = tmpResult; \
    }                       \
}

Result AudioRenderSystem::Update(void* out, size_t outSize, void* performanceMetricsBuffer, size_t performanceMetricsBufferSize, const void* in, size_t inSize) NN_NOEXCEPT
{
#if defined(NN_AUDIO_ENABLE_PROFILER) && defined(NN_AUDIO_SYSTEM_PROCESS)
    nn::profiler::ScopedCodeBlock block("Update");
    nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_User1);
#endif

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    const auto beginTick = nn::os::GetSystemTick();

    NN_SDK_REQUIRES_NOT_NULL(out);
    NN_SDK_REQUIRES_NOT_NULL(in);
    memset(out, 0, outSize);

    uintptr_t inParameter = reinterpret_cast<uintptr_t>(in);
    uintptr_t outStatus = reinterpret_cast<uintptr_t>(out);

    InfoUpdater infoUpdater(inParameter, inSize, outStatus, outSize, m_ClientProcessHandle, m_BehaviorInfo);

    NN_RESULT_DO(infoUpdater.UpdateBehaviorInfo(&m_BehaviorInfo));
    NN_RESULT_DO(infoUpdater.UpdateMemoryPools(m_pMemoryPoolInfos, m_MemoryPoolCount));
    NN_RESULT_DO(infoUpdater.UpdateVoiceChannelResources(m_VoiceContext));
    NN_RESULT_DO(infoUpdater.UpdateVoices(m_VoiceContext, m_pMemoryPoolInfos, m_MemoryPoolCount, m_ServicePoolInfo.Translate(m_AudioCodecBuffer, m_AudioCodecBufferSize)));
    NN_RESULT_DO(infoUpdater.UpdateEffects(m_EffectContext, m_IsActive, m_pMemoryPoolInfos, m_MemoryPoolCount));
    if (m_BehaviorInfo.IsSplitterSupported())
    {
        NN_RESULT_DO(infoUpdater.UpdateSplitterInfo(m_SplitterContext));
    }
    NN_RESULT_DO(infoUpdater.UpdateMixes(m_MixContext, m_MixBufferCount, m_EffectContext, m_SplitterContext));
    NN_RESULT_DO(infoUpdater.UpdateSinks(m_SinkContext, m_pMemoryPoolInfos, m_MemoryPoolCount));

    auto performanceManager = m_PerformanceManager.IsInitialized() ? &m_PerformanceManager : nullptr;
    NN_RESULT_DO(infoUpdater.UpdatePerformanceBuffer(performanceMetricsBuffer, performanceMetricsBufferSize, performanceManager));
    NN_RESULT_DO(infoUpdater.UpdateErrorInfo(&m_BehaviorInfo));
    if(m_BehaviorInfo.IsElapsedFrameCountSupported())
    {
        NN_RESULT_DO(infoUpdater.UpdateRendererInfo(m_ElapsedFrameCount));
    }
    NN_RESULT_DO(infoUpdater.CheckConsumedSize());

    m_SystemEvent.Clear();

    m_Statistics.updateTotalTick += nn::os::GetSystemTick() - beginTick;
    ++m_Statistics.updateCount;
    NN_RESULT_SUCCESS;
}

int AudioRenderSystem::DropVoices(CommandBuffer& commandBuffer, uint32_t voiceProcessingTime, uint32_t voiceProcessingTimeBudget) NN_NOEXCEPT
{
    int voiceDropCount = 0;
    auto pCommand = reinterpret_cast<uint8_t*>(commandBuffer.GetBuffer()) + sizeof(CommandListHeader);
    int currentCommandIndex = 0;
    int commandCount = commandBuffer.GetCount();

    // 最初のボイスコマンドを検索
    for(; currentCommandIndex < commandCount; ++currentCommandIndex)
    {
        auto pCommandHeader = reinterpret_cast<CommandHeader*>(pCommand);
        NN_SDK_ASSERT(pCommandHeader->magic == CommandHeaderMagicNumber);

        if(pCommandHeader->id == CommandId_PcmInt16DataSource || pCommandHeader->id == CommandId_AdpcmDataSource || pCommandHeader->id == CommandId_Performance)
        {
            break;
        }

        pCommand += pCommandHeader->size;
    }

    for(;currentCommandIndex < commandCount; ++currentCommandIndex)
    {
        auto pCommandHeader = reinterpret_cast<CommandHeader*>(pCommand);
        NN_SDK_ASSERT(pCommandHeader->magic == CommandHeaderMagicNumber);

        const auto nodeId = pCommandHeader->nodeId;
        const auto nodeType = common::NodeIdManager::GetType(nodeId);
        const int32_t voiceId = common::NodeIdManager::GetBase(nodeId);

        // ボイスドロップ処理を終了
        if(voiceProcessingTime <= voiceProcessingTimeBudget ||
           nodeType != common::NodeIdManager::NodeIdType::Voice ||
           m_VoiceContext.GetInfo(voiceId)._inParameter._priority == nn::audio::VoiceType::PriorityHighest)
        {
            break;
        }

        ++voiceDropCount;

        NN_SDK_ASSERT_RANGE(voiceId, 0, m_VoiceContext.GetCount());

        NN_SDK_ASSERT_EQUAL(m_VoiceContext.GetInfo(voiceId)._inParameter._nodeId, nodeId);
        m_VoiceContext.GetInfo(voiceId).voiceDroppedFlag = true;

        // ボイスドロップ
        for(; currentCommandIndex < commandCount; ++currentCommandIndex)
        {
            pCommandHeader = reinterpret_cast<CommandHeader*>(pCommand);
            NN_SDK_ASSERT(pCommandHeader->magic == CommandHeaderMagicNumber);

            if(pCommandHeader->nodeId != nodeId)
            {
                break;
            }

            if(pCommandHeader->id == CommandId_DepopPrepare)
            {
                pCommandHeader->isEnabled = true;
            }
            else if(pCommandHeader->isEnabled)
            {
                // TODO: PerformanceManager::CopyHistory 側で対処するまでのワークアラウンド
                if(pCommandHeader->id != CommandId_Performance)
                {
                    voiceProcessingTime -= pCommandHeader->estimatedProcessingTime;
                    pCommandHeader->isEnabled = false;
                }
            }

            pCommand += pCommandHeader->size;
        }
    }

    return voiceDropCount;
}


uint32_t AudioRenderSystem::GetDspProcessingTimeBudget() const NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
    const auto AdspFrequency = nn::audio::server::detail::AdspFrequency * 1000; // Convert unit from [KHz] to [Hz].
    const auto AudioFramePerSecond = 200;

    int applicationProcessingTimeLimitPercent;

    // Note:
    // This sequence should be Newer -> Older
    // because when Is80PercentSupported is true, Is75PercentSupported is also true.
    if (m_BehaviorInfo.IsAudioRenererProcessingTimeLimit80PercentSupported())
    {
        applicationProcessingTimeLimitPercent = 80;
    }
    else if (m_BehaviorInfo.IsAudioRenererProcessingTimeLimit75PercentSupported())
    {
        applicationProcessingTimeLimitPercent = 75;
    }
    else if (m_BehaviorInfo.IsAudioRenererProcessingTimeLimit70PercentSupported())
    {
        applicationProcessingTimeLimitPercent = 70;
    }
    else
    {
        applicationProcessingTimeLimitPercent = 70;
    }


    return static_cast<uint32_t>(AdspFrequency / AudioFramePerSecond * (applicationProcessingTimeLimitPercent / 100.0f) * (m_RenderingTimeLimitPercent / 100.0f));
#else
    // Voice drop is always disabled on Windows.
    return 0;
#endif
}

size_t AudioRenderSystem::GenerateCommand(void* pBuffer, size_t size) NN_NOEXCEPT
{
#if defined(NN_AUDIO_ENABLE_PROFILER) && defined(NN_AUDIO_SYSTEM_PROCESS)
    nn::profiler::ScopedCodeBlock block("GenerateCommand");
    nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_Main);
#endif

    // Reset whole memory pool use state
    server::PoolMapper::ClearUseState(m_pMemoryPoolInfos, m_MemoryPoolCount);

    const auto beginTick = nn::os::GetSystemTick();
    size_t offset = sizeof(CommandListHeader);
    auto header = reinterpret_cast<CommandListHeader*>(pBuffer);

    header->buffer = m_ServicePoolInfo.Translate(m_MixBuffer, sizeof(int32_t) * m_SampleCount * (m_MixBufferCount + m_VoiceChannelCountMax));
    header->bufferCount = m_MixBufferCount + m_VoiceChannelCountMax;
    header->sampleCount = m_SampleCount;
    header->sampleRate = m_SampleRate;

    if (m_SplitterContext.UsingSplitter() == false)
    {
        // require original sort.
        m_MixContext.SortInfo();
    }

    // backup performance data into user frame & prepare performance buffer for next commands.
    if (m_PerformanceManager.IsInitialized())
    {
        m_PerformanceManager.TapFrame(m_IsRenderingTimeLimitExceededOnPreviousFrame, m_VoiceDropCountOnPreviousFrame, m_RenderingStartTimeOnPreviousFrame);
        m_IsRenderingTimeLimitExceededOnPreviousFrame = false;
        m_VoiceDropCountOnPreviousFrame = 0;
        m_RenderingStartTimeOnPreviousFrame = 0;
    }

    NN_SDK_ASSERT_NOT_NULL(m_pCommandProcessingTimeEstimator);
    CommandBuffer commandBuffer(pBuffer, size, offset, m_ServicePoolInfo, m_pCommandProcessingTimeEstimator);

    int currentChannelCount;
    if(m_ExecutionMode == AudioRendererExecutionMode_AutoExecution)
    {
        currentChannelCount = GetCurrentChannelCount(m_SessionId);
    }
    else
    {
        // この値は DeviceSink で使われるが、手動実行時は使用しないのでダミーの値
        currentChannelCount = 2;
    }

    RendererSystemContext sysContext = {
        m_SessionId,
        currentChannelCount,
        m_MixBufferCount,
        m_DepopBuffer,
        m_UpsamplerManager,
        &m_ServicePoolInfo
    };

    // MEMO: コマンド単位の負荷計測を有効にする場合はここを有効にする
#if 0
    PerformanceMeasuredCommandBuffer PerformanceMeasuredCommandBuffer(&commandBuffer, m_PerformanceManager);
    ICommandBuffer* pCommandBuffer = &PerformanceMeasuredCommandBuffer;
#else
    ICommandBuffer* pCommandBuffer = &commandBuffer;
#endif

    auto performanceManager = m_PerformanceManager.IsInitialized() ? &m_PerformanceManager : nullptr;
    CommandGenerator generator(pCommandBuffer, header, &sysContext, &m_VoiceContext, &m_MixContext, &m_EffectContext, &m_SinkContext, &m_SplitterContext, performanceManager);
    m_VoiceContext.SortInfo();

    generator.GenerateVoiceCommands();
    uint32_t processingTimeBudget = GetDspProcessingTimeBudget();
    // Tap current estimation, only voice related commands are generated.
    uint32_t voiceProcessingTime = commandBuffer.GetEstimatedProcessingTime();
    generator.GenerateSubMixCommands();
    generator.GenerateFinalMixCommands();
    generator.GenerateSinkCommands();

    uint32_t processingTimeWithoutVoiceProcessing = commandBuffer.GetEstimatedProcessingTime() - voiceProcessingTime;
    uint32_t voiceProcessingTimeBudget = std::max(0, static_cast<int32_t>(processingTimeBudget) - static_cast<int32_t>(processingTimeWithoutVoiceProcessing));

    if(m_IsVoiceDropEnabled)
    {
        m_VoiceDropCountOnPreviousFrame = DropVoices(commandBuffer, voiceProcessingTime, voiceProcessingTimeBudget);
    }

    header->size = static_cast<int32_t>(commandBuffer.GetCommandTotalSize());
    header->commandCount = commandBuffer.GetCount();

    m_VoiceContext.UpdateStateByDspShared();

    m_Statistics.generateCommandsTotalTick += nn::os::GetSystemTick() - beginTick;
    ++m_Statistics.generateCommandsCount;

    m_RenderingStartTimeOnPreviousFrame = dsp::GetRenderingStartTick(m_SessionId);
    ++m_ElapsedFrameCount;

    return commandBuffer.GetCommandTotalSize();
}

void AudioRenderSystem::SendCommandToDsp() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if(m_IsActive)
    {
        // Clear ReleasedFromDspEvent when this session starts a DSP frame.
        // Clear should not be called if m_IsActive == false.
        m_ReleasedFromDspEvent.Clear();

#if defined(NN_AUDIO_ENABLE_APPLET_VOLUME_MANAGER)
        if(!AppletVolumeManager::IsRunning(AppletVolumeManager::SessionType_AudioRenderer, m_SessionId))
        {
            return;
        }
#endif // defined(NN_AUDIO_ENABLE_APPLET_VOLUME_MANAGER)

        nn::audio::dsp::InvalidateDataCache(m_SharedBuffer, m_SharedBufferSize - m_CommandBufferSize);

        void * buffer;
        size_t size;
        const auto isCommandGenerationNeeded = (dsp::GetRemainCommandCount(m_SessionId) == 0);

        if(isCommandGenerationNeeded)
        {
            size = GenerateCommand(m_CommandBuffer, m_CommandBufferSize);
            buffer = m_CommandBuffer;
        }
        else
        {
            m_IsRenderingTimeLimitExceededOnPreviousFrame = true;
            buffer = m_CommandBuffer;
            size = m_CurrentCommandBufferSize;
        }
        NN_SDK_ASSERT(size <= m_CommandBufferSize, "CommandList size exceed maximum size.");

        nn::audio::dsp::FlushDataCache(m_SharedBuffer, m_SharedBufferSize - m_CommandBufferSize + size);
        nn::audio::dsp::SendCommandBuffer(GetSessionId(), m_ServicePoolInfo.Translate(buffer, size), size, GetDspProcessingTimeBudget(), m_IsNewForDspRenderer, m_AppletResourceUserId.lower);
        m_IsNewForDspRenderer = false;

        m_CurrentCommandBufferSize = size;

        if(isCommandGenerationNeeded)
        {
            SignalSystemEvent();
        }
    }
    else
    {
        dsp::ClearRemainCommandCount(m_SessionId);

        // Notify this AudioRenderSystem is released from DSP.
        m_ReleasedFromDspEvent.Signal();
    }
}

void* AudioRenderSystem::GetCommandBufferWorkBuffer() const NN_NOEXCEPT
{
    return m_CommandBuffer;
}

size_t AudioRenderSystem::GetCommandBufferWorkBufferSize() const NN_NOEXCEPT
{
    return m_CommandBufferSize;
}

int AudioRenderSystem::GetSampleRate() const NN_NOEXCEPT
{
    return m_SampleRate;
}

int AudioRenderSystem::GetSampleCount() const NN_NOEXCEPT
{
    return m_SampleCount;
}

int AudioRenderSystem::GetMixBufferCount() const NN_NOEXCEPT
{
    return m_MixBufferCount;
}

int AudioRenderSystem::GetVoiceCount() const NN_NOEXCEPT
{
    return m_VoiceContext.GetCount();
}

uint32_t AudioRenderSystem::GetSessionId() const NN_NOEXCEPT
{
    return m_SessionId;
}

AudioRendererExecutionMode AudioRenderSystem::GetExecutionMode() const NN_NOEXCEPT
{
    return m_ExecutionMode;
}

AudioRendererRenderingDevice AudioRenderSystem::GetRenderingDevice() const NN_NOEXCEPT
{
    return m_RenderingDevice;
}

Result AudioRenderSystem::QuerySystemEvent(nn::os::NativeHandle* outHandle) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outHandle);
    NN_RESULT_THROW_UNLESS(m_ExecutionMode == AudioRendererExecutionMode_AutoExecution, nn::audio::ResultNotSupported());

    *outHandle = m_SystemEvent.GetReadableHandle();
    NN_RESULT_SUCCESS;
}

void AudioRenderSystem::SignalSystemEvent() NN_NOEXCEPT
{
    m_SystemEvent.Signal();
}

// TODO: Dump more detail information
// TODO: Add auto indent
// TODO: Reduce NN_DETAIL_AUDIO_INFO calls
void AudioRenderSystem::Dump() NN_NOEXCEPT
{
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    NN_DETAIL_AUDIO_INFO("\n");
    NN_DETAIL_AUDIO_INFO("AudioRenderSystem (0x%p) \n", this);
    NN_DETAIL_AUDIO_INFO("  m_IsActive (%d)\n", m_IsActive);
    NN_DETAIL_AUDIO_INFO("  m_RunState (%d)\n", m_RunState);
    NN_DETAIL_AUDIO_INFO("  m_SampleRate (%d)\n", m_SampleRate);
    NN_DETAIL_AUDIO_INFO("  m_SampleCount(%d)\n", m_SampleCount);
    NN_DETAIL_AUDIO_INFO("  m_MixBufferCount (%d)\n", m_MixBufferCount);
    NN_DETAIL_AUDIO_INFO("  m_MixBuffer (0x%p)\n", m_MixBuffer);
    NN_DETAIL_AUDIO_INFO("  m_VoiceCount (%d)\n", m_VoiceContext.GetCount());
    NN_DETAIL_AUDIO_INFO("  m_VoiceInfo (0x%p) m_VoiceStates (0x%p)\n",
        m_VoiceContext.GetCount() > 0 ? &m_VoiceContext.GetInfo(0) : nullptr);

    // TODO: Move to somewhere.
    for(int i = 0; i < m_VoiceContext.GetCount(); ++i)
    {
        const auto& voiceInfo = m_VoiceContext.GetInfo(i);
        const auto& voiceState = m_VoiceContext.GetState(i);

        // TODO: Full dump mode
        if(voiceInfo._inParameter._isInUse == false)
        {
            continue;
        }

        // TODO: Dump all values in VoiceInfo, VoiceState
        NN_DETAIL_AUDIO_INFO("    m_Voices[%d] (0x%p)\n", i, &voiceInfo);
        NN_DETAIL_AUDIO_INFO("      _inParameter._isInUse (%d)\n", voiceInfo._inParameter._isInUse);
        NN_DETAIL_AUDIO_INFO("      _inParameter._id (%d)\n", voiceInfo._inParameter._id);
        NN_DETAIL_AUDIO_INFO("      _inParameter._nodeId (%d)\n", voiceInfo._inParameter._nodeId);
        NN_DETAIL_AUDIO_INFO("      _inParameter._playState (%d)\n", voiceInfo._inParameter._playState);
        NN_DETAIL_AUDIO_INFO("      _inParameter._sampleRate (%d)\n", voiceInfo._inParameter._sampleRate);
        NN_DETAIL_AUDIO_INFO("      _inParameter._sampleFormat (%d)\n", voiceInfo._inParameter._sampleFormat);
        NN_DETAIL_AUDIO_INFO("      _inParameter._channelCount (%d)\n", voiceInfo._inParameter._channelCount);
        NN_DETAIL_AUDIO_INFO("      _inParameter._pitch (%.2f)\n", voiceInfo._inParameter._pitch);
        NN_DETAIL_AUDIO_INFO("      _inParameter._volume (%.2f)\n", voiceInfo._inParameter._volume);
        NN_DETAIL_AUDIO_INFO("      _inParameter._waveBufferCount (%d)\n", voiceInfo._inParameter._waveBufferCount);
        NN_DETAIL_AUDIO_INFO("      _inParameter._waveBufferHead (%d)\n", voiceInfo._inParameter._waveBufferHead);
        NN_DETAIL_AUDIO_INFO("      _inParameter._destinationMixId (%d)\n", voiceInfo.GetDestinationMixId());
        NN_DETAIL_AUDIO_INFO("      _inParameter._isNew (%d)\n", voiceInfo._inParameter._isNew);

        NN_DETAIL_AUDIO_INFO("    m_VoiceStates[%d] (0x%p)\n", i, &voiceState);
        NN_DETAIL_AUDIO_INFO("      playedSampleCount (%lld)\n", voiceState.playedSampleCount);
        NN_DETAIL_AUDIO_INFO("      offset (%d)\n", voiceState.offset);
        NN_DETAIL_AUDIO_INFO("      waveBufferIndex (%d)\n", voiceState.waveBufferIndex);
        NN_DETAIL_AUDIO_INFO("      isWaveBufferValid (%d %d %d %d)\n", voiceState.isWaveBufferValid[0]
                                                            , voiceState.isWaveBufferValid[1]
                                                            , voiceState.isWaveBufferValid[2]
                                                            , voiceState.isWaveBufferValid[3]);
        NN_DETAIL_AUDIO_INFO("      waveBufferConsumed (%d)\n", voiceState.waveBufferConsumed);
        NN_DETAIL_AUDIO_INFO("      fraction (%d)\n", voiceState.fraction);
        NN_DETAIL_AUDIO_INFO("      externalContext (0x%p)\n", voiceState.externalContext);
    }

    NN_DETAIL_AUDIO_INFO("  m_VoiceStatesDspShared (0x%p)\n",
        m_VoiceContext.GetCount() > 0 ? &m_VoiceContext.GetDspSharedState(0) : nullptr);
    NN_DETAIL_AUDIO_INFO("  m_CurrentVoiceCount (%d)\n", m_VoiceContext.GetActiveCount());
    NN_DETAIL_AUDIO_INFO("  m_EffectCount (%d)\n", m_EffectContext.GetCount());
    NN_DETAIL_AUDIO_INFO("  m_EffectInfos (0x%p)\n", m_EffectContext.GetCount() > 0 ? &m_EffectContext.GetInfo(0) : nullptr);
    NN_DETAIL_AUDIO_INFO("  m_MemoryPoolCount (%d)\n", m_MemoryPoolCount);
    NN_DETAIL_AUDIO_INFO("  m_pMemoryPoolInfos (0x%p)\n", m_pMemoryPoolInfos);
    NN_DETAIL_AUDIO_INFO("  m_CommandBuffer (0x%p)\n", m_CommandBuffer);
    NN_DETAIL_AUDIO_INFO("  m_UpsamplerCount (%d)\n", m_UpsamplerCount);
    NN_DETAIL_AUDIO_INFO("  m_UpsamplerManager (0x%p)\n", m_UpsamplerManager);
    NN_DETAIL_AUDIO_INFO("  m_UpsamplerInfos (0x%p)\n", m_UpsamplerInfos);
    NN_DETAIL_AUDIO_INFO("  m_MixInfosCount (%d)\n", m_MixContext.GetCount());
    NN_DETAIL_AUDIO_INFO("  m_MixInfos (0x%p)\n", m_MixContext.GetCount() > 0 ? &m_MixContext.GetInfo(0) : nullptr);
    NN_DETAIL_AUDIO_INFO("  m_pMixInfosSortWork (0x%p)\n", m_MixContext.GetCount() > 0 ? &m_MixContext.GetSortedInfo(0) : nullptr);
    NN_DETAIL_AUDIO_INFO("  m_SinkCount (%d)\n", m_SinkContext.GetCount());
    NN_DETAIL_AUDIO_INFO("  m_SinkInfos (0x%p)\n", m_SinkContext.GetCount() > 0 ? &m_SinkContext.GetInfo(0) : nullptr);
    NN_DETAIL_AUDIO_INFO("  m_SessionId (%u)\n", m_SessionId);
    NN_DETAIL_AUDIO_INFO("  m_VoiceChannelCountMax (%d)\n", m_VoiceChannelCountMax);
    NN_DETAIL_AUDIO_INFO("  m_PerformanceBuffer (0x%p)\n", m_PerformanceBuffer);
    NN_DETAIL_AUDIO_INFO("  m_WorkBuffer (0x%p)\n", m_WorkBuffer);
    NN_DETAIL_AUDIO_INFO("  m_WorkBufferSize (%zu)\n", m_WorkBufferSize);
    NN_DETAIL_AUDIO_INFO("  m_Statistics\n");
    NN_DETAIL_AUDIO_INFO("    generateCommandsAve (%lld) [us]\n", nn::os::ConvertToTimeSpan(m_Statistics.generateCommandsTotalTick).GetNanoSeconds() / m_Statistics.generateCommandsCount / 1000);
    NN_DETAIL_AUDIO_INFO("    generateCommandsCount (%llu)\n", m_Statistics.generateCommandsCount);
    NN_DETAIL_AUDIO_INFO("    updateAve (%lld) [us]\n", nn::os::ConvertToTimeSpan(m_Statistics.updateTotalTick).GetNanoSeconds() / m_Statistics.updateCount / 1000);
    NN_DETAIL_AUDIO_INFO("    updateCount (%llu)\n", m_Statistics.updateCount);

    NN_DETAIL_AUDIO_INFO(" CommandList:\n");
    common::DumpCommandList(m_CommandBuffer, m_CurrentCommandBufferSize);
#endif // #if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
}

}}}  // namespace nn::audio::server
