﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <mutex>
#include <nn/audio/detail/audio_Log.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/dd/dd_DeviceAddressSpace.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/audio/audio_ResultForPrivate.h>

#include "audio_ServiceDiagnostics.h"
#include "audio_AudioOutManagerImpl.h"
#include "audio_SystemEventHolderPool.h"
#include "../common/audio_AudioOutSession.h"

#include "audio_AudioOutSession.h"

namespace nn { namespace audio { namespace server {

namespace {
nn::audio::server::SystemEventHolderPool<nn::audio::common::AudioOutSessionCountMax> g_SystemEventHolderPool;
} // anonymous namespace

class AudioOutManagerImpl::AudioOutImpl
{
public:
    explicit AudioOutImpl(AudioOutManagerImpl* pParent, uint32_t sessionId) NN_NOEXCEPT
        : m_SystemEventHolder(g_SystemEventHolderPool.Acquire())
        , m_AudioOutClass(sessionId, m_SystemEventHolder.GetSystemEvent())
        , m_Parent(pParent, true)
    {
        m_pParentMutex = m_Parent->GetMutex();
    }

    ~AudioOutImpl() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lock(*m_pParentMutex);
        m_AudioOutClass.Close();
        m_Parent->ReportCloseSession(m_AudioOutClass.GetSessionId());
        g_SystemEventHolderPool.Release(m_SystemEventHolder);
    }

    nn::Result GetAudioOutState(nn::sf::Out<uint32_t> outAudioOut) NN_NOEXCEPT;
    nn::Result Start() NN_NOEXCEPT;
    nn::Result Stop() NN_NOEXCEPT;
    nn::Result Sleep() NN_NOEXCEPT;
    nn::Result Wake() NN_NOEXCEPT;
    nn::Result AppendAudioOutBuffer(nn::sf::InBuffer inAudioBuffer, uint64_t outBufferClientPtr) NN_NOEXCEPT;
    nn::Result AppendAudioOutBufferAuto(nn::sf::InBuffer inAudioBuffer, uint64_t outBufferClientPtr) NN_NOEXCEPT;
    nn::Result RegisterBufferEvent(nn::sf::Out<nn::sf::NativeHandle> bufferEvent) NN_NOEXCEPT;
    nn::Result GetReleasedAudioOutBuffers(nn::sf::OutBuffer outAudioBuffer, nn::sf::Out<int> count) NN_NOEXCEPT;
    nn::Result GetReleasedAudioOutBuffersAuto(nn::sf::OutBuffer outAudioBuffer, nn::sf::Out<int> count) NN_NOEXCEPT;
    nn::Result GetAudioOutBufferCount(nn::sf::Out<int> outAudioOutBufferCount) NN_NOEXCEPT;
    nn::Result GetAudioOutPlayedSampleCount(nn::sf::Out<int64_t> outAudioOutPlayedSampleCount) NN_NOEXCEPT;
    nn::Result FlushAudioOutBuffers(nn::sf::Out<bool> isAllBufferFlushed) NN_NOEXCEPT;
    nn::Result SetAudioOutVolume(float volume) NN_NOEXCEPT;
    nn::Result GetAudioOutVolume(nn::sf::Out<float> volume) NN_NOEXCEPT;
    nn::Result ContainsAudioOutBuffer(uint64_t audioBufferPointer, nn::sf::Out<bool> contains) NN_NOEXCEPT;
    nn::Result Initialize(nn::sf::InBuffer name,  nn::audio::AudioOutParameter param, nn::sf::NativeHandle& processHandle, nn::sf::Out<nn::audio::detail::AudioOutParameterInternal> audioOutInter, nn::sf::OutBuffer nameOut, const nn::applet::AppletResourceUserId& appletId) NN_NOEXCEPT;
    nn::Result Update() NN_NOEXCEPT;

private:
    nn::Result AppendAudioOutBufferInternal(nn::sf::InBuffer inAudioBuffer, uint64_t outBufferClientPtr) NN_NOEXCEPT;
    nn::Result GetReleasedAudioOutBuffersInternal(nn::sf::OutBuffer outAudioBuffer, nn::sf::Out<int> count) NN_NOEXCEPT;

private:
    SystemEventHolder& m_SystemEventHolder;
    nn::audio::detail::AudioOutSession m_AudioOutClass;

    nn::sf::SharedPointer<AudioOutManagerImpl> m_Parent;
    nn::os::Mutex* m_pParentMutex;
};

nn::Result AudioOutManagerImpl::AudioOutImpl::GetAudioOutState(nn::sf::Out<uint32_t> outAudioOut) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    //TODO: Session state -> AudioOut state
    outAudioOut.Set(m_AudioOutClass.GetState());
    NN_RESULT_SUCCESS;
}

nn::Result AudioOutManagerImpl::AudioOutImpl::Start() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);

    return m_AudioOutClass.Start();
}

nn::Result AudioOutManagerImpl::AudioOutImpl::Stop() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);

    m_AudioOutClass.Stop();
    NN_RESULT_SUCCESS;
}

nn::Result AudioOutManagerImpl::AudioOutImpl::Sleep() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    m_AudioOutClass.Sleep();
    NN_RESULT_SUCCESS;
}

nn::Result AudioOutManagerImpl::AudioOutImpl::Wake() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    m_AudioOutClass.Wake();
    NN_RESULT_SUCCESS;
}

nn::Result AudioOutManagerImpl::AudioOutImpl::AppendAudioOutBufferInternal(nn::sf::InBuffer inAudioBuffer, uint64_t outBufferClientPtr) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    const AudioOutBuffer* pTempBuffer = reinterpret_cast<const AudioOutBuffer*>(inAudioBuffer.GetPointerUnsafe());

    auto couldAppend = m_AudioOutClass.AppendBuffer(pTempBuffer->buffer, pTempBuffer->size, reinterpret_cast<const void*>(outBufferClientPtr));

    NN_RESULT_THROW_UNLESS(couldAppend, nn::audio::ResultBufferCountMax());

    NN_RESULT_SUCCESS;
}

nn::Result AudioOutManagerImpl::AudioOutImpl::AppendAudioOutBuffer(nn::sf::InBuffer inAudioBuffer, uint64_t outBufferClientPtr) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    return AppendAudioOutBufferInternal(inAudioBuffer, outBufferClientPtr);
}

nn::Result AudioOutManagerImpl::AudioOutImpl::AppendAudioOutBufferAuto(nn::sf::InBuffer inAudioBuffer, uint64_t outBufferClientPtr) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    return AppendAudioOutBufferInternal(inAudioBuffer, outBufferClientPtr);
}

nn::Result AudioOutManagerImpl::AudioOutImpl::RegisterBufferEvent(nn::sf::Out<nn::sf::NativeHandle>  bufferEvent) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    nn::os::NativeHandle handle;
    m_AudioOutClass.RegisterBufferEvent(&handle);
    bufferEvent.Set(nn::sf::NativeHandle(handle, false));
    NN_RESULT_SUCCESS;
}

nn::Result AudioOutManagerImpl::AudioOutImpl::GetReleasedAudioOutBuffersInternal(nn::sf::OutBuffer outAudioBuffer, nn::sf::Out<int> count) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    uintptr_t* pAudioArray = reinterpret_cast<uintptr_t*>(outAudioBuffer.GetPointerUnsafe());
    pAudioArray[0] = 0;
    size_t length = outAudioBuffer.GetSize() / sizeof(*pAudioArray);

    int loops = 0;
    for( unsigned int i = 0; i < length; ++i )
    {
        auto sessionBuffer = m_AudioOutClass.GetReleasedBuffer();

        if(!sessionBuffer)
        {
            break;
        }

        void* userAddr = const_cast<void*>(sessionBuffer->userAddr);
        nn::audio::AudioOutBuffer* pAudioBuffer = reinterpret_cast<nn::audio::AudioOutBuffer*>(userAddr);

        if( pAudioBuffer != nullptr )
        {
            pAudioArray[i] = reinterpret_cast<uintptr_t>(pAudioBuffer);
            ++loops;
        }
        else
        {
            break;
        }
    }
    count.Set(loops);
    NN_RESULT_SUCCESS;
}

nn::Result AudioOutManagerImpl::AudioOutImpl::GetReleasedAudioOutBuffers(nn::sf::OutBuffer outAudioBuffer, nn::sf::Out<int> count) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    return GetReleasedAudioOutBuffersInternal(outAudioBuffer, count);
}

nn::Result AudioOutManagerImpl::AudioOutImpl::GetReleasedAudioOutBuffersAuto(nn::sf::OutBuffer outAudioBuffer, nn::sf::Out<int> count) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    return GetReleasedAudioOutBuffersInternal(outAudioBuffer, count);
}

nn::Result AudioOutManagerImpl::AudioOutImpl::ContainsAudioOutBuffer(uint64_t audioBufferPointer, nn::sf::Out<bool> contains) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    //TODO this can just be a direct internal call once we remove transfer memory
    bool found = m_AudioOutClass.ContainsBuffer(reinterpret_cast<const void*>(audioBufferPointer));
    contains.Set(found);
    NN_RESULT_SUCCESS;
}

nn::Result AudioOutManagerImpl::AudioOutImpl::Initialize(nn::sf::InBuffer name,  nn::audio::AudioOutParameter param, nn::sf::NativeHandle& processHandle, nn::sf::Out<nn::audio::detail::AudioOutParameterInternal> audioOutInter, nn::sf::OutBuffer nameOut, const nn::applet::AppletResourceUserId& appletId) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    const char* audioName = name.GetPointerUnsafe();

    NN_SDK_REQUIRES_NOT_NULL(audioName);

    SessionFormat format;
    format.sampleRate = param.sampleRate;
    format.channelCount = param.channelCount;
    format.format = SampleFormat_PcmInt16;

    NN_RESULT_DO(m_AudioOutClass.Open(audioName, format, processHandle.GetOsHandle(), appletId));
    processHandle.Detach();
    audioOutInter->sampleRate = m_AudioOutClass.GetSampleRate();
    audioOutInter->channelCount = m_AudioOutClass.GetChannelCount();
    audioOutInter->state = m_AudioOutClass.GetState();
    audioOutInter->sampleFormat = m_AudioOutClass.GetSampleFormat();
    const char* nameSrc = m_AudioOutClass.GetName();
    strcpy(nameOut.GetPointerUnsafe(),nameSrc);

    NN_RESULT_SUCCESS;
}

nn::Result AudioOutManagerImpl::AudioOutImpl::GetAudioOutBufferCount(nn::sf::Out<int> outAudioOutBufferCount) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    int count = m_AudioOutClass.GetAppendedBufferCount();
    outAudioOutBufferCount.Set(count);
    NN_RESULT_SUCCESS;
}

nn::Result AudioOutManagerImpl::AudioOutImpl::GetAudioOutPlayedSampleCount(nn::sf::Out<int64_t> outAudioOutPlayedSampleCount) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    int64_t sampleCount = m_AudioOutClass.GetSamplesProcessed();
    outAudioOutPlayedSampleCount.Set(sampleCount);
    NN_RESULT_SUCCESS;
}

nn::Result AudioOutManagerImpl::AudioOutImpl::FlushAudioOutBuffers(nn::sf::Out<bool> isAllBufferFlushed) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    *isAllBufferFlushed = m_AudioOutClass.FlushBuffers();
    NN_RESULT_SUCCESS;
}

nn::Result AudioOutManagerImpl::AudioOutImpl::SetAudioOutVolume(float volume) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    m_AudioOutClass.SetAudioOutVolume(volume);
    NN_RESULT_SUCCESS;
}

nn::Result AudioOutManagerImpl::AudioOutImpl::GetAudioOutVolume(nn::sf::Out<float> volume) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    *volume = m_AudioOutClass.GetAudioOutVolume();
    NN_RESULT_SUCCESS;
}

nn::Result AudioOutManagerImpl::AudioOutImpl::Update() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    return m_AudioOutClass.Update();
}

AudioOutManagerImpl::AudioOutManagerImpl() NN_NOEXCEPT
    : m_IsInitialized(false)
    , m_pSemaphore(nullptr)
    , m_Mutex(true)
    , m_IsThreadActive(false)
    , m_Thread()
    , m_IsAwake(true)
{
    m_HeapHandle = nn::lmem::CreateExpHeap( &m_HeapBuffer, sizeof(m_HeapBuffer), nn::lmem::CreationOption_ThreadSafe );
    m_Allocator.Attach(m_HeapHandle);

    os::InitializeEvent(&m_EventSemaphoreSignal, false, nn::os::EventClearMode_AutoClear);
    for( int i = 0; i < NumberOfSessions; ++i)
    {
        m_Session[i] = nullptr;
        m_AppletId[i] = nn::applet::AppletResourceUserId::GetInvalidId();
    }
}

AudioOutManagerImpl::~AudioOutManagerImpl() NN_NOEXCEPT
{
    if( m_IsInitialized )
    {
        m_IsThreadActive = false;
        nn::os::WaitThread( &m_Thread );
        nn::os::DestroyThread( &m_Thread );

        m_Allocator.Deallocate(m_Stack, SessionStackSize);
        nn::audio::server::AudioOutFinalize();
    }
    os::FinalizeEvent(&m_EventSemaphoreSignal);
    nn::lmem::DestroyExpHeap(m_HeapHandle);
}

nn::Result AudioOutManagerImpl::OpenAudioOutInternal(nn::sf::Out<nn::sf::SharedPointer<detail::IAudioOut>> outAudioOut, nn::sf::InBuffer name,  nn::audio::AudioOutParameter param, nn::sf::NativeHandle& processHandle, nn::sf::Out<nn::audio::detail::AudioOutParameterInternal> audioOutInter, nn::sf::OutBuffer nameOut, nn::applet::AppletResourceUserId appletId) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    if( !m_IsInitialized )
    {
        nn::audio::server::AudioOutInitialize(&m_pSemaphore, &m_EventSemaphoreSignal);
        m_Stack = static_cast<char*>(nn::lmem::AllocateFromExpHeap(m_HeapHandle, SessionStackSize, nn::os::ThreadStackAlignment));
        m_IsThreadActive = true;

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_Thread, ThreadFunc, this, m_Stack, SessionStackSize, NN_SYSTEM_THREAD_PRIORITY(audio, AudioOutManager)));
        nn::os::SetThreadNamePointer(&m_Thread, NN_SYSTEM_THREAD_NAME(audio, AudioOutManager));
        nn::os::StartThread(&m_Thread);
        m_IsInitialized = true;
    }

    int session=-1;
    nn::Result res = m_Sessions.GetNextSessions(&session);

    if( res.IsFailure() )
    {
        return res;
    }

    typedef nn::sf::ObjectFactory<MyAllocator::Policy> Factory;

    auto p = Factory::CreateSharedEmplaced<detail::IAudioOut, AudioOutManagerImpl::AudioOutImpl>(&m_Allocator, this, static_cast<uint32_t>(session));
    NN_SDK_ASSERT_NOT_NULL(p);
    m_Session[session] = Factory::GetEmplacedImplPointer<AudioOutManagerImpl::AudioOutImpl>(p);
    m_AppletId[session] = appletId;
    res = m_Session[session]->Initialize(name, param, processHandle, audioOutInter, nameOut, appletId);
    *outAudioOut = std::move(p);

    NN_AUDIO_SERVICE_LOG("[AudioOut] Open sessionId(%d) appletResourceUserId.lower(%llu) name(%s) sampleRate(%d) channelCount(%d)\n",
            session, appletId.lower, name.GetPointerUnsafe(), param.sampleRate, param.channelCount);

    return res;
}

nn::Result AudioOutManagerImpl::OpenAudioOut(nn::sf::Out<nn::sf::SharedPointer<detail::IAudioOut>> outAudioOut, nn::sf::InBuffer name,  nn::audio::AudioOutParameter param, nn::sf::NativeHandle processHandle, nn::sf::Out<nn::audio::detail::AudioOutParameterInternal> audioOutInter, nn::sf::OutBuffer nameOut, nn::applet::AppletResourceUserId appletId) NN_NOEXCEPT
{
    return OpenAudioOutInternal(outAudioOut, name, param, processHandle, audioOutInter, nameOut, appletId);
}

nn::Result AudioOutManagerImpl::OpenAudioOutAuto(nn::sf::Out<nn::sf::SharedPointer<detail::IAudioOut>> outAudioOut, nn::sf::InBuffer name,  nn::audio::AudioOutParameter param, nn::sf::NativeHandle processHandle, nn::sf::Out<nn::audio::detail::AudioOutParameterInternal> audioOutInter, nn::sf::OutBuffer nameOut, nn::applet::AppletResourceUserId appletId) NN_NOEXCEPT
{
    return OpenAudioOutInternal(outAudioOut, name, param, processHandle, audioOutInter, nameOut, appletId);
}

nn::Result AudioOutManagerImpl::ListAudioOutsInternal(nn::sf::OutBuffer outAudioInfo, nn::sf::Out<int> amountRet) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    if( !m_IsInitialized )
    {
        nn::audio::server::AudioOutInitialize(&m_pSemaphore, &m_EventSemaphoreSignal);
        m_Stack = static_cast<char*>(nn::lmem::AllocateFromExpHeap(m_HeapHandle, SessionStackSize, nn::os::ThreadStackAlignment));
        m_IsThreadActive = true;

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_Thread, ThreadFunc, this, m_Stack, SessionStackSize, NN_SYSTEM_THREAD_PRIORITY(audio, AudioOutManager)));
        nn::os::SetThreadNamePointer(&m_Thread, NN_SYSTEM_THREAD_NAME(audio, AudioOutManager));
        nn::os::StartThread(&m_Thread);
        m_IsInitialized = true;
    }

    nn::audio::AudioOutInfo* pInfo = reinterpret_cast<nn::audio::AudioOutInfo*>(outAudioInfo.GetPointerUnsafe());
    size_t length = outAudioInfo.GetSize() / sizeof(*pInfo);

    int count = nn::audio::server::ListAudioOuts(pInfo, static_cast<int>(length));
    amountRet.Set( count );
    NN_RESULT_SUCCESS;
}

nn::Result AudioOutManagerImpl::ListAudioOuts(nn::sf::OutBuffer outAudioInfo, nn::sf::Out<int> amountRet) NN_NOEXCEPT
{
    return ListAudioOutsInternal(outAudioInfo, amountRet);
}

nn::Result AudioOutManagerImpl::ListAudioOutsAuto(nn::sf::OutBuffer outAudioInfo, nn::sf::Out<int> amountRet) NN_NOEXCEPT
{
    return ListAudioOutsInternal(outAudioInfo, amountRet);
}

void AudioOutManagerImpl::ReportCloseSession(int sessionId) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    m_Sessions.InsertSession(sessionId);
    m_Session[sessionId] = nullptr;
    m_AppletId[sessionId] = nn::applet::AppletResourceUserId::GetInvalidId();
}

nn::Result AudioOutManagerImpl::Sleep() NN_NOEXCEPT
{
    if( m_IsAwake )
    {
        std::lock_guard<os::Mutex> lock(m_Mutex);
        for( int i = 0; i < NumberOfSessions; ++i )
        {
            if( m_Session[i] != nullptr )
            {
                m_Session[i]->Sleep();
            }
        }
        m_IsAwake = false;
    }
    NN_RESULT_SUCCESS;
}

nn::Result AudioOutManagerImpl::Wake() NN_NOEXCEPT
{
    if( !m_IsAwake )
    {
        std::lock_guard<os::Mutex> lock(m_Mutex);
        for( int i = 0; i < NumberOfSessions; ++i )
        {
            if( m_Session[i] != nullptr )
            {
                m_Session[i]->Wake();
            }
        }
        m_IsAwake = true;
    }
    NN_RESULT_SUCCESS;
}

nn::os::Mutex* AudioOutManagerImpl::GetMutex() NN_NOEXCEPT
{
    return &m_Mutex;
}

void AudioOutManagerImpl::ThreadFunc(void* arg) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(arg);
    reinterpret_cast<AudioOutManagerImpl*>(arg)->ThreadFuncImpl();
}

void AudioOutManagerImpl::ThreadFuncImpl() NN_NOEXCEPT
{
    const nn::TimeSpan SemaphoreTimeoutMilliseconds = nn::TimeSpan::FromMilliSeconds(20);
    while( m_IsThreadActive )
    {
        auto acquired = nn::os::TimedAcquireSemaphore( m_pSemaphore, SemaphoreTimeoutMilliseconds);
        if (acquired)
        {
            std::lock_guard<os::Mutex> lock(m_Mutex);
            for (int i = 0; i < NumberOfSessions; ++i)
            {
                if (m_Session[i] != nullptr)
                {
                    m_Session[i]->Update();
                }
            }
            nn::os::SignalEvent(&m_EventSemaphoreSignal);
        }
    }
}

}}}
