﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once
#include <nn/audio/audio_AudioInTypes.h>
#include <nn/audio/audio_AudioOutTypes.h>
#include <nn/audio/audio_FinalOutputRecorderTypes.h>
#include <nn/applet/applet_Types.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/os/os_Event.h>

#include "../common/audio_AudioRendererSession.h"

namespace nn { namespace audio {

class AppletVolumeManager
{
public:
    enum SessionType
    {
        SessionType_AudioRenderer = 0,
        SessionType_AudioOut,
        SessionType_AudioIn,
        SessionType_FinalOutputRecorder,
        SessionType_Count,
    };

    enum SessionState
    {
        SessionState_Closed,
        SessionState_Running,
        SessionState_Suspended,
        SessionState_None,
    };

    enum DeviceType
    {
        DeviceType_AhubHeadphone = 0,
        DeviceType_AhubSpeaker,
        DeviceType_Hda,
        DeviceType_UsbOutputDevice,
        DeviceType_Count,
    };

    //TODO: Investigate why MaxRegisteredApplets had to be lowered for SIGLO-52119
    static const int MaxRegisteredApplets = 16;
    static const int MaxRendererSessions = common::AudioRendererSessionCountMax;

    //TODO SIGLO-71863: Use AudioOutMangaer's "NumberOfSessions" or gmix "NumAudioOutSessions"
    static const int MaxOutSessions = 12; //AudioOutCountMax
    static const int MaxInSessions = 4; //AudioInCountMax
    static const int MaxRecorderSessions = FinalOutputRecorderCountMax;
    static const int MaxTotalSessions = MaxRendererSessions + MaxOutSessions + MaxInSessions + MaxRecorderSessions;
    static const int LerpIntervalMilliseconds = 5;
    static const int MaxDevices = DeviceType_Count;
    static const int NumSessionTypes = SessionType_Count;

public:
    static Result Initialize() NN_NOEXCEPT;
    static Result Finalize() NN_NOEXCEPT;

    static nn::Result RegisterAppletResourceUserId(nn::applet::AppletResourceUserId id) NN_NOEXCEPT;
    static nn::Result UnregisterAppletResourceUserId(nn::applet::AppletResourceUserId id) NN_NOEXCEPT;

    static nn::Result Sleep() NN_NOEXCEPT;
    static nn::Result Wake() NN_NOEXCEPT;

    static nn::Result Suspend(SessionType type, int64_t durationNano, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    static nn::Result Resume(SessionType type, int64_t durationNano, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    static nn::Result ForceResume(SessionType type, int sessionId, bool shouldMute, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    static nn::Result SetAppletVolume(SessionType type, float endVolume, int64_t durationNano, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    static nn::Result GetAppletVolume(float* outVolume, SessionType type, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    static nn::Result SetRecordVolume(SessionType type, float endVolume, int64_t durationNano, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    static nn::Result GetRecordVolume(float* outVolume, SessionType type, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    static nn::Result SetVolume(SessionType type, int sessionId, float volume, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    static nn::Result GetVolume(float* outVolume, SessionType type, int sessionId, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    static nn::Result SetDevice(AppletVolumeManager::DeviceType device, int channelCount) NN_NOEXCEPT;
    static nn::Result GetActiveDevice(DeviceType* pDevice) NN_NOEXCEPT;
    static nn::Result QueryDeviceEvent(nn::os::NativeHandle* outHandle, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    static nn::Result QueryInputNotificationEvent(nn::os::NativeHandle* outHandle, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    static nn::Result QueryOutputNotificationEvent(nn::os::NativeHandle* outHandle, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    static nn::Result SetDeviceVolume(float volume, AppletVolumeManager::DeviceType device, SessionType type, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    static nn::Result GetDeviceVolume(float* outVolume, AppletVolumeManager::DeviceType device, SessionType type, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    static bool StartDeviceFadeOut(nn::os::Event* pEvent, int64_t durationNano) NN_NOEXCEPT;
    static bool StartDeviceFadeIn(nn::os::Event* pEvent, int64_t durationNano) NN_NOEXCEPT;
    static float GetDeviceMasterVolume() NN_NOEXCEPT;
    static int GetDeviceChannelCount() NN_NOEXCEPT;

    static void SignalInputNotification() NN_NOEXCEPT;
    static void SignalOutputNotification() NN_NOEXCEPT;

    static nn::Result SuspendForDebugger(SessionType type, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    static nn::Result ResumeForDebugger(SessionType type, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;

    static nn::Result OpenSession(SessionType type, int index, const nn::applet::AppletResourceUserId& id, bool isGmixSession) NN_NOEXCEPT;
    static nn::Result CloseSession(SessionType type, int index) NN_NOEXCEPT;
    static bool IsRunning(SessionType type, int index) NN_NOEXCEPT;
    static void Dump() NN_NOEXCEPT;
};

}}  // namespace nn::audio
