﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include "audio_MailBoxImplByMessageQueue.h"

namespace nn { namespace audio { namespace dsp {

NN_DEFINE_STATIC_CONSTANT(const int MailBoxImplByMessageQueue::InvalidMailBoxIndex);
NN_DEFINE_STATIC_CONSTANT(const int MailBoxImplByMessageQueue::InvalidMailBoxId);
NN_DEFINE_STATIC_CONSTANT(const int MailBoxImplByMessageQueue::MailBoxCount);
NN_DEFINE_STATIC_CONSTANT(const int MailBoxImplByMessageQueue::MessageQueueCount);
nn::os::MessageQueueType MailBoxImplByMessageQueue::g_MessageQueues[MailBoxImplByMessageQueue::MessageQueueCount];
uintptr_t MailBoxImplByMessageQueue::g_MessageQueueBuffer[MailBoxImplByMessageQueue::MessageQueueCount];
int MailBoxImplByMessageQueue::g_IdMap[MailBoxCount];

void MailBoxImplByMessageQueue::Initialize() NN_NOEXCEPT
{
    for(int i = 0; i < MessageQueueCount; ++i)
    {
        auto& messageQueue = g_MessageQueues[i];
        nn::os::InitializeMessageQueue(&messageQueue, &g_MessageQueueBuffer[i], 1);
    }

    for(int i = 0; i < MailBoxCount; ++i)
    {
        g_IdMap[i] = InvalidMailBoxId;
    }
}

void MailBoxImplByMessageQueue::Finalize() NN_NOEXCEPT
{
    for(int i = 0; i < MessageQueueCount; ++i)
    {
        auto& messageQueue = g_MessageQueues[i];
        nn::os::FinalizeMessageQueue(&messageQueue);
    }

    for(int i = 0; i < MailBoxCount; ++i)
    {
        g_IdMap[i] = InvalidMailBoxId;
    }
}

void MailBoxImplByMessageQueue::Open(int id, bool isServer) NN_NOEXCEPT
{
    int mailBoxIndex = InvalidMailBoxIndex;
    int freeMailBoxIndex = InvalidMailBoxIndex;

    for(int i = 0; i < MailBoxCount; ++i)
    {
        if(g_IdMap[i] == id)
        {
            mailBoxIndex = i;
            NN_SDK_ASSERT_RANGE(mailBoxIndex, 0, MailBoxCount);

            g_IdMap[mailBoxIndex] = id;
            break;
        }

        if(freeMailBoxIndex == InvalidMailBoxId && g_IdMap[i] == InvalidMailBoxId)
        {
            freeMailBoxIndex = i;
        }
    }

    NN_SDK_ASSERT(mailBoxIndex != InvalidMailBoxId || freeMailBoxIndex != InvalidMailBoxId, "Free mailbox is not found.");

    if(mailBoxIndex == InvalidMailBoxIndex)
    {
        mailBoxIndex = freeMailBoxIndex;
    }

    m_MailBoxIndex = mailBoxIndex;

    if(isServer)
    {
        m_SenderId = mailBoxIndex * 2;
        m_ReceiverId = mailBoxIndex * 2 + 1;
    }
    else
    {
        m_SenderId = mailBoxIndex * 2 + 1;
        m_ReceiverId = mailBoxIndex * 2;
    }
}

void MailBoxImplByMessageQueue::Close() NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(m_SenderId, 0, MessageQueueCount);
    NN_SDK_ASSERT_RANGE(m_ReceiverId, 0, MessageQueueCount);
    NN_SDK_ASSERT_RANGE(m_MailBoxIndex, 0, MailBoxCount);

    m_SenderId = MessageQueueCount;
    m_ReceiverId = MessageQueueCount;
    g_IdMap[m_MailBoxIndex] = InvalidMailBoxId;
    m_MailBoxIndex = InvalidMailBoxIndex;
}

void MailBoxImplByMessageQueue::Send(int32_t message) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(m_SenderId, 0, MessageQueueCount);

    auto& messageQueue = g_MessageQueues[m_SenderId];
    nn::os::SendMessageQueue(&messageQueue, static_cast<uintptr_t>(message));
}

int32_t MailBoxImplByMessageQueue::Recv() NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(m_ReceiverId, 0, MessageQueueCount);

    uintptr_t ret;
    auto& messageQueue = g_MessageQueues[m_ReceiverId];

    nn::os::ReceiveMessageQueue(&ret, &messageQueue);
    return static_cast<int32_t>(ret);
}

}}}  // namespace nn::audio::dsp

