﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "audio_DspCommon.h"
#include "audio_EffectDelay.h"

// Note: NEON optimized version has not been implemented yet.
#if (defined(__ARM_NEON__) || defined(__ARM_NEON)) && 0
#include "detail/audio_EffectDelay.neon.h"
#else  // defined(__ARM_NEON__) || defined(__ARM_NEON)
#include "detail/audio_EffectDelay.generic.h"
#endif  // defined(__ARM_NEON__) || defined(__ARM_NEON)

namespace nn { namespace audio {

// Initialises the delay effect structure.
void InitializeDelayEffect(const DelayParameter* pParameter, DelayState* state, void* workBuffer)
{
    memset(state, 0, sizeof(DelayState));

    auto basePtr = reinterpret_cast<uintptr_t>(workBuffer);
    for (auto i = 0 ; i < pParameter->_numChannels; i++)
    {
        const auto maxDelay = pParameter->_delayTimeMax * (pParameter->_sampleRate >> QF_FRACTIONAL_BIT_COUNT) / 1000;
        DelayLineInit(&state->_delayLines[i], maxDelay, reinterpret_cast<qf*>(basePtr));
        const auto sampleCount = DelayLineCalculateDelaySamples(pParameter->_sampleRate / 1000, pParameter->_delayTime * (1 << QF_FRACTIONAL_BIT_COUNT));
        DelayLineSetDelay(&state->_delayLines[i], sampleCount);
        DelayLineClear(&state->_delayLines[i]);

        basePtr += sizeof(qf) * maxDelay;

        // Reset LPF
        *(state->_pLowPassFilterHistory + i) = 0;
    }

    SetDelayEffectParameter(pParameter, state);
}

void SetDelayEffectParameter(const DelayParameter *pParameter, DelayState* state)
{
    qf t;

    // calc feedback coeffs
    state->_feedback = MultiQfQf(pParameter->_feedbackGain, QF_CONST(0.98f)); // feedback = feedback * 0.98 (BF: was 0.99)

    t = SubtractQfQf(QF_CONST(1.0f), pParameter->_channelSpread);
    state->_feedbackDirect = MultiQfQf(t, state->_feedback);              // feedbackDirect = (1.0 - channel_spread) * fb

    // Channel spread
    t = pParameter->_channelSpread;
    if (pParameter->_numChannels == 4 || pParameter->_numChannels == 6)
    {
        t = MultiQfQf(t, QF_CONST(0.5f));
    }
    state->_feedbackCross = MultiQfQf(t, state->_feedback);               // feedbackCross  = (channel_spread (divided by 2 if four channel)) * fb

    // set LPF coeff
    state->_lowPassFilterCoefficientA1 = MultiQfQf(QF_CONST(0.95f), pParameter->_lowPassAmount); // lowPassFilterCoefficientA1 = (0.95 * low_pass_amount)

    // DC normalized
    state->_lowPassFilterCoefficientB0 = SubtractQfQf(QF_CONST(1.0f), state->_lowPassFilterCoefficientA1);           // lowPassFilterCoefficientB0 = 1.0 - lowPassFilterCoefficientA1
}

void ApplyDelayEffect(const DelayParameter* pParameter, DelayState* state, bool enabled, const int32_t** ppInData,
    int32_t** ppOutData, int sampleCount)
{
    // for klocwork inspection check.
    NN_AUDIO_DSP_ASSERT((pParameter->_numChannels == 1) ||
                        (pParameter->_numChannels == 2) ||
                        (pParameter->_numChannels == 4) ||
                        (pParameter->_numChannels == 6));

    if (!enabled) //bypass
    {
        detail::ApplyDelayEffectBypass(ppInData, ppOutData, pParameter->_numChannels, sampleCount);
    }
    else
    {
        switch (pParameter->_numChannels)
        {
        case 1:
            detail::ApplyDelayEffect1ch(pParameter, state, ppInData[0],  ppOutData[0], sampleCount);
            break;
        case 2:
            detail::ApplyDelayEffect2ch(pParameter, state, ppInData, ppOutData, sampleCount);
            break;
        case 4:
            detail::ApplyDelayEffect4ch(pParameter, state, ppInData, ppOutData, sampleCount);
            break;
        case 6:
            detail::ApplyDelayEffect6ch(pParameter, state, ppInData, ppOutData, sampleCount);
            break;
        default:
            detail::ApplyDelayEffectBypass(ppInData, ppOutData, pParameter->_numChannels, sampleCount);
            break;
        }
    }
}

}}  // namespace nn::audio
