﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "audio_UpdateDataHeader.h"

namespace nn { namespace audio { namespace common {

namespace {

inline uint32_t EndianSwap(uint32_t val)
{
    return ((val & 0xff000000) >> 24) |
           ((val & 0x00ff0000) >> 8)  |
           ((val & 0x0000ff00) << 8)  |
           ((val & 0x000000ff) << 24);
}
const RevisionInfo MagicRevisionBase = common::CreateMagic("REV0");
const RevisionInfo MagicRevision1 = MagicRevisionBase + EndianSwap(1); // ~ SDK 0.17
const RevisionInfo MagicRevision2 = MagicRevisionBase + EndianSwap(2); // SDK 1.0 ~
const RevisionInfo MagicRevision3 = MagicRevisionBase + EndianSwap(3); // SDK 1.2 ~
const RevisionInfo MagicRevision4 = MagicRevisionBase + EndianSwap(4); // SDK 4.0 ~
const RevisionInfo MagicRevision5 = MagicRevisionBase + EndianSwap(5); // SDK 6.0 ~
const RevisionInfo MagicRevisionCurrent = MagicRevision5;

inline RevisionInfo GetRevision1() NN_NOEXCEPT
{
    return MagicRevision1;
}
inline RevisionInfo GetRevision2() NN_NOEXCEPT
{
    return MagicRevision2;
}
inline RevisionInfo GetRevision3() NN_NOEXCEPT
{
    return MagicRevision3;
}
inline RevisionInfo GetRevision4() NN_NOEXCEPT
{
    return MagicRevision4;
}

inline RevisionInfo GetRevision5() NN_NOEXCEPT
{
    return MagicRevision5;
}

inline bool CheckEqualOrNewerRevision(RevisionInfo a, RevisionInfo b) NN_NOEXCEPT
{
    auto revA = GetRevisionNumber(a);
    auto revB = GetRevisionNumber(b);
    NN_SDK_ASSERT_GREATER(revA, 0);
    NN_SDK_ASSERT_GREATER(revB, 0);
    if (revA < 0 || revB < 0)
    {
        return false;
    }
    return revA >= revB;
}

}

int GetRevisionNumber(RevisionInfo revision) NN_NOEXCEPT
{
    if ((MagicRevisionBase < revision) && (revision <= MagicRevisionCurrent))
    {
        return EndianSwap(static_cast<int>(revision - MagicRevisionBase));
    }
    return -1;
}

bool CheckValidRevision(const RevisionInfo rev) NN_NOEXCEPT
{
    return GetRevisionNumber(rev) > 0 &&  // Check if "rev" is valid magic variation.
           CheckEqualOrNewerRevision(GetCurrentRevision(), rev); // Check if Process is newer than user lib.
}

RevisionInfo GetCurrentRevision() NN_NOEXCEPT
{
    return MagicRevisionCurrent;
}

const BranchTag SupportTags::Latest = GetCurrentRevision;
const BranchTag SupportTags::VoicePlayedSampleCountResetAtLoopPoint = GetRevision5;
const BranchTag SupportTags::VoicePitchAndSrcSkipped = GetRevision5;
const BranchTag SupportTags::SplitterBugFix = GetRevision5;
const BranchTag SupportTags::FlushVoiceWaveBuffers = GetRevision5;
const BranchTag SupportTags::ElapsedFrameCount = GetRevision5;
const BranchTag SupportTags::AudioRendererProcessingTimeLimit80Percent = GetRevision5;
const BranchTag SupportTags::CommandProcessingTimeEstimatorVersion2 = GetRevision5;
const BranchTag SupportTags::AudioRendererVariadicCommandBufferSize = GetRevision5;
const BranchTag SupportTags::PerformanceMetricsDataFormatVersion2 = GetRevision5;
const BranchTag SupportTags::AudioRendererProcessingTimeLimit75Percent = GetRevision4;
const BranchTag SupportTags::AudioRendererProcessingTimeLimit70Percent = GetRevision1;
const BranchTag SupportTags::AudioUsbDeviceOutput = GetRevision4;
const BranchTag SupportTags::AdpcmLoopContextBugFix = GetRevision2;
const BranchTag SupportTags::Splitter = GetRevision2;
const BranchTag SupportTags::LongSizePreDelay = GetRevision3;
const BranchTag SupportTags::InitialRelease = GetRevision1;

RevisionInfo GetInitialReleaseRevision() NN_NOEXCEPT
{
    return GetRevision1();
}

bool CheckFeatureSupported(BranchTag tag, RevisionInfo rev) NN_NOEXCEPT
{
    return CheckEqualOrNewerRevision(rev, tag());
}

void UpdateDataHeader::Initialize(RevisionInfo revision) NN_NOEXCEPT
{
    m_Revision = revision;
    memset(&m_InfoSizes, 0, sizeof(UpdateDataSizes));
    m_TotalInfoSize = sizeof(UpdateDataHeader);
}

RevisionInfo UpdateDataHeader::GetRevision() const NN_NOEXCEPT
{
    return m_Revision;
}

size_t UpdateDataHeader::GetTotalSize() const NN_NOEXCEPT
{
    return m_TotalInfoSize;
}

void* UpdateDataHeader::GetCurrentPtr(uintptr_t base) const NN_NOEXCEPT
{
    return nn::util::BytePtr(reinterpret_cast<void*>(base), m_TotalInfoSize).Get();
}

}}} // namespace nn::audio::common

