﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <mutex>
#include <nn/os/os_Mutex.h>
#include "audio_ResourceExclusionChecker.h"

namespace nn { namespace audio { namespace detail {

std::atomic<bool> g_IsResourceExclusionCheckEnabled;

namespace {
nn::os::Mutex g_Mutex(false);
ResourceExclusionChecker g_ResourceExclusionCheckers[nn::audio::AudioRendererCountMax];
} // anonymous namespace

ResourceExclusionChecker* AcquireResourceExclusionChecker() NN_NOEXCEPT
{
    ResourceExclusionChecker* pChecker = nullptr;

    if(g_IsResourceExclusionCheckEnabled)
    {
        std::lock_guard<nn::os::Mutex> lock(g_Mutex);

        for(auto& checker : g_ResourceExclusionCheckers)
        {
            if(!checker.IsUsed())
            {
                checker.Reset();
                checker.SetUsed(true);
                pChecker = &checker;
                break;
            }
        }
    }

    return pChecker;
}

void ReleaseResourceExclusionChecker(ResourceExclusionChecker *pChecker) NN_NOEXCEPT
{
    if(g_IsResourceExclusionCheckEnabled)
    {
        if(pChecker)
        {
            std::lock_guard<nn::os::Mutex> lock(g_Mutex);
            pChecker->Reset();
        }
    }
}

ResourceExclusionChecker* FindResourceExclusionCheckerFromRegionInConfigImpl(void* p) NN_NOEXCEPT
{
    ResourceExclusionChecker* pChecker = nullptr;

    std::lock_guard<nn::os::Mutex> lock(g_Mutex);

    for(auto& checker : g_ResourceExclusionCheckers)
    {
        if(checker.IsUsed() && checker.IsInstanceIncludedInConfig(p))
        {
            pChecker = &checker;
            break;
        }
    }

    return pChecker;
}

ResourceExclusionChecker* FindResourceExclusionCheckerFromInternalHandleImpl(void* pInternalHandle) NN_NOEXCEPT
{
    ResourceExclusionChecker* pChecker = nullptr;

    std::lock_guard<nn::os::Mutex> lock(g_Mutex);

    for(auto& checker : g_ResourceExclusionCheckers)
    {
        if(checker.IsUsed() && checker.GetInternalHandle() == pInternalHandle)
        {
            pChecker = &checker;
            break;
        }
    }

    return pChecker;
}

}}} // nn::audio::detail
