﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "audio_MemoryPoolManager.h"
#include "audio_MemoryPoolInfo.h"
#include "audio_AddrTypes.h"
#include "audio_AudioRendererUtil.h"
#include "common/audio_WorkBufferAllocator.h"
#include "common/audio_MemoryPoolPolicy.h"

#include <mutex>
#include <nn/util/util_BytePtr.h>
#include <nn/audio/detail/audio_Log.h>

namespace nn { namespace audio {

size_t MemoryPoolManager::GetWorkBufferSize(const AudioRendererParameter& parameter) NN_NOEXCEPT
{
    auto infoCnt = common::GetMemoryPoolCount(detail::ConvertToAudioRendererParameterInternal(parameter));
    size_t rt = 0;
    rt += nn::util::align_up(sizeof(MemoryPoolInfo) * infoCnt, nn::audio::common::InfoTypeFieldAlignSize);
    rt += nn::util::align_up(sizeof(MemoryPoolManager), nn::audio::common::InfoTypeFieldAlignSize);
    rt += nn::util::align_up(InfoFlag::CalcBufferSize(infoCnt), nn::audio::common::InfoTypeFieldAlignSize);
    return rt;
}

MemoryPoolManager* MemoryPoolManager::CreateManager(common::WorkBufferAllocator& allocator, const AudioRendererParameter& parameter) NN_NOEXCEPT
{
    int infoCnt = common::GetMemoryPoolCount(detail::ConvertToAudioRendererParameterInternal(parameter));
    auto pInfos = common::PrepareArray<MemoryPoolInfo>(allocator.Allocate(sizeof(MemoryPoolInfo) * infoCnt), infoCnt);
    if (pInfos == nullptr)
    {
        return nullptr;
    }

    auto mgrBuf = allocator.Allocate(sizeof(MemoryPoolManager));
    auto bitArrBufSize = InfoFlag::CalcBufferSize(infoCnt);
    auto bitArrBuf = allocator.Allocate(InfoFlag::CalcBufferSize(infoCnt));
    if (mgrBuf == nullptr || bitArrBuf == nullptr)
    {
        return nullptr;
    }

    return new(mgrBuf) MemoryPoolManager(pInfos, infoCnt, bitArrBuf, bitArrBufSize);
}

MemoryPoolManager::MemoryPoolManager(
    MemoryPoolInfo* pInfos,
    int count,
    void* activeFlagBuf,
    size_t bitArrayBufSize) NN_NOEXCEPT
    : m_Mutex(true)
    , m_PoolCountMax(count)
    , m_FreePoolCount(count)
    , m_pInfos(pInfos)
{
    // NN_DETAIL_AUDIO_TRACE("MemoryPoolManager initialized | count:%d, releaseCount:%d\n", m_PoolCountMax, m_FreePoolCount);
    m_ActiveFlag.Initialize(activeFlagBuf, bitArrayBufSize);
}

int MemoryPoolManager::GetFreeCount() const NN_NOEXCEPT
{
    return m_FreePoolCount;
}

MemoryPoolInfo* MemoryPoolManager::Acquire(void* address, size_t size) NN_NOEXCEPT
{
    if (address == nullptr || size == 0)
    {
        return nullptr;
    }

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    MemoryPoolInfo* pInfo = nullptr;
    for (auto i = 0; i < m_PoolCountMax; ++i)
    {
        if (m_ActiveFlag[i] == false)
        {
            m_ActiveFlag.Set(i);
            pInfo = &m_pInfos[i];
            pInfo->Initialize(address, size);
            --m_FreePoolCount;
            NN_SDK_ASSERT(m_FreePoolCount >= 0);
            break;
        }
    }
    return pInfo;
}

void MemoryPoolManager::Release(MemoryPoolInfo* pPool) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    NN_SDK_ASSERT(pPool->IsUsed());
    pPool->Finalize();
    ++m_FreePoolCount;

    // To notify the "Released" transition to server side,
    // "m_ActiveFlag.Reset()" is deferred to next "UpdateOutStatus()" call.
}

size_t MemoryPoolManager::UpdateInParameter(void* pInParameter) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    size_t offset = 0;
    auto ptr = nn::util::BytePtr(pInParameter);
    for (auto i = 0; i < m_PoolCountMax; ++i)
    {
        if (m_ActiveFlag[i])
        {
            offset += m_pInfos[i].Prepare(reinterpret_cast<MemoryPoolInfo::InParameter*>((ptr + offset).Get()));
        }
        else
        {
            offset += MemoryPoolInfo::GetInParameterSize();
        }
    }
    return offset;
}

size_t MemoryPoolManager::UpdateOutStatus(void* pOutOutStatus) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    size_t offset = 0;
    auto ptr = nn::util::BytePtr(pOutOutStatus);
    for (auto i = 0; i < m_PoolCountMax; ++i)
    {
        if (m_ActiveFlag[i])
        {
            offset += m_pInfos[i].Store(reinterpret_cast<MemoryPoolInfo::OutStatus*>((ptr + offset).Get()));
            if (m_pInfos[i].GetState() == MemoryPoolInfo::State_Released)
            {
                m_ActiveFlag.Reset(i);
            }
        }
        else
        {
            offset += MemoryPoolInfo::GetOutStatusSize();
        }
    }
    return offset;
}

bool MemoryPoolManager::IsOverlapped(void* address, size_t size) const NN_NOEXCEPT // TODO: maybe too slow, need to optimize;
{
    for (auto i = 0; i < m_PoolCountMax; ++i)
    {
        if (m_pInfos[i].IsUsed() && m_pInfos[i].IsOverlapped(address, size))
        {
            return true;
        }
    }
    return false;
}

}} // namespace nn::audio
