﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "audio_MemoryPoolManager.h"
#include "audio_MemoryPoolInfo.h"
#include "audio_AddrTypes.h"

#include <nn/util/util_BytePtr.h>

namespace nn { namespace audio {

MemoryPoolInfo::MemoryPoolInfo() NN_NOEXCEPT
    : m_Address(nullptr)
    , m_Size(0)
    , m_State(State_Released)
    , m_InUse(false)
{
}

MemoryPoolInfo::~MemoryPoolInfo() NN_NOEXCEPT
{
    Finalize();
}

void MemoryPoolInfo::Initialize(void* address, size_t size) NN_NOEXCEPT
{
    m_Address = address;
    m_Size = size;
    m_State = State_Acquired;
    m_InUse = true;
    memset(_reserved, 0, sizeof(_reserved));
}

void MemoryPoolInfo::Finalize() NN_NOEXCEPT
{
    m_Address = nullptr;
    m_Size = 0;
    m_State = State_Released;
    m_InUse = false;
}

bool MemoryPoolInfo::IsUsed() const NN_NOEXCEPT
{
    return m_InUse;
}

bool MemoryPoolInfo::Contains(const void* address, size_t size) const NN_NOEXCEPT
{
    auto addr = reinterpret_cast<uintptr_t>(address);
    return (m_Address.GetAddress() <= addr) && ( (addr + size) <= (m_Address.GetAddress() + m_Size));
}

bool MemoryPoolInfo::IsOverlapped(const void* address, size_t size) const NN_NOEXCEPT
{
    auto addr = reinterpret_cast<uintptr_t>(address);
    return (m_Address.GetAddress() < addr + size) && (addr < m_Address.GetAddress() + m_Size);
}

bool MemoryPoolInfo::Attach() NN_NOEXCEPT
{
    return SetState(State_RequestAttach);
}

bool MemoryPoolInfo::Detach() NN_NOEXCEPT
{
    return SetState(State_RequestDetach);
}

bool MemoryPoolInfo::SetState(State nextState) NN_NOEXCEPT
{
    bool result = false;
    switch (m_State)
    {
    case State_Acquired:
        if (nextState == State_RequestAttach || nextState == State_Released)
        {
            result = true;
            m_State = nextState;
        }
        break;
    case State_RequestAttach:
        if (nextState == State_RequestDetach)
        {
            result = true;
            m_State = State_Detached;
        }
        else if (nextState == State_Attached)
        {
            result = true;
            m_State = nextState;
        }
        break;
    case State_Attached:
        if (nextState == State_RequestDetach)
        {
            result = true;
            m_State = nextState;
        }
        break;
    case State_RequestDetach:
        if (nextState == State_RequestAttach)
        {
            result = true;
            m_State = State_Attached;
        }
        else if (nextState == State_Detached)
        {
            result = true;
            m_State = nextState;
        }
        break;
    case State_Detached:
        if (nextState == State_RequestAttach || nextState == State_Released)
        {
            result = true;
            m_State = nextState;
        }
        break;
    case State_Released:
        if (nextState == State::State_Acquired)
        {
            result = true;
            m_State = nextState;
        }
        break;
    case State_Invalid:
    default:
        break;
    }
    return result;
}

bool MemoryPoolInfo::IsAttached() const NN_NOEXCEPT
{
    return (m_State == State_RequestAttach) ||
           (m_State == State_Attached) ||
           (m_State == State_RequestDetach);
}

bool MemoryPoolInfo::IsInTransition() const NN_NOEXCEPT
{
    return (m_State == State_RequestAttach) ||
           (m_State == State_RequestDetach);
}

void* MemoryPoolInfo::GetAddress() const NN_NOEXCEPT
{
    return m_Address.GetPointer();
}

size_t MemoryPoolInfo::GetSize() const NN_NOEXCEPT
{
    return static_cast<size_t>(m_Size);
}

MemoryPoolInfo::State MemoryPoolInfo::GetState() const NN_NOEXCEPT
{
    return m_State;
};

size_t MemoryPoolInfo::Prepare(InParameter* pInParameter) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pInParameter);
    pInParameter->_address = m_Address;
    pInParameter->_size = m_Size;
    pInParameter->_state = GetState();
    return sizeof(InParameter);
}

size_t MemoryPoolInfo::Store(const OutStatus* pOutStatus) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutStatus);
    SetState(pOutStatus->_state);
    return sizeof(OutStatus);
}

}} // namespace nn::audio
