﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <cstring>
#include <limits> //std::numeric_limits

#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_Cache.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_BitUtil.h> // nn::util::align_up
#include <nn/util/util_BytePtr.h> // nn::util::align_up
#include <nn/audio/audio_Result.h>
#include <nn/audio/audio_Common.h>

#include "audio_AuxTypes.h"
#include "audio_EffectManager.h"
#include "audio_EffectPrivateTypes.h"
#include "dsp/audio_EffectReverb.h"

namespace nn {
namespace audio {

NN_DEFINE_STATIC_CONSTANT(const int32_t EffectInfo::InParameter::InvalidProcessingOrder);

EffectInfo::EffectInfo()
{
    memset(&_inParameter, 0, sizeof(_inParameter));
    memset(&_outStatus, 0, sizeof(OutStatus));
}

EffectManager::EffectManager(int count, EffectInfo* info) NN_NOEXCEPT
    : m_InfoCount(count)
    , m_Infos(info)
    , m_Mutex(true)
{
    memset(info, 0, sizeof(EffectInfo) * count);
}

Result EffectManager::Add(EffectInfo** ppOutEffectInfo, EffectType type, void* buffer, size_t bufferSize, MixInfo* attachMixInfo) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    for (int i = 0; i < m_InfoCount; ++i)
    {
        if (m_Infos[i]._inParameter._type == EffectType_Invalid)
        {
            m_Infos[i]._inParameter._bufferBase = buffer;
            m_Infos[i]._inParameter._bufferSize = bufferSize;
            m_Infos[i]._inParameter._type = type;
            m_Infos[i]._inParameter._attachMixId = attachMixInfo->mixId;
            m_Infos[i]._inParameter._processingOrder = attachMixInfo->applyingEffectCount++;
            m_Infos[i]._inParameter._isNew = true;

            m_Infos[i]._outStatus._usageStatus = EffectInfo::OutStatus::UsageStatus_New;

            memset(m_Infos[i]._reserved, 0, sizeof(m_Infos[i]._reserved));

            // invalidate work buffer's cache
            if (buffer != nullptr)
            {
                memset(buffer, 0, bufferSize);
                nn::os::FlushDataCache(buffer, bufferSize);
            }

            *ppOutEffectInfo = &m_Infos[i];
            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(ResultOutOfResource());
}

bool EffectManager::IsInitialized(const EffectInfo* pEffectInfo) const NN_NOEXCEPT
{
    for (int i = 0; i < m_InfoCount; ++i)
    {
        if ( reinterpret_cast<uintptr_t>(&m_Infos[i]) == reinterpret_cast<uintptr_t>(pEffectInfo) &&
             m_Infos[i]._inParameter._type != EffectType_Invalid )
        {
            return true;
        }
    }
    return false;
}

void* EffectManager::Remove(EffectInfo* pEffectInfo, MixInfo* attachedMixInfo) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(attachedMixInfo->mixId, pEffectInfo->_inParameter._attachMixId);
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    // pop effect
    for (int i = 0; i < m_InfoCount; ++i)
    {
        if ((m_Infos[i]._inParameter._attachMixId == pEffectInfo->_inParameter._attachMixId) &&
            (m_Infos[i]._inParameter._processingOrder > pEffectInfo->_inParameter._processingOrder))
        {
            --m_Infos[i]._inParameter._processingOrder;
        }
    }

    // Update MixInfo (decrement effect count
    --attachedMixInfo->applyingEffectCount;

    // clean up self fields
    pEffectInfo->_inParameter._type = EffectType_Invalid;
    pEffectInfo->_inParameter._attachMixId = Invalid_MixId;
    pEffectInfo->_inParameter._bufferSize = 0;
    auto returnAddress = pEffectInfo->_inParameter._bufferBase;
    pEffectInfo->_inParameter._bufferBase = nullptr;
    return const_cast<void*>(returnAddress.GetPointer());
}

size_t EffectManager::UpdateEffectsInParameter(void* pInParameter) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    auto pOutParam = static_cast<EffectInfo::InParameter*>(pInParameter);
    for (int i = 0; i < m_InfoCount; ++i)
    {
        pOutParam[i] = m_Infos[i]._inParameter;
        switch(m_Infos[i]._inParameter._type)
        {
        case EffectType_Reverb:
            m_Infos[i]._inParameter._reverbParameter._parameterStatus = EffectParameterStatus_Updated;
            break;
        case EffectType_I3dl2Reverb:
            m_Infos[i]._inParameter._i3dl2ReverbParameter._parameterStatus = EffectParameterStatus_Updated;
            break;
        case EffectType_Delay:
            m_Infos[i]._inParameter._delayParameter._parameterStatus = EffectParameterStatus_Updated;
            break;
        case EffectType_BiquadFilter:
            m_Infos[i]._inParameter._biquadFilterParameter._parameterStatus = EffectParameterStatus_Updated;
            break;
        case EffectType_Aux:
        case EffectType_BufferMixer:
        default:
            break;
        }

        m_Infos[i]._inParameter._isNew = false;
    }

    return sizeof(EffectInfo::InParameter) * m_InfoCount;
}

size_t EffectManager::UpdateEffectOutStatus(EffectInfo::OutStatus* pOutStatus) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    for (int i = 0; i < m_InfoCount; ++i)
    {
        m_Infos[i]._outStatus._usageStatus = pOutStatus[i]._usageStatus;
    }

    return sizeof(EffectInfo::OutStatus) * m_InfoCount;
}

}  // namespace audio
}  // namespace nn

