﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/arp/detail/arp_IService.sfdl.h>
#include <nn/arp/detail/arp_InternalTypes.h>

#include <nn/arp/arp_Api.h>
#include <nn/arp/arp_Configs.h>
#include <nn/arp/arp_Types.h>
#include <nn/arp/arp_Result.h>

#include <nn/nn_SdkAssert.h>
#include <nn/sf/sf_ShimLibraryUtility.h>
#include <nn/sf/sf_Out.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/result/result_HandlingUtility.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/arp/arp_Resource.h>
#endif

namespace nn {
namespace arp {
namespace {

#if defined(NN_BUILD_CONFIG_OS_WIN)
DefaultServiceResource* g_pResource = nullptr;
std::aligned_storage<sizeof(DefaultServiceResource), std::alignment_of<DefaultServiceResource>::value>::type g_ResourceStorage;

#define NN_ARP_INITIALIZER_INITIALIZER { 0, NN_OS_MUTEX_INITIALIZER(false) }
struct Initializer
{
    uint32_t _initializationCount;
    mutable os::MutexType _mutex;
    void lock() const NN_NOEXCEPT
    {
        os::LockMutex(&_mutex);
    }
    void unlock() const NN_NOEXCEPT
    {
        os::UnlockMutex(&_mutex);
    }
    NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return _initializationCount > 0;
    }
    void Initialize() NN_NOEXCEPT
    {
        std::lock_guard<Initializer> lock(*this);
        NN_SDK_ASSERT(_initializationCount < std::numeric_limits<uint32_t>::max());

        if (_initializationCount <= 0)
        {
            g_pResource = new(&g_ResourceStorage) DefaultServiceResource;
        }
        ++ _initializationCount;
    }
    void Finalize() NN_NOEXCEPT
    {
        std::lock_guard<Initializer> lock(*this);
        NN_SDK_ASSERT(_initializationCount > 0);

        -- _initializationCount;
        if (_initializationCount <= 0)
        {
            g_pResource->~ServiceResource();
            g_pResource = nullptr;
        }
    }
} g_Initializer = NN_ARP_INITIALIZER_INITIALIZER;

void Finalizer(void*) NN_NOEXCEPT
{
    g_Initializer.Finalize();
}
#undef NN_ARP_INITIALIZER_INITIALIZER

#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
sf::SimpleAllInOneHipcSubDomainClientManager<8> g_ManagerForWriter = NN_SF_SIMPLE_ALL_IN_ONE_HIPC_SUB_DOMAIN_CLIENT_MANAGER_INITIALIZER;
sf::SimpleAllInOneHipcSubDomainClientManager<4> g_ManagerForReader = NN_SF_SIMPLE_ALL_IN_ONE_HIPC_SUB_DOMAIN_CLIENT_MANAGER_INITIALIZER;

#endif // ~NN_BUILD_CONFIG_OS

sf::ShimLibraryObjectHolder<detail::IWriter> g_pWriter = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;
sf::ShimLibraryObjectHolder<detail::IReader> g_pReader = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;

void InitializeWriterImplicitly() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_WIN)
    auto Initializer = [&](sf::SharedPointer<detail::IWriter>* p) -> Result
    {
        g_Initializer.Initialize();
        *p = g_pResource->GetWriterPtr();
        NN_RESULT_SUCCESS;
    };
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_pWriter.InitializeHolder<>(Initializer, Finalizer, nullptr));

#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ManagerForWriter.InitializeShimLibraryHolder(&g_pWriter, PortNameForArpWriter));
#endif // ~NN_BUILD_CONFIG_OS
}

void InitializeReaderImplicitly() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_WIN)
    auto Initializer = [&](sf::SharedPointer<detail::IReader>* p) -> Result
    {
        g_Initializer.Initialize();
        *p = g_pResource->GetReaderPtr();
        NN_RESULT_SUCCESS;
    };
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_pReader.InitializeHolder<>(Initializer, Finalizer, nullptr));

#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ManagerForReader.InitializeShimLibraryHolder(&g_pReader, PortNameForArpReader));
#endif // ~NN_BUILD_CONFIG_OS
}

} // ~namespace nn::arp::<anonymous>

// API ------------------------------------------------------------

Result AcquireRegistrar(Registrar* pRegistrar) NN_NOEXCEPT
{
    InitializeWriterImplicitly();

    sf::SharedPointer<detail::IRegistrar> p;
    NN_RESULT_DO(g_pWriter->AcquireRegistrar(&p));
    *pRegistrar = Registrar(p.Detach());
    NN_RESULT_SUCCESS;
}

void DeleteProperties(const os::ProcessId& pid) NN_NOEXCEPT
{
    InitializeWriterImplicitly();

    NN_ABORT_UNLESS_RESULT_SUCCESS(g_pWriter->DeleteProperties(pid));
}

Result GetApplicationLaunchProperty(ApplicationLaunchProperty* pOut, const os::ProcessId& pid) NN_NOEXCEPT
{
    InitializeReaderImplicitly();

    return g_pReader->GetApplicationLaunchProperty(pOut, pid);
}

Result GetApplicationLaunchProperty(ApplicationLaunchProperty* pOut, const ApplicationId& applicationId) NN_NOEXCEPT
{
    NN_UNUSED(pOut);
    NN_UNUSED(applicationId);

    InitializeReaderImplicitly();

    return g_pReader->GetApplicationLaunchPropertyWithApplicationId(pOut, applicationId);
}

Result GetApplicationControlProperty(ns::ApplicationControlProperty* pOut, const os::ProcessId& pid) NN_NOEXCEPT
{
    InitializeReaderImplicitly();

    return g_pReader->GetApplicationControlProperty(pOut, pid);
}

Result GetApplicationControlProperty(ns::ApplicationControlProperty* pOut, const ApplicationId& applicationId) NN_NOEXCEPT
{
    NN_UNUSED(pOut);
    NN_UNUSED(applicationId);

    InitializeReaderImplicitly();

    return g_pReader->GetApplicationControlPropertyWithApplicationId(pOut, applicationId);
}

// Registrar ------------------------------------------------------------

Registrar::Registrar(detail::IRegistrar* ptr) NN_NOEXCEPT
    : m_Ptr(ptr)
{
    NN_SDK_ASSERT_NOT_NULL(ptr);
}
Registrar::Registrar() NN_NOEXCEPT
    : m_Ptr(nullptr)
{
}
Registrar::Registrar(Registrar&& rhs) NN_NOEXCEPT
    : m_Ptr(rhs.m_Ptr)
{
    rhs.m_Ptr = nullptr;
}
Registrar::~Registrar() NN_NOEXCEPT
{
    if (m_Ptr != nullptr)
    {
        Release();
    }
}
Registrar& Registrar::operator=(Registrar&& rhs) NN_NOEXCEPT
{
    Registrar tmp(std::move(rhs));
    std::swap(this->m_Ptr, tmp.m_Ptr);
    return *this;
}
void Registrar::Release() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Ptr != nullptr);
    sf::ReleaseSharedObject(m_Ptr);
    m_Ptr = nullptr;
}
Result Registrar::SetApplicationLaunchProperty(const ApplicationLaunchProperty& launchProp) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Ptr != nullptr);
    return m_Ptr->SetApplicationLaunchProperty(launchProp);
}
Result Registrar::SetApplicationControlProperty(const ns::ApplicationControlProperty& ctrlProp) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Ptr != nullptr);
    return m_Ptr->SetApplicationControlProperty(ctrlProp);
}
Result Registrar::Issue(const os::ProcessId& pid) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Ptr != nullptr);
    return m_Ptr->Issue(pid);
}

}} // ~namespace nn::arp
