﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/applet/applet_Storage.h>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>

#include <utility>
#include <nn/am/am_Shim.h>
#include <nn/am/am_Result.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/sf/sf_Types.h>
#include <nn/sf/sf_NativeHandle.h>
#include "applet_StorageUtility.h"
#include <nn/applet/applet_Result.h>

namespace nn { namespace applet {

using detail::GetPointer;

namespace {

const size_t HandleStorageCountMax = 8;

struct HandleStorage
{
    void* _pStorageAccessor{nullptr};
    size_t _size;
    os::TransferMemoryType _trmem;
};

HandleStorage g_HandleStorageArray[HandleStorageCountMax];
os::Mutex g_HandleStorageMutex{false};

HandleStorage* FindHandleStorageUnsafe(void* pStorageAccessor) NN_NOEXCEPT
{
    for (int i=0; i<HandleStorageCountMax; ++i)
    {
        auto p = &g_HandleStorageArray[i];
        if (p->_pStorageAccessor == pStorageAccessor)
        {
            return p;
        }
    }
    return nullptr;
}

Result AllocateAndMapTransferStorageImpl(void** pOutAddress, void* pStorageAccessor, size_t size, sf::NativeHandle sfHandle) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pStorageAccessor);

    std::lock_guard<os::Mutex> lk(g_HandleStorageMutex);
    {
        auto p = FindHandleStorageUnsafe(pStorageAccessor);
        NN_RESULT_THROW_UNLESS(!p, applet::ResultAlreadyMappedTransferStorage());
    }
    auto p = FindHandleStorageUnsafe(nullptr);
    NN_RESULT_THROW_UNLESS(p, applet::ResultTooManyTransferStorageMapping());

    os::AttachTransferMemory(&p->_trmem, size, sfHandle.GetOsHandle(), sfHandle.IsManaged());
    p->_size = size;
    p->_pStorageAccessor = pStorageAccessor;
    sfHandle.Detach();
    return os::MapTransferMemory(pOutAddress, &p->_trmem, os::MemoryPermission_None);
}

void UnmapAndFreeTransferStorageImpl(void* pStorageAccessor) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pStorageAccessor);

    std::lock_guard<os::Mutex> lk(g_HandleStorageMutex);
    auto p = FindHandleStorageUnsafe(pStorageAccessor);
    if (p)
    {
        // 自動でアンマップされる
        os::DestroyTransferMemory(&p->_trmem);
        p->_pStorageAccessor = nullptr;
    }
}

}   // namespace

Result CreateStorage(StorageHandle* pOut, size_t size) NN_NOEXCEPT
{
    sf::SharedPointer<am::service::IStorage> p;
    NN_RESULT_DO(am::GetLibraryAppletCreator()->CreateStorage(&p, size));
    pOut->_p = p.Detach();
    NN_RESULT_SUCCESS;
}

Result CreateLargeStorage(StorageHandle* pOut, void* buffer, size_t size, bool isWritable) NN_NOEXCEPT
{
    os::TransferMemory transferMemory(const_cast<void*>(buffer), size, os::MemoryPermission_None);
    sf::SharedPointer<am::service::IStorage> p;
    NN_RESULT_DO(am::GetLibraryAppletCreator()->CreateTransferMemoryStorage(&p, sf::NativeHandle(transferMemory.Detach(), true), size, isWritable));
    pOut->_p = p.Detach();
    NN_RESULT_SUCCESS;
}

Result CreateTransferStorage(StorageHandle* pOut, void* buffer, size_t size) NN_NOEXCEPT
{
    os::TransferMemory transferMemory(const_cast<void*>(buffer), size, os::MemoryPermission_None);
    sf::SharedPointer<am::service::IStorage> p;
    NN_RESULT_DO(am::GetLibraryAppletCreator()->CreateHandleStorage(&p, sf::NativeHandle(transferMemory.Detach(), true), size));
    pOut->_p = p.Detach();
    NN_RESULT_SUCCESS;
}

void ReleaseStorage(StorageHandle handle) NN_NOEXCEPT
{
    sf::ReleaseSharedObject(GetPointer(handle));
}

size_t GetStorageSize(StorageHandle handle) NN_NOEXCEPT
{
    sf::SharedPointer<am::service::IStorageAccessor> pStorageAccessor;
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetPointer(handle)->Open(&pStorageAccessor));
    int64_t size;
    NN_ABORT_UNLESS_RESULT_SUCCESS(pStorageAccessor->GetSize(&size));
    return static_cast<size_t>(size);
}

Result WriteToStorage(StorageHandle handle, int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
{
    sf::SharedPointer<am::service::IStorageAccessor> pStorageAccessor;
    auto result = GetPointer(handle)->Open(&pStorageAccessor);
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH_CONVERT(am::ResultIllegalAccessToStorage, applet::ResultNotMemoryStorage());
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    NN_RESULT_END_TRY
    NN_ABORT_UNLESS_RESULT_SUCCESS(pStorageAccessor->Write(offset, sf::InBuffer(reinterpret_cast<const char*>(buffer), size)));
    NN_RESULT_SUCCESS;
}

Result ReadFromStorage(StorageHandle handle, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
{
    sf::SharedPointer<am::service::IStorageAccessor> pStorageAccessor;
    auto result = GetPointer(handle)->Open(&pStorageAccessor);
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH_CONVERT(am::ResultIllegalAccessToStorage, applet::ResultNotMemoryStorage());
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    NN_RESULT_END_TRY
    NN_ABORT_UNLESS_RESULT_SUCCESS(pStorageAccessor->Read(offset, sf::OutBuffer(reinterpret_cast<char*>(buffer), size)));
    NN_RESULT_SUCCESS;
}

Result MapHandleStorage(void** pOutAddress, size_t* pOutSize, StorageHandle storageHandle) NN_NOEXCEPT
{
    sf::SharedPointer<am::service::ITransferStorageAccessor> pStorageAccessor;
    auto result = GetPointer(storageHandle)->OpenTransferStorage(&pStorageAccessor);
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH_CONVERT(am::ResultIllegalAccessToStorage, applet::ResultNotTransferStorage());
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    NN_RESULT_END_TRY

    sf::NativeHandle sfHandle;
    uint64_t size64;
    NN_ABORT_UNLESS_RESULT_SUCCESS(pStorageAccessor->GetHandle(&sfHandle, &size64));
    NN_RESULT_DO(AllocateAndMapTransferStorageImpl(pOutAddress, pStorageAccessor.Get(), static_cast<size_t>(size64), std::move(sfHandle)));
    *pOutSize = static_cast<size_t>(size64);
    NN_RESULT_SUCCESS;
}

void UnmapHandleStorage(StorageHandle storageHandle) NN_NOEXCEPT
{
    sf::SharedPointer<am::service::ITransferStorageAccessor> pStorageAccessor;
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetPointer(storageHandle)->OpenTransferStorage(&pStorageAccessor));
    UnmapAndFreeTransferStorageImpl(pStorageAccessor.Get());
}

}}
