﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/os/os_Mutex.h>
#include <nn/am/am_Shim.h>
#include <nn/applet/applet_IdlePolicyApi.h>
#include <nn/applet/applet_Types.h>
#include <nn/applet/applet_Result.h>
#include <nn/idle/idle_SystemTypes.h>
#include <nn/hid/system/hid_InputDetection.h>

namespace nn { namespace applet {

namespace {
    os::Mutex g_AutoSleepDisabledSectionMutex{false};
    uint64_t  g_AutoSleepDisabledSectionCount{0};
}

//-----------------------------------------------------------------------------
//  無操作状態検出時の自動スリープ時間、自動低輝度化時間の上書き設定
//
void OverrideAutoSleepTimeAndDimmingTime(int autoSleepTimeInHandheld, int autoSleepTimeInConsole, int dimmingTimeInHandheld, int dimmingTimeInConsole) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(autoSleepTimeInHandheld >= 0 && autoSleepTimeInHandheld < idle::MaxOverrideAutoSleepTimeInSeconds);
    NN_SDK_REQUIRES(autoSleepTimeInConsole  >= 0 && autoSleepTimeInConsole  < idle::MaxOverrideAutoSleepTimeInSeconds);
    NN_SDK_REQUIRES(dimmingTimeInHandheld   >= 0 && dimmingTimeInHandheld   < idle::MaxOverrideDimmingTimeInSeconds);
    NN_SDK_REQUIRES(dimmingTimeInConsole    >= 0 && dimmingTimeInConsole    < idle::MaxOverrideDimmingTimeInSeconds);

    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->OverrideAutoSleepTimeAndDimmingTime(autoSleepTimeInHandheld, autoSleepTimeInConsole, dimmingTimeInHandheld, dimmingTimeInConsole));
}

//-----------------------------------------------------------------------------
//  メディア再生中か否かをシステムに通知
//
void SetMediaPlaybackState(bool isInMediaPlayback) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->SetMediaPlaybackState(isInMediaPlayback));
}

//-----------------------------------------------------------------------------
//  無操作状態検知における検知対象デバイスの指定
//
void SetInputDetectionSourceSet(Bit32 inputSources) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(inputSources != 0);
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->SetInputDetectionSourceSet(inputSources));
}

//-----------------------------------------------------------------------------
//  自動スリープの有効化・無効化
//
void BeginAutoSleepDisabledSection() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(g_AutoSleepDisabledSectionMutex);
    if (g_AutoSleepDisabledSectionCount == 0)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->SetAutoSleepDisabled(true));
    }
    ++g_AutoSleepDisabledSectionCount;
}

void EndAutoSleepDisabledSection() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(g_AutoSleepDisabledSectionMutex);
    if (g_AutoSleepDisabledSectionCount > 0)
    {
        --g_AutoSleepDisabledSectionCount;
        if (g_AutoSleepDisabledSectionCount == 0)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->SetAutoSleepDisabled(false));
        }
    }
}

bool IsAutoSleepDisabled() NN_NOEXCEPT
{
    bool isDisabled;
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->IsAutoSleepDisabled(&isDisabled));
    return isDisabled;
}

//-----------------------------------------------------------------------------

}}  // namespace nn::applet

