﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include "apm_FirmwareDebugSettingsAccessor.h"

namespace nn { namespace apm { namespace server {

class SettingsHolder
{
public:
    enum class PerformanceModePolicy
    {
        Auto = 0,
        Normal,
        Boost,
    };

private:
    //! デフォルトの Throttling の追加時間。
    static const int DefaultSdevThrottlingAdditionalDelayUs = 16000;

public:
    SettingsHolder() NN_NOEXCEPT;

    //! settings からの設定の取得。
    void LoadSettings() NN_NOEXCEPT;

    //! settings からの動的に変更可能な設定の再取得。
    void ReloadSettings() NN_NOEXCEPT;

    //! 性能モードポリシーの取得。
    PerformanceModePolicy GetPerformanceModePolicy() NN_NOEXCEPT;

    //! SDEV での Throttling の有効フラグの取得。
    bool IsSdevThrottlingEnabled() NN_NOEXCEPT;

    //! SDEV での Throttling の追加時間の取得。
    int GetSdevThrottlingAdditionalDelayUs() NN_NOEXCEPT;

    //! 電力消費量の大きい性能の使用可能フラグの取得。
    bool IsBatteryDrainingEnabled() NN_NOEXCEPT;

    //! SDEV で CPU クロックレートを上昇させるフラグの取得。
    bool IsSdevCpuOverclockEnabled() NN_NOEXCEPT;

    //! Undock 時にスロットリングを有効にするフラグの取得。
    bool IsThrottlingForUndockEnabled() NN_NOEXCEPT;

    //! Sudden Momentary Power Drop 時にスロットリングを有効にするフラグの取得。
    bool IsThrottlingForSmpdEnabled() NN_NOEXCEPT;

    //! Enough Power Chage Emulation の要不要の取得。
    bool IsEnoughPowerChargeEmulationRequired() NN_NOEXCEPT;

private:
    NN_DISALLOW_COPY(SettingsHolder);
    NN_DISALLOW_MOVE(SettingsHolder);

    //! Board Type が SDEV であるか否かの確認。
    void CheckBoardType() NN_NOEXCEPT;

private:
    //! 性能モードポリシー。
    PerformanceModePolicy m_PerformanceModePolicy;

    //! SDEV での Throttling の有効フラグ。
    bool m_SdevThrottlingEnabled;

    //! SDEV での Throttling の追加時間。
    int m_SdevThrottlingAdditionalDelayUs;

    //! 電力消費量の大きい性能の使用可能フラグ。
    bool m_BatteryDrainingEnabled;

    //! SDEV で CPU クロックレートを上昇させるフラグ。
    bool m_SdevCpuOverclockEnabled;

    //! Undock 時にスロットリングを有効にするフラグ。
    bool m_ThrottlingForUndockEnabled;

    //! Sudden Momentary Power Drop 時にスロットリングを有効にするフラグ。
    bool m_ThrottlingForSmpdEnabled;

    //! Board Type が SDEV であることを表すフラグ。
    bool m_BoardTypeIsSdev;

    //! Firmware Debug Settings にアクセスするためのオブジェクト。
    FirmwareDebugSettingsAccessor m_FirmwareDebugSettingsAccessor;
};

}}} // namespace nn::apm::server
