﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nn;
using nn.sf;
using std;
using Nintendo.ServiceFramework;
using Nintendo.ServiceFramework.CppCode;
using Nintendo.ServiceFramework.Hipc;

namespace nn.am.service
{
    /// <summary>
    /// アプレットプロセスの起動理由
    /// </summary>
    /// <remarks>
    /// 以下がある。
    /// - Normal
    /// - Unwind
    /// </remarks>
    [CppRefPath(Location.FunctionsSfdlPath)]
    [ExternalStruct(4, 4)]
    public struct AppletProcessLaunchReason
    {
    }

    [CppRefPath(Location.FoundationSfdlPath)]
    public interface IStorageAccessor : IServiceObject
    {
        [MethodId(0)]
        Result GetSize(Out<int64_t> pOut);

        [MethodId(10)]
        Result Write(int64_t offset, [AutoSelectBuffer] InBuffer buffer);

        [MethodId(11)]
        Result Read(int64_t offset, [AutoSelectBuffer] OutBuffer buffer);
    }

    [CppRefPath(Location.FoundationSfdlPath)]
    public interface ITransferStorageAccessor : IServiceObject
    {
        [MethodId(0)]
        Result GetSize(Out<int64_t> pOut);

        /// <summary>
        /// 転送ストレージのハンドルを取得する
        /// </summary>
        /// <param name="pOutHandle"></param>
        /// <param name="pOutSize"></param>
        /// <returns></returns>
        [MethodId(1)]
        Result GetHandle(Out<NativeHandle> pOutHandle, Out<uint64_t> pOutSize);
    }

    /// <summary>
    /// データ受け渡し用ストレージ
    /// </summary>
    /// <remarks>
    /// 実際の操作を行うためには、Open メソッドで IStorageAccessor を取得する必要がある。
    /// 各 Push 系メソッドの引数とするためには IStorageAccessor を閉じる必要がある。
    /// </remarks>
    [CppRefPath(Location.FoundationSfdlPath)]
    public interface IStorage : IServiceObject
    {
        /// <summary>
        /// IStorageAccessor を取得する
        /// </summary>
        /// <param name="pOut"></param>
        /// <param name="processId"></param>
        /// <returns></returns>
        /// <remarks>
        /// このメソッドは、このオブジェクトを Pop 系関数に渡すと、操作権が消失し、以後使用できなくなる。
        /// </remarks>
        [MethodId(0)]
        Result Open(Out<IStorageAccessor> pOut);

        /// <summary>
        /// ITransferStorageAccessor を取得する
        /// </summary>
        /// <param name="pOut"></param>
        /// <returns></returns>
        /// <remarks>
        /// このメソッドは、このオブジェクトを Pop 系関数に渡すと、操作権が消失し、以後使用できなくなる。
        /// </remarks>
        [MethodId(1)]
        Result OpenTransferStorage(Out<ITransferStorageAccessor> pOut);

        /// <summary>
        /// このオブジェクトからの Open を不可にし、Open 可能なオブジェクトを生成する
        /// (MethodId 未定義のため、プロキシ経由でのアクセス不可)
        /// </summary>
        /// <param name="pOut"></param>
        /// <returns></returns>
        [MethodId(100)]
        [UseCmif(false)]
        Result GetAndInvalidate(Out<IStorage> pOut);
    }

    [CppRefPath(Location.FunctionsSfdlPath)]
    public interface ILibraryAppletCreator : IServiceObject
    {
        /// <summary>
        /// ライブラリアプレットを作成する
        /// </summary>
        /// <param name="pOut"></param>
        /// <param name="appletId"></param>
        /// <param name="hasWindow"></param>
        /// <returns></returns>
        [MethodId(0)]
        Result CreateLibraryApplet(Out<ILibraryAppletAccessor> pOut, Bit32 appletId, uint32_t libraryAppletMode);

        /// <summary>
        /// 自アプレット起点以降の全ての LA を強制的に終了します。
        /// Terminate 要求だけ出してすぐに返ります。
        /// </summary>
        /// <returns>
        /// </returns>
        /// <remarks>
        /// </remarks>
        [MethodId(1)]
        Result TerminateAllLibraryApplets();

        /// <summary>
        /// 自アプレット起点の LA が残存しているか否かを返します。
        /// </summary>
        /// <returns>
        /// </returns>
        /// <remarks>
        /// </remarks>
        [MethodId(2)]
        Result AreAnyLibraryAppletsLeft(Out<bool> isLeft);

        /// <summary>
        /// 指定サイズでコンテキスト領域を作成する
        /// </summary>
        /// <param name="pOut"></param>
        /// <param name="size"></param>
        /// <returns></returns>
        [MethodId(10)]
        Result CreateStorage(Out<IStorage> pOut, int64_t size);

        /// <summary>
        /// TransferMemory を使用してストレージを作成する（大容量ストレージ）
        /// </summary>
        /// <param name="pOut"></param>
        /// <param name="transferMemory"></param>
        /// <param name="size"></param>
        /// <returns></returns>
        /// <remarks>作成時の transferMemory の内容は保持される</remarks>
        [MethodId(11)]
        Result CreateTransferMemoryStorage(Out<IStorage> pOut, NativeHandle transferMemory, int64_t size, bool isWritable);

        /// <summary>
        /// TransferMemory を使用してストレージを作成する（ハンドルストレージ）
        /// </summary>
        /// <param name="pOut"></param>
        /// <param name="transferMemory"></param>
        /// <param name="size"></param>
        /// <returns></returns>
        /// <remarks>作成時の transferMemory の内容は保持される</remarks>
        [MethodId(12)]
        Result CreateHandleStorage(Out<IStorage> pOut, NativeHandle transferMemory, int64_t size);
    }

    /// <summary>
    /// 自身の winding に関する操作
    /// </summary>
    [CppRefPath(Location.FunctionsSfdlPath)]
    public interface IProcessWindingController : IServiceObject
    {
        /// <summary>
        /// プロセスが起動された理由を取得する
        /// </summary>
        /// <param name="pOut"></param>
        /// <returns></returns>
        [MethodId(0)]
        Result GetLaunchReason(Out<AppletProcessLaunchReason> pOut);

        /// <summary>
        /// unwind 時に、呼び出していた LA オブジェクトを取得する
        /// </summary>
        /// <returns></returns>
        /// <remarks>
        /// - [post] GetLaunchReason == RelaunchFromCaller
        /// - [pre] pOut は WaitingForPop
        ///
        /// スクラッチパッド起動時には、このメソッドで ILibraryAppletAccessor を取得し、
        /// - LoadPreviousContext
        /// - Pop
        /// - GetOutParameters
        /// の順で呼ぶ想定。
        /// </remarks>
        [MethodId(11)]
        Result OpenCallingLibraryApplet(Out<ILibraryAppletAccessor> pOut);

        /// <summary>
        /// コンテキストをセットする
        /// </summary>
        /// <param name="storage"></param>
        /// <returns></returns>
        /// <remarks>
        /// 指定した storage は無効となる
        /// </remarks>
        [MethodId(21)]
        Result PushContext(IStorage storage);

        /// <summary>
        /// コンテキストを取得する
        /// </summary>
        /// <param name="pOut"></param>
        /// <returns></returns>
        /// <remarks>
        /// コンテキストは除去される
        /// </remarks>
        [MethodId(22)]
        Result PopContext(Out<IStorage> pOut);

        /// <summary>
        /// winding 時の処理予約をキャンセルする。
        /// </summary>
        /// <returns></returns>
        [MethodId(23)]
        Result CancelWindingReservation();

        /// <summary>
        /// 自プロセスを wind し、予約されていた処理を行う。
        /// </summary>
        /// <returns></returns>
        /// <remarks>
        /// 何も予約されていない場合にはエラー。
        /// </remarks>
        [MethodId(30)]
        Result WindAndDoReserved();

        /// <summary>
        /// 「現プロセスが wind したら、LA を起動し終了を待ち、再度このプロセスを unwind する」ことを予約する。
        /// </summary>
        /// <param name="libraryApplet"></param>
        /// <returns></returns>
        /// <remarks>
        /// 現プロセスが wind した際に、以下を行うように、対応するプロキシに予約する。
        /// - LA を OpenCallingLibraryApplet で取得できるようにする
        /// - LA を開始する
        /// - LA の終了を待機する
        /// - このプロセスを unwind する
        /// </remarks>
        [MethodId(40)]
        Result ReserveToStartAndWaitAndUnwindThis(ILibraryAppletAccessor libraryApplet);

        /// <summary>
        /// 「現プロセスが wind したら、LA を起動し終了を待つ」ことを予約する。
        /// </summary>
        /// <param name="libraryApplet"></param>
        /// <returns></returns>
        /// <remarks>
        /// 現プロセスが wind した際に、以下を行うように、対応するプロキシに予約する。
        /// - LA を OpenCallingLibraryApplet で取得できるようにする
        /// - LA を開始する
        /// - LA の終了を待機する
        /// LA 終了後に現プロセスは Unwind せずに呼出元に戻る。
        /// </remarks>
        [MethodId(41)]
        Result ReserveToStartAndWait(ILibraryAppletAccessor libraryApplet);
    }

    /// <summary>
    /// ライブラリアプレットの操作クラス
    /// </summary>
    public interface ILibraryAppletAccessor : IAppletAccessor
    {
        /// <summary>
        /// メモリ領域をライブラリアプレットに転送する
        /// </summary>
        /// <param name="storage"></param>
        /// <returns></returns>
        [MethodId(100)]
        Result PushInData(IStorage storage);

        /// <summary>
        /// メモリ領域をライブラリアプレットから取得する
        /// </summary>
        /// <param name="storage"></param>
        /// <returns></returns>
        [MethodId(101)]
        Result PopOutData(Out<IStorage> pOut);

        /// <summary>
        /// ライブラリアプレットが使用できるストレージをセットする
        /// </summary>
        /// <param name="storage"></param>
        /// <returns>大量の返り値を受け渡すのに使用するバッファを caller で確保して渡すことを意図している</returns>
        [MethodId(102)]
        Result PushExtraStorage(IStorage storage);

        /// <summary>
        /// メモリ領域をライブラリアプレットに転送する
        /// </summary>
        /// <param name="storage"></param>
        /// <returns></returns>
        [MethodId(103)]
        Result PushInteractiveInData(IStorage storage);

        /// <summary>
        /// メモリ領域をライブラリアプレットに転送する
        /// </summary>
        /// <param name="storage"></param>
        /// <returns></returns>
        [MethodId(104)]
        Result PopInteractiveOutData(Out<IStorage> storage);

        [MethodId(105)]
        Result GetPopOutDataEvent(Out<NativeHandle> pOut);

        [MethodId(106)]
        Result GetPopInteractiveOutDataEvent(Out<NativeHandle> pOut);

        /// <summary>
        /// この LA を起動するために、自プロセスを終了する必要があるかどうか
        /// </summary>
        /// <param name="pOut"></param>
        /// <returns></returns>
        [MethodId(110)]
        Result NeedsToExitProcess(Out<bool> pOut);

        [MethodId(120)]
        Result GetLibraryAppletInfo(Out<LibraryAppletInfo> pOut);

        /// <summary>
        /// 末端にある LA が再び FG となるように要求を出す
        /// </summary>
        /// <returns>
        /// </returns>
        /// <remarks>
        /// </remarks>
        [MethodId(150)]
        Result RequestForAppletToGetForeground();

        [MethodId(160)]
        Result GetIndirectLayerConsumerHandle(Out<vi.IndirectConsumerHandleType> pOut, CheckedAppletResourceUserId aruid);

        /// <summary>
        /// OutOfFocus 状態のアプリケーションをサスペンドするか否かを指定
        /// </summary>
        /// <param name="isEnabled">OutOfFocus 状態アプリの自動中断の有無</param>
        /// <remarks>
        /// 自ライブラリアプレットの Window より下位にあるアプリが
        /// OutOfFocus 状態の時にそのアプリをサスペンドするか否かを指定する。
        /// </remarks>
        [MethodId(50)]
        Result SetOutOfFocusApplicationSuspendingEnabled(bool isEnabled);
    }
}
