﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/process/am_NsProcess.h>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/am/service/am_ServiceStaticAllocator.h>
#include <nn/am/service/am_ServiceDiagnostics.h>
#include <nn/am/service/am_StuckChecker.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Thread.h>
#include <mutex>
#include <algorithm>
#include <utility>

#include <nn/arp/arp_Api.h>
#include <nn/arp/arp_Result.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_ApplicationControlDataSystemApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/am/service/am_NsWrapper.h>

namespace nn { namespace am { namespace service { namespace process {

class NsProcessManager::ProcessBase
    : public NsProcess
    , public util::IntrusiveListBaseNode<ProcessBase>
{
    friend NsProcessManager;
protected:

    explicit ProcessBase(NsProcessManager* pManager, os::ProcessId processId) NN_NOEXCEPT
        : m_pManager(pManager)
        , m_ProcessId(processId)
    {
        NN_SDK_ASSERT(m_ProcessId != os::ProcessId::GetInvalidId());
        NN_UNUSED(m_pManager);
    }

    virtual ~ProcessBase() NN_NOEXCEPT
    {
        // join されていることの assert
        NN_SDK_ASSERT(m_ProcessId == os::ProcessId::GetInvalidId());
    }

public:

    virtual os::ProcessId GetProcessId() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_ProcessId;
    }

    virtual void Join() NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_ProcessId != os::ProcessId::GetInvalidId())
        {
            m_CompletedEvent.Wait();
            this->m_ProcessId = os::ProcessId::GetInvalidId();
        }
    }

    virtual void InitializeMultiWaitHolder(os::MultiWaitHolderType* pHolder) NN_NOEXCEPT NN_OVERRIDE
    {
        os::InitializeMultiWaitHolder(pHolder, m_CompletedEvent.GetBase());
    }

private:

    NsProcessManager* const m_pManager;
    os::ProcessId m_ProcessId;
    os::Event m_CompletedEvent{os::EventClearMode_ManualClear};

};

void NsProcessManager::Add(ProcessBase* p) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    m_List.push_back(*p);
}

void NsProcessManager::NotifyExit(os::ProcessId processId) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    auto it = std::find_if(m_List.begin(), m_List.end(), [processId] (decltype(m_List.front())& e)
    {
        return e.GetProcessId() == processId;
    });
    if (it == m_List.end())
    {
        // TODO: ありえない？
        return;
    }
    it->m_CompletedEvent.Signal();
    m_List.erase(it);
}

class NsProcessManager::LibraryAppletProcess
    : public ProcessBase
{
public:

    LibraryAppletProcess(NsProcessManager* pManager, os::ProcessId processId) NN_NOEXCEPT
        : ProcessBase(pManager, processId)
    {
    }

    virtual void Terminate() NN_NOEXCEPT NN_OVERRIDE
    {
        auto result = NN_AM_SERVICE_STUCK_CHECKED(ns_TerminateLibraryApplet, 60, ns::TerminateLibraryApplet(GetProcessId()));
        NN_UNUSED(result); // 要ハンドリング
    }

};

Result NsProcessManager::MakeLibraryAppletProcess(std::shared_ptr<NsProcess>* pOut, ncm::SystemProgramId id) NN_NOEXCEPT
{
    os::ProcessId processId;
    NN_RESULT_DO(NN_AM_SERVICE_STUCK_CHECKED(ns_LaunchLibraryApplet, 60, ns::LaunchLibraryApplet(&processId, id)));
    auto p = MakeShared<LibraryAppletProcess>(this, processId);
    Add(p.get());
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

Result NsProcessManager::MakeSystemAppletProcess(std::shared_ptr<NsProcess>* pOut) NN_NOEXCEPT
{
    os::ProcessId processId;
    NN_RESULT_DO(NN_AM_SERVICE_STUCK_CHECKED(ns_LaunchSystemApplet, 60, ns::LaunchSystemApplet(&processId)));
    auto p = MakeShared<LibraryAppletProcess>(this, processId);
    Add(p.get());
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

Result NsProcessManager::MakeOverlayAppletProcess(std::shared_ptr<NsProcess>* pOut) NN_NOEXCEPT
{
    os::ProcessId processId;
    NN_RESULT_DO(NN_AM_SERVICE_STUCK_CHECKED(ns_LaunchOverlayApplet, 60, ns::LaunchOverlayApplet(&processId)));
    auto p = MakeShared<LibraryAppletProcess>(this, processId);
    Add(p.get());
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

std::shared_ptr<NsProcess> NsProcessManager::MakeLibraryAppletProcess(os::ProcessId processId) NN_NOEXCEPT
{
    auto p = MakeShared<LibraryAppletProcess>(this, processId);
    Add(p.get());
    return p;
}

class NsProcessManager::ApplicationProcess
    : public ProcessBase
{
public:

    ApplicationProcess(NsProcessManager* pManager, os::ProcessId processId) NN_NOEXCEPT
        : ProcessBase(pManager, processId)
    {
    }

    virtual void Terminate() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_AM_SERVICE_LOG(seq, "Invoke ns::TerminateApplication(pid=%lld)\n", GetProcessId());
        auto result = NN_AM_SERVICE_STUCK_CHECKED(ns_TerminateApplication, 60, ns::TerminateApplication(GetProcessId()));
        NN_UNUSED(result); // 要ハンドリング
    }

};

Result NsProcessManager::GetApplicationMainProgramIndex(uint8_t* pOut, const ns::ApplicationLaunchInfo& launchInfo) NN_NOEXCEPT
{
    return NN_AM_SERVICE_STUCK_CHECKED(ns_GetApplicationMainProgramIndex, 60, ns::GetMainApplicationProgramIndex(pOut, launchInfo));
}

Result NsProcessManager::MakeApplicationProcess(std::shared_ptr<NsProcess>* pOut, ncm::ApplicationId id, uint8_t programIndex) NN_NOEXCEPT
{
    WaitForNotBoostSystemMemoryResourceLimit();
    os::ProcessId processId;
    NN_RESULT_DO(NN_AM_SERVICE_STUCK_CHECKED(ns_LaunchApplication, 60, ns::LaunchApplication(&processId, id, programIndex)));
    auto p = MakeShared<ApplicationProcess>(this, processId);
    Add(p.get());
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

std::shared_ptr<NsProcess> NsProcessManager::MakeApplicationProcess(os::ProcessId processId) NN_NOEXCEPT
{
    auto p = MakeShared<ApplicationProcess>(this, processId);
    Add(p.get());
    return p;
}

Result NsProcessManager::GetApplicationLaunchInfo(ns::ApplicationLaunchInfo* pOut, ncm::ApplicationId id) NN_NOEXCEPT
{
    return NN_AM_SERVICE_STUCK_CHECKED(ns_GetApplicationLaunchInfo, 60, ns::GetApplicationLaunchInfo(pOut, id));
}

Result NsProcessManager::AcquireApplicationLaunchInfoForDevelop(ns::ApplicationLaunchInfo* pOut, os::ProcessId processId) NN_NOEXCEPT
{
    return NN_AM_SERVICE_STUCK_CHECKED(ns_GetAcquireApplicationLaunchInfo, 60, ns::AcquireApplicationLaunchInfo(pOut, processId));
}

Result NsProcessManager::GetApplicationControlProperty(ns::ApplicationControlProperty* pOut, const ns::ApplicationLaunchInfo& launchInfo, uint8_t programIndex) NN_NOEXCEPT
{
    return NN_AM_SERVICE_STUCK_CHECKED(ns_GetApplicationControlProperty, 60, ns::GetApplicationControlProperty(pOut, launchInfo, programIndex));
}

Result NsProcessManager::GetApplicationIdByProcessId(ncm::ApplicationId* pOut, os::ProcessId processId) NN_NOEXCEPT
{
    int retryCount = 0;
retry:
    arp::ApplicationLaunchProperty launchProperty;
    NN_RESULT_TRY(arp::GetApplicationLaunchProperty(&launchProperty, processId))
        NN_RESULT_CATCH(arp::ResultNotRegistered)
        {
            ++retryCount;
            if (retryCount == 1000)
            {
                NN_RESULT_RETHROW;
            }
            os::SleepThread(TimeSpan::FromMilliSeconds(10));
            goto retry;
        }
    NN_RESULT_END_TRY
    *pOut = launchProperty.id;
    NN_RESULT_SUCCESS;
}

os::ProcessId AttachedNsProcess::GetProcessId() const NN_NOEXCEPT
{
    return os::ProcessId::GetInvalidId();
}

void AttachedNsProcess::Terminate() NN_NOEXCEPT
{
    // 運用上アプリしかないはずなので、アプリと仮定している
    auto result = NN_AM_SERVICE_STUCK_CHECKED(ns_TerminateApplication, 60, (ns::TerminateApplication(GetProcessId())));
    NN_UNUSED(result);  // 要ハンドリング
}

namespace {

os::Event& GetNeverSignaledEvent() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(os::Event, g_Event, {os::EventClearMode_ManualClear});
    return g_Event;
}

}

void AttachedNsProcess::Join() NN_NOEXCEPT
{
    // infinite sleep
    GetNeverSignaledEvent().Wait();
}

void AttachedNsProcess::InitializeMultiWaitHolder(os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
{
    os::InitializeMultiWaitHolder(pHolder, GetNeverSignaledEvent().GetBase());
}

}}}}
