﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/display/am_ExtraSharedLayerManager.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/am/am_Result.h>
#include <nn/am/service/display/am_DisplayControlLog.h>
#include <nn/am/service/display/am_DisplayPrimitiveOperation.h>
#include <nn/am/service/display/am_IntegratedDisplayManager.h>
#include <nn/oe/oe_SelfControlApis.h>

namespace nn{ namespace am{ namespace service{ namespace display{ namespace detail{

    void ExtraSharedLayerManager::Initialize(const ExtraSharedLayerResourceAccessor& resAccessor) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(sizeof(m_ResourceAccessorStorage) >= sizeof(ExtraSharedLayerResourceAccessor));
        NN_STATIC_ASSERT(NN_ALIGNOF(m_ResourceAccessorStorage) >= NN_ALIGNOF(ExtraSharedLayerResourceAccessor));
        m_pResource = new(m_ResourceAccessorStorage) ExtraSharedLayerResourceAccessor(resAccessor);
    }

    void ExtraSharedLayerManager::Finalize() NN_NOEXCEPT
    {
        Deactivate();
        DestroyLayer();

        m_pResource->~ExtraSharedLayerResourceAccessor();
        m_pResource = nullptr;
        std::memset(m_ResourceAccessorStorage, 0, sizeof(m_ResourceAccessorStorage));
    }

    void ExtraSharedLayerManager::Activate() NN_NOEXCEPT
    {
        // CreateLayer していなくても呼べる
        NN_SDK_REQUIRES(IsInitialized());
        m_IsActive = true;
        UpdateVisibilityImpl();
    }

    void ExtraSharedLayerManager::Deactivate() NN_NOEXCEPT
    {
        // CreateLayer していなくても呼べる
        NN_SDK_REQUIRES(IsInitialized());
        m_IsActive = false;
        UpdateVisibilityImpl();
    }

    void ExtraSharedLayerManager::RequestShow(int index) NN_NOEXCEPT
    {
        // CreateLayer していなくても呼べる
        NN_SDK_REQUIRES(IsInitialized());
        m_RequestedTextureIndex = index;
        UpdateVisibilityImpl();
    }

    void ExtraSharedLayerManager::RequestHide() NN_NOEXCEPT
    {
        // CreateLayer していなくても呼べる
        NN_SDK_REQUIRES(IsInitialized());
        m_RequestedTextureIndex = -1;
        UpdateVisibilityImpl();
    }


    nn::Result ExtraSharedLayerManager::CreateLayer(
        nn::sf::NativeHandle&& hTransferMemory,
        size_t transferMemorySize,
        const nn::vi::fbshare::SharedMemoryPoolLayout& layout,
        int zOrder,
        nn::vi::LayerStackFlagType stacks
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_RESULT_THROW_UNLESS(transferMemorySize % (256u * 1024) == 0, am::ResultInvalidParameter());
        NN_RESULT_THROW_UNLESS(!m_IsLayerCreated, am::ResultInvalidCall());

        bool isSuccess = false;

        // 共有バッファを作成
        nn::vi::fbshare::SharedBufferHandle hBuffer = {};
        NN_RESULT_DO(DisplayPrimitiveOperation::CreateSharedBufferOnTransferMemory(&hBuffer, std::move(hTransferMemory), transferMemorySize, layout));
        NN_UTIL_SCOPE_EXIT{
            if(!isSuccess){ DisplayPrimitiveOperation::DestroySharedBuffer(hBuffer); }
        };

        // 出画用レイヤを作成
        NN_RESULT_DO(m_LowLayer.Initialize(hBuffer, false, zOrder));
        NN_UTIL_SCOPE_EXIT{
            if(!isSuccess){ m_LowLayer.Finalize(); }
        };

        isSuccess       = true;

        m_Layout       = layout;
        m_BufferHandle = hBuffer;
        m_LayerStacks  = stacks;
        std::memset(m_ImageRegionList, 0, sizeof(m_ImageRegionList));

        m_IsLayerCreated = true;
        NN_RESULT_SUCCESS;
    }

    void ExtraSharedLayerManager::DestroyLayer() NN_NOEXCEPT
    {
        if(m_IsLayerCreated)
        {
            ForceHideImpl();
            NN_SDK_ASSERT(m_PresentedTextureIndex < 0);

            m_LowLayer.Finalize();
            DisplayPrimitiveOperation::DestroySharedBuffer(m_BufferHandle);

            m_Layout       = {};
            m_BufferHandle = {};
            m_LayerStacks  = {};
            std::memset(m_ImageRegionList, 0, sizeof(m_ImageRegionList));
        }

        // 表示要求をクリア
        m_RequestedTextureIndex = -1;

        m_IsLayerCreated = false;
    }

    nn::Result ExtraSharedLayerManager::SetTexture(int index, int x, int y, int width, int height, const void* buffer, size_t size, int originMode) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(IsLayerCreated());
        NN_RESULT_THROW_UNLESS(index >= 0 && index < m_Layout.count, nn::am::ResultInvalidCall());

        nn::vi::ImageTransform dstTransform = (originMode == oe::WindowOriginMode_LowerLeft) ? vi::ImageTransform_YFlip : vi::ImageTransform_None;
        nn::vi::ImageTransform srcTransform = nn::vi::ImageTransform_None;

        // 直接書込み
        vi::fbshare::SharedTextureOption dstOption = {};
        dstOption.transform   = dstTransform;
        dstOption.colorOption = vi::fbshare::SharedTextureColorOption_PreMultipledAlpha;
        dstOption.alphaOption = vi::fbshare::SharedTextureAlphaOption_None;
        dstOption.stacks      = m_LayerStacks;

        NN_RESULT_DO(DisplayPrimitiveOperation::SetDetachedSharedFrameBufferSubImage(
            m_BufferHandle,
            static_cast<int64_t>(index),
            static_cast<int32_t>(x),
            static_cast<int32_t>(y),
            static_cast<int32_t>(width),
            static_cast<int32_t>(height),
            0x00000000 /* transparent */,
            buffer,
            size,
            dstOption,
            srcTransform
        ));

        m_ImageRegionList[index].m_X = x;
        m_ImageRegionList[index].m_Y = y;
        m_ImageRegionList[index].m_Width = width;
        m_ImageRegionList[index].m_Height = height;

        UpdateVisibilityImpl();
        NN_RESULT_SUCCESS;
    }

    nn::Result ExtraSharedLayerManager::SetTextureThroughTempTexture(int index, int x, int y, int width, int height, const void* buffer, size_t size, int originMode, int temporalSystemTextureIndex) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(IsLayerCreated());
        NN_RESULT_THROW_UNLESS(index >= 0 && index < m_Layout.count, nn::am::ResultInvalidCall());

        nn::vi::ImageTransform dstTransform = (originMode == oe::WindowOriginMode_LowerLeft) ? vi::ImageTransform_YFlip : vi::ImageTransform_None;
        nn::vi::ImageTransform srcTransform = nn::vi::ImageTransform_None;

        auto hSystemBuffer = m_pResource->GetSystemSharedBufferHandle();

        // ① 一時バッファに書き込み。
        {
            vi::fbshare::SharedTextureOption tmpOption = {};
            tmpOption.transform   = vi::ImageTransform_None;    // システム共有バッファ上では必ず左上原点にする
            tmpOption.colorOption = vi::fbshare::SharedTextureColorOption_PreMultipledAlpha;
            tmpOption.alphaOption = vi::fbshare::SharedTextureAlphaOption_None;
            tmpOption.stacks      = m_LayerStacks;

            NN_RESULT_DO(DisplayPrimitiveOperation::SetDetachedSharedFrameBufferSubImage(
                hSystemBuffer,
                temporalSystemTextureIndex,
                static_cast<int32_t>(x),
                static_cast<int32_t>(y),
                static_cast<int32_t>(width),
                static_cast<int32_t>(height),
                0x00000000 /* transparent */,
                buffer,
                size,
                tmpOption,
                srcTransform
            ));
        }

        // ② 権利レイヤにコピー
        NN_RESULT_DO(DisplayPrimitiveOperation::CopyDetachedSharedFrameBufferImage(
            m_BufferHandle,
            index,
            hSystemBuffer,
            temporalSystemTextureIndex,
            dstTransform,   // 必要に応じて回転属性を設定
            nn::vi::fbshare::SharedTextureAlphaOption_None,
            0,
            m_LayerStacks
        ));

        m_ImageRegionList[index].m_X = x;
        m_ImageRegionList[index].m_Y = y;
        m_ImageRegionList[index].m_Width = width;
        m_ImageRegionList[index].m_Height = height;

        UpdateVisibilityImpl();
        NN_RESULT_SUCCESS;
    }

    void ExtraSharedLayerManager::UpdateVisibilityImpl() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());

        if(!m_IsLayerCreated)
        {
            return;
        }

        int newTextureIndex = -1;
        if(m_IsActive)
        {
            if(m_RequestedTextureIndex >= 0 && m_RequestedTextureIndex < m_Layout.count)
            {
                if(HasImage(m_RequestedTextureIndex))
                {
                    newTextureIndex = m_RequestedTextureIndex;
                }
            }
        }

        // 変化しないなら何もしない
        if(newTextureIndex == m_PresentedTextureIndex)
        {
            return;
        }

        if(newTextureIndex >= 0)
        {
            auto presentResult = DisplayPrimitiveOperation::PresentDetachedSharedFrameBufferToLowLevelLayer(m_BufferHandle, m_LowLayer.GetLayerId(), newTextureIndex);
            NN_AM_DISPCTRL_LOG_TRACE("  presenting application-copyright-texture() -> 0x%08x\n", presentResult.GetInnerValueForDebug());
            NN_UNUSED(presentResult);
            m_PresentedTextureIndex = newTextureIndex;
        }
        else
        {
            ForceHideImpl();
        }
    }

    void ExtraSharedLayerManager::ForceHideImpl() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(IsLayerCreated());

        if(m_PresentedTextureIndex < 0)
        {
            return;
        }

        DisplayPrimitiveOperation::BlankSharedLowLevelLayer(m_LowLayer.GetLayerId(), m_BufferHandle);
        m_PresentedTextureIndex = -1;
    }

}}}}}

