﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/core/am_ThreadImplementedApplet.h>

#include <nn/nn_Common.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>

#include <nn/os/os_ThreadApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os/os_Mutex.h>
#include <mutex>
#include <nn/os/os_Event.h>
#include <nn/am/am_ResultPrivate.h>
#include <nn/am/service/am_StuckChecker.h>

namespace nn { namespace am { namespace service { namespace core {

namespace {

    const size_t StackSize = 1024 * 16;
    const int ThreadCountMax = 12;
    struct Stack
    {
        NN_OS_ALIGNAS_THREAD_STACK char buffer[StackSize];
    };
    Stack g_Stacks[ThreadCountMax];
    bool g_StackUsed[ThreadCountMax];

    struct StaticMutex
    {
        os::MutexType _mutex;
        void lock() NN_NOEXCEPT
        {
            os::LockMutex(&_mutex);
        }
        void unlock() NN_NOEXCEPT
        {
            os::UnlockMutex(&_mutex);
        }
    };
    StaticMutex g_Mutex = { NN_OS_MUTEX_INITIALIZER(false) };

    void* AllocateStack(size_t size) NN_NOEXCEPT
    {
        NN_UNUSED(size);
        NN_SDK_ASSERT(size == StackSize);
        std::lock_guard<decltype(g_Mutex)> lk(g_Mutex);
        for (auto i = 0u; i < sizeof(g_StackUsed) / sizeof(*g_StackUsed); ++i)
        {
            if (!g_StackUsed[i])
            {
                g_StackUsed[i] = true;
                return &g_Stacks[i];
            }
        }
        return nullptr;
    }

    void DeallocateStack(void* p, size_t size) NN_NOEXCEPT
    {
        NN_UNUSED(size);
        NN_SDK_ASSERT(size == StackSize);
        auto i = static_cast<Stack*>(p) - g_Stacks;
        std::lock_guard<decltype(g_Mutex)> lk(g_Mutex);
        NN_SDK_ASSERT(g_StackUsed[i]);
        g_StackUsed[i] = false;
    }

    class AlwaysSignaledEvent
        : public os::Event
    {
    public:

        AlwaysSignaledEvent() NN_NOEXCEPT
            : Event(os::EventClearMode_ManualClear)
        {
            this->Signal();
        }

    };

    os::EventType* GetAlwaysSignaledEvent() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(AlwaysSignaledEvent, g_AlwaysSignaledEvent);
        return g_AlwaysSignaledEvent.GetBase();
    }

}

namespace detail {

ThreadImplementedAppletImpl::ThreadImplementedAppletImpl(StackableApplet* pApplet) NN_NOEXCEPT
    : m_pApplet(pApplet)
    , m_Result(am::ResultAppletTerminatedInternal())
{
    NN_UNUSED(m_pApplet);
}

ThreadImplementedAppletImpl::~ThreadImplementedAppletImpl() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(!m_ThreadRunning, "not joined");
}

inline Result ThreadImplementedAppletImpl::StartImpl2() NN_NOEXCEPT
{
    auto success = false;
    auto stack = AllocateStack(StackSize);
    NN_RESULT_THROW_UNLESS(stack, ResultOutOfAppletThreadStack());
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            NN_AM_SERVICE_APPLET_LOG(error, m_pApplet, "failed to allocate thread stack");
            DeallocateStack(stack, StackSize);
        }
    };
    NN_RESULT_DO(os::CreateThread(&m_Thread, [](void* p)
    {
        NN_AM_SERVICE_DIAGNOSTICS_SET_CONTEXT_NAME(AppletRunning);
        auto this_ = static_cast<ThreadImplementedAppletImpl*>(p);
        NN_AM_SERVICE_APPLET_LOG(call, this_->m_pApplet, "begin Run()");
        this_->m_Result = this_->Run();
        NN_AM_SERVICE_APPLET_LOG(call, this_->m_pApplet, "end Run() " NN_AM_SERVICE_DIAGNOSTICS_RESULT_STRING_FORMAT, NN_AM_SERVICE_DIAGNOSTICS_RESULT_STRING_FORMAT_PARAMETER(this_->m_Result));
    }, this, stack, StackSize, NN_SYSTEM_THREAD_PRIORITY(am, AppletRunning)));
    success = true;
    this->m_Stack = stack;
    NN_AM_SERVICE_APPLET_LOG(seq, m_pApplet, NN_AM_SERVICE_DIAGNOSTICS_THREAD_STRING_FORMAT " start", NN_AM_SERVICE_DIAGNOSTICS_THREAD_STRING_FORMAT_PARAMETER(&m_Thread));
    os::StartThread(&m_Thread);
    NN_RESULT_SUCCESS;
}

void ThreadImplementedAppletImpl::StartImpl() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    if (m_Started)
    {
        return;
    }
    this->m_Started = true;
    auto result = StartImpl2();
    if (result.IsFailure())
    {
        NN_AM_SERVICE_APPLET_LOG(error, m_pApplet, "failed to create thread " NN_AM_SERVICE_DIAGNOSTICS_RESULT_STRING_FORMAT, NN_AM_SERVICE_DIAGNOSTICS_RESULT_STRING_FORMAT_PARAMETER(result));
        this->m_Result = result;
        this->m_ThreadRunning = false;
        return;
    }
    this->m_ThreadRunning = true;
}

bool ThreadImplementedAppletImpl::TryJoinImpl() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    if (m_ThreadRunning)
    {
        if (!(os::TryWaitAny(&m_Thread) == 0))
        {
            return false;
        }
        os::WaitThread(&m_Thread);
        NN_AM_SERVICE_APPLET_LOG(seq, m_pApplet, NN_AM_SERVICE_DIAGNOSTICS_THREAD_STRING_FORMAT " join", NN_AM_SERVICE_DIAGNOSTICS_THREAD_STRING_FORMAT_PARAMETER(&m_Thread));
        os::DestroyThread(&m_Thread);
        DeallocateStack(m_Stack, StackSize);
        this->m_Stack = nullptr;
        this->m_ThreadRunning = false;
    }
    return true;
}

void ThreadImplementedAppletImpl::InitializeMultiWaitHolderImpl(os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    if (m_ThreadRunning)
    {
        os::InitializeMultiWaitHolder(pHolder, &m_Thread);
    }
    else
    {
        os::InitializeMultiWaitHolder(pHolder, GetAlwaysSignaledEvent());
    }
}

}

}}}}
