﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/am_RightsEnvironment.h>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_ScopedTransaction.h>

#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <limits>
#include <nn/am/am_Result.h>

namespace nn { namespace am { namespace service { namespace rightsManagement {

RightsEnvironmentStatus GetInitialRightsEnvironmentStatus() NN_NOEXCEPT
{
    return {true, os::Tick(std::numeric_limits<int64_t>::max())};
}


Result NoCheckRightsEnvironment::SetUsers(util::Span<const account::Uid>) NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

Result NoCheckRightsEnvironment::SetUsers(AllUserTag) NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

void NoCheckRightsEnvironment::Start() NN_NOEXCEPT
{
    // nop
}

void NoCheckRightsEnvironment::Stop() NN_NOEXCEPT
{
    // nop
}

RightsEnvironmentStatus NoCheckRightsEnvironment::GetStatus() const NN_NOEXCEPT
{
    return GetInitialRightsEnvironmentStatus();
}

void NoCheckRightsEnvironment::RequestExpandExpirationDate() NN_NOEXCEPT
{
    // nop
}

Result NoCheckRightsEnvironment::GetNsHandle(Bit64* pOut) NN_NOEXCEPT
{
    NN_UNUSED(pOut);
    NN_RESULT_THROW(am::ResultInvalidCall());
}

Result ApplicationRightsEnvironment::Initialize(ncm::ApplicationId applicationId) NN_NOEXCEPT
{
    NN_SDK_ASSERT(!IsInitialized());

    util::ScopedTransaction t;

    ns::RightsEnvironmentHandle handle;
    NN_RESULT_DO(ns::CreateRightsEnvironment(&handle));
    NN_UTIL_RESERVE_SCOPED_ROLLBACK(t)
    {
        ns::DestroyRightsEnvironment(handle);
    };

    NN_RESULT_DO(ns::AddTargetApplicationToRightsEnvironment(handle, applicationId));

    t.Commit();
    this->m_ApplicationId = applicationId;
    this->m_Handle = handle;
    NN_RESULT_SUCCESS;
}

void ApplicationRightsEnvironment::FinalizeImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized());
    ns::DestroyRightsEnvironment(m_Handle);
}

ApplicationRightsEnvironment::~ApplicationRightsEnvironment() NN_NOEXCEPT
{
    if (IsInitialized())
    {
        FinalizeImpl();
    }
}

Result ApplicationRightsEnvironment::SetUsers(util::Span<const account::Uid> uids) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized());
    return ns::SetUsersToRightsEnvironment(m_Handle, false, uids);
}

Result ApplicationRightsEnvironment::SetUsers(AllUserTag) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized());
    account::Uid dummy[1];
    return ns::SetUsersToRightsEnvironment(m_Handle, true, {dummy, dummy});
}

void ApplicationRightsEnvironment::Start() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized());
    NN_ABORT_UNLESS_RESULT_SUCCESS(ns::ActivateRightsEnvironment(m_Handle));
}

void ApplicationRightsEnvironment::Stop() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized());
    ns::DeactivateRightsEnvironment(m_Handle);
}

RightsEnvironmentStatus ApplicationRightsEnvironment::GetStatus() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized());
    ns::RightsEnvironmentStatus nsStatus;
    NN_ABORT_UNLESS_RESULT_SUCCESS(ns::GetRightsEnvironmentStatus(&nsStatus, m_Handle));
    return {nsStatus.isAvailable, os::Tick(nsStatus.expirationTimeInTick)};
}

void ApplicationRightsEnvironment::RequestExpandExpirationDate() NN_NOEXCEPT
{
    // TODO: implement RightsEnvironment::RequestExpandExpirationDate()
}

Result ApplicationRightsEnvironment::GetNsHandle(Bit64* pOut) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized());
    *pOut = static_cast<Bit64>(m_Handle);
    NN_RESULT_SUCCESS;
}

}}}}
