﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/am_NsWrapper.h>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/am/service/am_ServiceDiagnostics.h>
#include <nn/am/service/am_StuckChecker.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_SdkConditionVariable.h>

#include <mutex>

namespace nn { namespace am { namespace service {

namespace {

os::SdkMutex g_Mutex;
os::SdkConditionVariable g_Cv;
int64_t g_BoostSystemMemoryResourceLimitSize;

}

void NsBoostSystemMemoryResourceLimit(int64_t boostSize) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(ns_BoostSystemMemoryResourceLimit, 30);

    auto endTick = os::GetSystemTick() + os::ConvertToTick(TimeSpan::FromSeconds(40));
    for (;;)
    {
        auto result = ns::BoostSystemMemoryResourceLimit( boostSize );
        if (result.IsSuccess())
        {
            std::lock_guard<decltype(g_Mutex)> lk(g_Mutex);
            g_BoostSystemMemoryResourceLimitSize = boostSize;
            if (g_BoostSystemMemoryResourceLimitSize == 0)
            {
                g_Cv.Signal();
            }
            return;
        }
        if (os::GetSystemTick() < endTick)
        {
            os::SleepThread( TimeSpan::FromMilliSeconds(100) );
            continue;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
#else
    NN_UNUSED(boostSize);
#endif
}

void WaitForNotBoostSystemMemoryResourceLimit() NN_NOEXCEPT
{
    std::lock_guard<decltype(g_Mutex)> lk(g_Mutex);
    while (!(g_BoostSystemMemoryResourceLimitSize == 0))
    {
        g_Cv.Wait(g_Mutex);
    }
}

}}}
