﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/am_IntegratedLibraryApplet.h>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <utility>
#include <memory>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/am/am_ResultPrivate.h>

#include <list>
#include <queue>
#include <nn/am/am_Result.h>
#include <nn/am/service/am_IntegratedApplication.h>
#include <nn/am/service/am_PlayDataManager.h>
#include <nn/am/service/am_ServiceStaticAllocator.h>
#include <nn/os/os_Tick.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SystemEvent.h>
#include <mutex>

#include <nn/am/service/am_AppletProxy.h>
#include <nn/am/service/am_ErrorReport.h>
#include "am_GrcControl.h"

namespace nn { namespace am { namespace service {

IntegratedLibraryApplet::IntegratedLibraryApplet(AppletSystem* pAppletSystem, applet::AppletId appletId, applet::LibraryAppletMode mode) NN_NOEXCEPT
    : IntegratedApplet(pAppletSystem, mode != applet::LibraryAppletMode::LibraryAppletMode_NoUi)
    , m_AppletId(appletId)
    , m_LibraryAppletMode(mode)
    , m_InChannel(MakeShared<Channel>(true))
    , m_OutChannel(MakeShared<Channel>(true))
    , m_InteractiveInChannel(MakeShared<Channel>(true))
    , m_InteractiveOutChannel(MakeShared<Channel>(true))
    , m_ExtraChannel(MakeShared<Channel>(true))
    , m_Context(MakeShared<Channel>(false))
{
    if (mode == applet::LibraryAppletMode_PartialForegroundWithIndirectDisplay)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(this->CreateIndirectLayer());
    }

    this->SetDestinationLayerIndirect(mode == applet::LibraryAppletMode_PartialForegroundWithIndirectDisplay);
}

applet::AppletIdentityInfo IntegratedLibraryApplet::GetAppletIdentityInfo() NN_NOEXCEPT
{
    applet::AppletIdentityInfo ret = {};
    ret.appletId = m_AppletId;
    return ret;
}

Result IntegratedLibraryApplet::CreateProcessImpl(std::shared_ptr<process::NsProcess>* pOut) NN_NOEXCEPT
{
    return CreateAppletProcessImpl(pOut, m_AppletId);
}

std::shared_ptr<process::NsProcess> IntegratedLibraryApplet::MakeAttachedProcess(os::ProcessId processId) NN_NOEXCEPT
{
    return GetAppletSystem()->GetNsProcessManager()->MakeLibraryAppletProcess(processId);
}

void IntegratedLibraryApplet::IncrementChannelRef() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_ChannelClearMutex)> lk(m_ChannelClearMutex);
    ++this->m_ChannelRefCount;
}

void IntegratedLibraryApplet::DecrementChannelRef() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_ChannelClearMutex)> lk(m_ChannelClearMutex);
    --this->m_ChannelRefCount;
    if (m_ChannelRefCount == 0)
    {
        // チャネルなどへのアクセスをする参照がすべて消えた場合には、チャネルをすべてクリアする
        m_InChannel->Clear();
        m_OutChannel->Clear();
        m_InteractiveInChannel->Clear();
        m_InteractiveOutChannel->Clear();
        m_ExtraChannel->Clear();
        m_Context->Clear();
    }
}

void IntegratedLibraryApplet::Cleanup() NN_NOEXCEPT
{
    IntegratedApplet::Cleanup();
}

void IntegratedLibraryApplet::OnProcessBegin(os::ProcessId processId) NN_NOEXCEPT
{
    IntegratedApplet::OnProcessBegin(processId);
    ++this->m_ProcessBeginCount;
}

class IntegratedLibraryApplet::LibraryAppletAccessorImpl
    : public AppletAccessorImpl
{
private:

    IntegratedLibraryApplet* GetApplet() const NN_NOEXCEPT
    {
        return static_cast<IntegratedLibraryApplet*>(AppletAccessorImpl::GetApplet());
    }

    virtual void OnStartByAccessor() NN_NOEXCEPT NN_OVERRIDE
    {
        GetApplet()->GetParent()->SetLastLibraryApplet(GetApplet()->shared_from_this());
    }

public:

    explicit LibraryAppletAccessorImpl(std::shared_ptr<IntegratedLibraryApplet> p) NN_NOEXCEPT
        : AppletAccessorImpl(std::move(p))
    {
        GetApplet()->IncrementChannelRef();
    }

    ~LibraryAppletAccessorImpl() NN_NOEXCEPT
    {
        GetApplet()->DecrementChannelRef();
    }

    using AppletAccessorImpl::GetIndirectLayerConsumerHandle;

    nn::Result PushInData(nn::sf::SharedPointer<nn::am::service::IStorage> storage) NN_NOEXCEPT
    {
        decltype(storage) other;
        NN_RESULT_DO(storage->GetAndInvalidate(&other));
        GetApplet()->m_InChannel->Push(std::move(other));
        NN_AM_SERVICE_APPLET_LOG(call, GetApplet(), "%s", NN_CURRENT_FUNCTION_NAME);
        NN_RESULT_SUCCESS;
    }

    nn::Result PushInteractiveInData(nn::sf::SharedPointer<nn::am::service::IStorage> storage) NN_NOEXCEPT
    {
        decltype(storage) other;
        NN_RESULT_DO(storage->GetAndInvalidate(&other));
        GetApplet()->m_InteractiveInChannel->Push(std::move(other));
        NN_AM_SERVICE_APPLET_LOG(call, GetApplet(), "%s", NN_CURRENT_FUNCTION_NAME);
        NN_RESULT_SUCCESS;
    }

    nn::Result PopOutData(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IStorage>> pOut) NN_NOEXCEPT
    {
        auto ret = GetApplet()->m_OutChannel->Pop();
        NN_RESULT_THROW_UNLESS(ret, ResultNoStorage());
        *pOut = std::move(ret);
        NN_AM_SERVICE_APPLET_LOG(call, GetApplet(), "%s", NN_CURRENT_FUNCTION_NAME);
        NN_RESULT_SUCCESS;
    }

    nn::Result GetPopOutDataEvent(nn::sf::Out<nn::sf::NativeHandle> pOut) NN_NOEXCEPT
    {
        *pOut = GetApplet()->m_OutChannel->GetPopEventHandle();
        NN_RESULT_SUCCESS;
    }

    nn::Result PopInteractiveOutData(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IStorage>> pOut) NN_NOEXCEPT
    {
        auto ret = GetApplet()->m_InteractiveOutChannel->Pop();
        NN_RESULT_THROW_UNLESS(ret, ResultNoStorage());
        *pOut = std::move(ret);
        NN_AM_SERVICE_APPLET_LOG(call, GetApplet(), "%s", NN_CURRENT_FUNCTION_NAME);
        NN_RESULT_SUCCESS;
    }

    nn::Result GetPopInteractiveOutDataEvent(nn::sf::Out<nn::sf::NativeHandle> pOut) NN_NOEXCEPT
    {
        *pOut = GetApplet()->m_InteractiveOutChannel->GetPopEventHandle();
        NN_AM_SERVICE_APPLET_LOG(call, GetApplet(), "%s", NN_CURRENT_FUNCTION_NAME);
        NN_RESULT_SUCCESS;
    }

    nn::Result PushExtraStorage(nn::sf::SharedPointer<IStorage> storage) NN_NOEXCEPT
    {
        decltype(storage) other;
        NN_RESULT_DO(storage->GetAndInvalidate(&other));
        GetApplet()->m_ExtraChannel->Push(std::move(other));
        NN_AM_SERVICE_APPLET_LOG(call, GetApplet(), "%s", NN_CURRENT_FUNCTION_NAME);
        NN_RESULT_SUCCESS;
    }

    nn::Result GetLibraryAppletInfo(sf::Out<LibraryAppletInfo> pOut)
    {
        LibraryAppletInfo ret = {};
        ret.appletId = GetApplet()->m_AppletId;
        ret.mode = GetApplet()->m_LibraryAppletMode;
        *pOut = ret;
        NN_RESULT_SUCCESS;
    }

    nn::Result GetResult() NN_NOEXCEPT
    {
        NN_RESULT_DO(AppletAccessorImpl::GetResult());
        {
            std::unique_lock<decltype(GetApplet()->m_ReturnMutex)> lk(GetApplet()->m_ReturnMutex);
            auto returnIsValid = GetApplet()->m_ReturnIsValid;
            lk.unlock();
            if (!returnIsValid)
            {
                NN_RESULT_THROW_UNLESS(GetApplet()->IsExitRequested(), am::ResultAppletExitAbnormally());
                NN_RESULT_THROW(am::ResultAppletCancelled());
            }
        }
        NN_RESULT_SUCCESS;
    }

    nn::Result SetOutOfFocusApplicationSuspendingEnabled(bool usesApplicationCore) NN_NOEXCEPT
    {
        GetApplet()->m_ApplicationCoreUsageRequest = usesApplicationCore ? CoreUsageRequest::Shared : CoreUsageRequest::None;
        NN_RESULT_SUCCESS;
    }

    nn::Result NeedsToExitProcess(nn::sf::Out<bool> pOut) NN_NOEXCEPT
    {
        NN_UNUSED(pOut);
        NN_AM_SERVICE_NOT_IMPLEMENTED("TODO: implement: IntegratedLibraryApplet::LibraryAppletAccessorImpl::NeedsToExitProcess()");
    }

    nn::Result RequestForAppletToGetForeground() NN_NOEXCEPT
    {
        NN_AM_SERVICE_NOT_IMPLEMENTED("TODO: implement: IntegratedLibraryApplet::LibraryAppletAccessorImpl::RequestForAppletToGetForeground()");
    }

};

sf::SharedPointer<ILibraryAppletAccessor> IntegratedLibraryApplet::CreateAccessor() NN_NOEXCEPT
{
    return sf::CreateSharedObjectEmplaced<ILibraryAppletAccessor, LibraryAppletAccessorImpl>(SharedFromThis());
}

void IntegratedLibraryApplet::SendPlayLog(pdm::AppletEventType eventType) NN_NOEXCEPT
{
    ncm::SystemProgramId mainProgramId;
    auto mainAppletInfo = GetMainApplet()->GetAppletIdentityInfo();
    if( mainAppletInfo.IsApplication() )
    {
        mainProgramId.value = mainAppletInfo.applicationId.value;
    }
    else if( mainAppletInfo.IsSystemApplication() )
    {
        mainProgramId.value = mainAppletInfo.systemApplicationId.value;
    }
    else
    {
        auto result = MakeSystemProgramIdFromAppletId(&mainProgramId, mainAppletInfo.appletId);
        if( result.IsFailure() )
        {
            NN_AM_SERVICE_APPLET_LOG(error, this, "IntegratedLibraryApplet::SendPlayLog: MakeSystemProgramIdFromAppletId(&out, %u) failed (%03d-%04d, 0x%08x).\n",
                mainAppletInfo.appletId, result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
            mainProgramId = mainProgramId.GetInvalidId();
        }
    }
    NotifyLibraryAppletEvent(eventType, mainProgramId, m_LibraryAppletMode, GetAppletIdentityInfo().appletId,
        ncm::StorageId::BuildInSystem, ns::PlayLogPolicy::LogOnly);
}

class IntegratedLibraryApplet::LibraryAppletSelfProxyImpl
    : public AppletSelfProxyImpl
{
public:

    explicit LibraryAppletSelfProxyImpl(std::shared_ptr<IntegratedLibraryApplet> p, os::ProcessId processId) NN_NOEXCEPT
        : AppletSelfProxyImpl(p, processId)
    {
        this->SetHandlesHomeButtonShortPressed(HomeButtonShortPressedHandleType_NotifyBlockingHomeButton);
        this->SetHandlesHomeButtonLongPressed(false);
        this->SetHandlesPowerButtonShortPressed(false);
        GetApplet()->IncrementChannelRef();
        NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "LA Create End: %lld ms", os::GetSystemTick().ToTimeSpan().GetMilliSeconds());
    }

    ~LibraryAppletSelfProxyImpl() NN_NOEXCEPT
    {

        GetApplet()->DecrementChannelRef();
    }

    using AppletSelfProxyImpl::GetIndirectLayerProducerHandle;

    nn::Result PopInData(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IStorage>> pOut) NN_NOEXCEPT
    {
        auto ret = GetApplet()->m_InChannel->Pop();
        NN_RESULT_THROW_UNLESS(ret, ResultNoStorage());
        *pOut = std::move(ret);
        NN_AM_SERVICE_APPLET_LOG(call, GetApplet(), "%s", NN_CURRENT_FUNCTION_NAME);
        NN_RESULT_SUCCESS;
    }

    nn::Result UnpopInData(nn::sf::SharedPointer<nn::am::service::IStorage> storage) NN_NOEXCEPT
    {
        decltype(storage) other;
        NN_RESULT_DO(storage->GetAndInvalidate(&other));
        GetApplet()->m_InChannel->Unpop(std::move(other));
        NN_AM_SERVICE_APPLET_LOG(call, GetApplet(), "%s", NN_CURRENT_FUNCTION_NAME);
        NN_RESULT_SUCCESS;
    }

    nn::Result GetPopInDataEvent(nn::sf::Out<nn::sf::NativeHandle> pOut) NN_NOEXCEPT
    {
        *pOut = GetApplet()->m_InChannel->GetPopEventHandle();
        NN_RESULT_SUCCESS;
    }

    nn::Result PopInteractiveInData(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IStorage>> pOut) NN_NOEXCEPT
    {
        auto ret = GetApplet()->m_InteractiveInChannel->Pop();
        NN_RESULT_THROW_UNLESS(ret, ResultNoStorage());
        *pOut = std::move(ret);
        NN_AM_SERVICE_APPLET_LOG(call, GetApplet(), "%s", NN_CURRENT_FUNCTION_NAME);
        NN_RESULT_SUCCESS;
    }

    nn::Result GetPopInteractiveInDataEvent(nn::sf::Out<nn::sf::NativeHandle> pOut) NN_NOEXCEPT
    {
        *pOut = GetApplet()->m_InteractiveInChannel->GetPopEventHandle();
        NN_AM_SERVICE_APPLET_LOG(call, GetApplet(), "%s", NN_CURRENT_FUNCTION_NAME);
        NN_RESULT_SUCCESS;
    }

    nn::Result PushOutData(nn::sf::SharedPointer<nn::am::service::IStorage> storage) NN_NOEXCEPT
    {
        decltype(storage) other;
        NN_RESULT_DO(storage->GetAndInvalidate(&other));
        GetApplet()->m_OutChannel->Push(std::move(other));
        NN_AM_SERVICE_APPLET_LOG(call, GetApplet(), "%s", NN_CURRENT_FUNCTION_NAME);
        NN_RESULT_SUCCESS;
    }

    nn::Result PushInteractiveOutData(nn::sf::SharedPointer<nn::am::service::IStorage> storage) NN_NOEXCEPT
    {
        decltype(storage) other;
        NN_RESULT_DO(storage->GetAndInvalidate(&other));
        GetApplet()->m_InteractiveOutChannel->Push(std::move(other));
        NN_AM_SERVICE_APPLET_LOG(call, GetApplet(), "%s", NN_CURRENT_FUNCTION_NAME);
        NN_RESULT_SUCCESS;
    }

    nn::Result PopExtraStorage(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IStorage>> pOut) NN_NOEXCEPT
    {
        auto ret = GetApplet()->m_ExtraChannel->Pop();
        NN_RESULT_THROW_UNLESS(ret, ResultNoStorage());
        *pOut = std::move(ret);
        NN_AM_SERVICE_APPLET_LOG(call, GetApplet(), "%s", NN_CURRENT_FUNCTION_NAME);
        NN_RESULT_SUCCESS;
    }

    nn::Result UnpopExtraStorage(nn::sf::SharedPointer<nn::am::service::IStorage> storage) NN_NOEXCEPT
    {
        decltype(storage) other;
        NN_RESULT_DO(storage->GetAndInvalidate(&other));
        GetApplet()->m_ExtraChannel->Unpop(std::move(other));
        NN_AM_SERVICE_APPLET_LOG(call, GetApplet(), "%s", NN_CURRENT_FUNCTION_NAME);
        NN_RESULT_SUCCESS;
    }

    nn::Result GetPopExtraStorageEvent(nn::sf::Out<nn::sf::NativeHandle> pOut) NN_NOEXCEPT
    {
        *pOut = GetApplet()->m_ExtraChannel->GetPopEventHandle();
        NN_RESULT_SUCCESS;
    }

    nn::Result ExitProcessAndReturn() NN_NOEXCEPT
    {
        {
            std::lock_guard<decltype(GetApplet()->m_ReturnMutex)> lk(GetApplet()->m_ReturnMutex);
            GetApplet()->m_ReturnIsValid = true;
        }
        this->InvalidateDisplayLayer();
        GetApplet()->UnlockExit();
        return GetApplet()->KillProcess();
    }

    nn::Result GetLibraryAppletInfo(sf::Out<LibraryAppletInfo> pOut)
    {
        LibraryAppletInfo ret = {};
        ret.appletId = GetApplet()->m_AppletId;
        ret.mode = GetApplet()->m_LibraryAppletMode;
        *pOut = ret;
        NN_RESULT_SUCCESS;
    }

    Result GetMainAppletIdentityInfo(sf::Out<AppletIdentityInfo> pOut) NN_NOEXCEPT
    {
        auto info = GetApplet()->GetMainApplet()->GetAppletIdentityInfo();
        *pOut = {};
        pOut->appletId = static_cast<decltype(pOut->appletId)>(info.appletId);
        pOut->applicationId = info.applicationId.value;
        NN_RESULT_SUCCESS;
    }

    Result GetCallerAppletIdentityInfo(sf::Out<AppletIdentityInfo> pOut) NN_NOEXCEPT
    {
        *pOut = {};
        auto parent = GetApplet()->GetParent();
        if (!parent)
        {
            pOut->appletId = applet::AppletId_None;
            NN_RESULT_SUCCESS;
        }
        auto info = parent->GetAppletIdentityInfo();
        pOut->appletId = static_cast<decltype(pOut->appletId)>(info.appletId);
        pOut->applicationId = info.applicationId.value;
        NN_RESULT_SUCCESS;
    }

    Result GetCallerAppletIdentityInfoStack(sf::Out<int> pOutCount, sf::OutArray<AppletIdentityInfo> pOutList) NN_NOEXCEPT
    {
        IntegratedApplet* pCurrent = this->GetApplet();
        auto i = 0u;
        for (; i < pOutList.GetLength(); ++i)
        {
            pCurrent = pCurrent->GetParent();
            if (!pCurrent)
            {
                break;
            }
            auto info = pCurrent->GetAppletIdentityInfo();
            pOutList[i].appletId = static_cast<decltype(pOutList[i].appletId)>(info.appletId);
            pOutList[i].applicationId = info.applicationId.value;
        }
        *pOutCount = static_cast<int>(i);
        NN_RESULT_SUCCESS;
    }

    Result GetDesirableKeyboardLayout(sf::Out<Bit32> pOutLayout) NN_NOEXCEPT
    {
        // 現実装ではメインアプレットの要望キーボード配列を取得する
        auto pMainApplet = GetApplet()->GetMainApplet();
        NN_RESULT_THROW_UNLESS(pMainApplet, am::ResultNoMainApplet());
        NN_RESULT_THROW_UNLESS(pMainApplet->GetDesirableKeyboardLayoutImpl(&(*pOutLayout)), am::ResultMainAppletNotSetParameter());
        NN_RESULT_SUCCESS;
    }

    Result GetNextReturnDestinationAppletIdentityInfo(sf::Out<AppletIdentityInfo> pOut) NN_NOEXCEPT
    {
        *pOut = {};
        auto returnDest = GetApplet()->GetParent();
        while (returnDest && returnDest->IsReservedWithoutUnwinding())
        {
            returnDest = returnDest->GetParent();
        }

        if (!returnDest)
        {
            pOut->appletId = applet::AppletId_None;
            NN_RESULT_SUCCESS;
        }
        auto info = returnDest->GetAppletIdentityInfo();
        pOut->appletId = static_cast<decltype(pOut->appletId)>(info.appletId);
        pOut->applicationId = info.applicationId.value;
        NN_RESULT_SUCCESS;
    }

    Result GetMainAppletApplicationControlProperty(sf::Out<ns::ApplicationControlProperty> pOut) NN_NOEXCEPT
    {
        auto pMainApplet = GetApplet()->GetMainApplet();
        NN_RESULT_THROW_UNLESS(pMainApplet, ResultNoMainApplet());
        NN_RESULT_THROW_UNLESS(pMainApplet->GetAppletIdentityInfo().IsApplication(), am::ResultInvalidCall());
        *pOut = static_cast<IntegratedApplication*>(pMainApplet)->RefControlProperty();
        NN_RESULT_SUCCESS;
    }

    Result GetMainAppletApplicationDesiredLanguage(sf::Out<settings::LanguageCode> pOut) NN_NOEXCEPT
    {
        auto pMainApplet = GetApplet()->GetMainApplet();
        NN_RESULT_THROW_UNLESS(pMainApplet, ResultNoMainApplet());
        NN_RESULT_THROW_UNLESS(pMainApplet->GetAppletIdentityInfo().IsApplication(), am::ResultMainAppletNotApplication());
        return static_cast<IntegratedApplication*>(pMainApplet)->GetDesiredLanguageImpl(pOut.GetPointer());
    }

    Result GetMainAppletStorageId(sf::Out<ncm::StorageId> pOut) NN_NOEXCEPT
    {
        auto pMainApplet = GetApplet()->GetMainApplet();
        NN_RESULT_THROW_UNLESS(pMainApplet, ResultNoMainApplet());
        NN_RESULT_THROW_UNLESS(pMainApplet->GetAppletIdentityInfo().IsApplication(), am::ResultInvalidCall());
        *pOut = static_cast<IntegratedApplication*>(pMainApplet)->GetLaunchedStorageId();
        NN_RESULT_SUCCESS;
    }

    Result CanUseApplicationCore(sf::Out<bool> pOut) NN_NOEXCEPT
    {
        // アプリが不活性（存在しない or サスペンド中）ならば true を返す
        *pOut = !GetApplet()->GetAppletSystem()->AreAnyApplicationActive();
        NN_RESULT_SUCCESS;
    }

    // アプリ起動要求
    Result CreateApplicationAndPushAndRequestToLaunch(ncm::ApplicationId applicationId, sf::SharedPointer<IStorage> storage) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(applicationId != ncm::ApplicationId::GetInvalidId(), am::ResultInvalidParameter());

        // アプリ起動要求元の情報
        applet::ApplicationLaunchRequestInfo launchRequestInfo = {};
        launchRequestInfo.requestingAppletId = GetApplet()->GetAppletIdentityInfo().appletId;
        auto mainAppletInfo = GetApplet()->GetMainApplet()->GetAppletIdentityInfo();
        launchRequestInfo.requestingMainAppletId = mainAppletInfo.appletId;
        launchRequestInfo.applicationId          = mainAppletInfo.applicationId;

        std::shared_ptr<IntegratedApplication> pApplication;
        NN_RESULT_DO(GetApplet()->GetAppletSystem()->CreateApplicationForReserved(&pApplication, applicationId, launchRequestInfo));
        NN_RESULT_DO(pApplication->PushToInChannel(applet::LaunchParameterKind_User, std::move(storage)));
        GetApplet()->GetAppletSystem()->PushLaunchRequestedApplication(pApplication);
        NN_RESULT_SUCCESS;
    }

    std::shared_ptr<LibraryAppletSelfProxyImpl> OpenLibraryAppletSelfAccessor() NN_NOEXCEPT
    {
        return std::static_pointer_cast<LibraryAppletSelfProxyImpl>(shared_from_this());
    }

    Result GetLaunchReason(sf::Out<am::service::AppletProcessLaunchReason> pOut) NN_NOEXCEPT
    {
        am::service::AppletProcessLaunchReason ret = {};
        ret.unwinding = GetApplet()->m_ProcessBeginCount > 1;
        *pOut = ret;
        NN_RESULT_SUCCESS;
    }

    Result OpenCallingLibraryApplet(sf::Out<sf::SharedPointer<am::service::ILibraryAppletAccessor>> pOut) NN_NOEXCEPT
    {
        auto p = GetApplet()->GetReserved();
        NN_RESULT_THROW_UNLESS(p, ResultNoAppletAccessor());
        *pOut = std::move(p);
        NN_RESULT_SUCCESS;
    }

    Result PushContext(sf::SharedPointer<am::service::IStorage> storage) NN_NOEXCEPT
    {
        decltype(storage) other;
        NN_RESULT_DO(storage->GetAndInvalidate(&other));
        GetApplet()->m_Context->Push(std::move(other));
        NN_AM_SERVICE_APPLET_LOG(call, GetApplet(), "%s", NN_CURRENT_FUNCTION_NAME);
        NN_RESULT_SUCCESS;
    }

    Result PopContext(sf::Out<sf::SharedPointer<am::service::IStorage>> pOut) NN_NOEXCEPT
    {
        auto ret = GetApplet()->m_Context->Pop();
        NN_RESULT_THROW_UNLESS(ret, ResultNoStorage());
        *pOut = std::move(ret);
        NN_AM_SERVICE_APPLET_LOG(call, GetApplet(), "%s", NN_CURRENT_FUNCTION_NAME);
        NN_RESULT_SUCCESS;
    }

    Result CancelWindingReservation() NN_NOEXCEPT
    {
        GetApplet()->ClearReservation();
        NN_RESULT_SUCCESS;
    }

    Result WindAndDoReserved() NN_NOEXCEPT
    {
        this->InvalidateDisplayLayer();
        GetApplet()->UnlockExit(); // see SIGLO-39618
        GetApplet()->KillProcess();
        NN_RESULT_SUCCESS;
    }

    Result ReserveToStartAndWaitAndUnwindThis(sf::SharedPointer<am::service::ILibraryAppletAccessor> p) NN_NOEXCEPT
    {
        GetApplet()->ReserveToStartAndWaitAndUnwindThis(std::move(p));
        NN_RESULT_SUCCESS;
    }

    Result ReserveToStartAndWait(sf::SharedPointer<am::service::ILibraryAppletAccessor> p) NN_NOEXCEPT
    {
        GetApplet()->ReserveToStartAndWait(std::move(p));
        NN_RESULT_SUCCESS;
    }

    std::shared_ptr<LibraryAppletSelfProxyImpl> GetProcessWindingController() NN_NOEXCEPT
    {
        return std::static_pointer_cast<LibraryAppletSelfProxyImpl>(shared_from_this());
    }

    Result ReportVisibleError(nn::err::ErrorCode errorCode) NN_NOEXCEPT
    {
        auto result = am::service::MakeVisibleErrorReport(errorCode, GetApplet());
        if( result.IsFailure() )
        {
            NN_AM_SERVICE_LOG(error, "Failed to create a visible error report : 0x%08x\n", result.GetInnerValueForDebug());
        }
        NN_RESULT_SUCCESS;
    }

    Result ReportVisibleErrorWithErrorContext(nn::err::ErrorCode errorCode, nn::err::ErrorContext errorContext) NN_NOEXCEPT
    {
        auto result = am::service::MakeVisibleErrorReport(errorCode, errorContext, GetApplet());
        if( result.IsFailure() )
        {
            NN_AM_SERVICE_LOG(error, "Failed to create a visible error report : 0x%08x\n", result.GetInnerValueForDebug());
        }
        NN_RESULT_SUCCESS;
    }

    Result CreateGameMovieTrimmer(sf::Out<sf::SharedPointer<grcsrv::IGameMovieTrimmer>> pOut, sf::NativeHandle&& workMemoryHandle, uint64_t workMemorySize) NN_NOEXCEPT
    {
        sf::SharedPointer<grcsrv::IGameMovieTrimmer> p;
        if (m_GrcOccupier.HasOccupation())
        {
            NN_RESULT_DO(service::CreateGameMovieTrimmer(&p, &m_GrcOccupier, std::move(workMemoryHandle), workMemorySize));
        }
        else
        {
            GrcOccupier grcOccupier{true};
            NN_RESULT_THROW_UNLESS(grcOccupier.HasOccupation(), am::ResultResourceForMovieOperationInUse());
            NN_RESULT_DO(service::CreateGameMovieTrimmer(&p, &grcOccupier, std::move(workMemoryHandle), workMemorySize));
        }
        *pOut = std::move(p);
        NN_RESULT_SUCCESS;
    }

    Result ReserveResourceForMovieOperation() NN_NOEXCEPT
    {
        if (!m_GrcOccupier.HasOccupation())
        {
            GrcOccupier grcOccupier{true};
            NN_RESULT_THROW_UNLESS(grcOccupier.HasOccupation(), am::ResultResourceForMovieOperationInUse());
            this->m_GrcOccupier = std::move(grcOccupier);
        }
        NN_RESULT_SUCCESS;
    }

    Result UnreserveResourceForMovieOperation() NN_NOEXCEPT
    {
        GrcOccupier tmp = std::move(m_GrcOccupier);
        NN_RESULT_SUCCESS;
    }

private:

    IntegratedLibraryApplet* GetApplet() NN_NOEXCEPT
    {
        return static_cast<IntegratedLibraryApplet*>(AppletSelfProxyImpl::GetApplet());
    }

    GrcOccupier m_GrcOccupier;

};

IntegratedLibraryApplet::AppletProxyInfo IntegratedLibraryApplet::DoCreateAppletProxy(os::ProcessId processId) NN_NOEXCEPT
{
    return MakeAppletProxyInfo<LibraryAppletProxy>(MakeShared<LibraryAppletSelfProxyImpl>(SharedFromThis(), processId));
}

}}}
