﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/am_IntegratedApplet.h>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <cstdlib>
#include <utility>
#include <memory>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/am/service/am_ErrorReport.h>
#include <nn/am/service/am_PlayDataManager.h>
#include <nn/am/service/am_ServiceStaticAllocator.h>
#include <nn/applet/applet_Result.h>
#include <nn/os/os_Tick.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/am/am_ResultPrivate.h>
#include <nn/am/service/am_Storage.h>
#include <nn/am/service/am_SystemProgramIds.h>
#include <nn/ae/ae_Types.h>
#include <nn/omm/omm_Api.h>
#include <nn/apm/apm_Api.h>
#include <nn/oe/oe_OperationModeApis.h>
#include <nn/oe/oe_HdcpApis.private.h>
#include <nn/pm/pm_BootModeApi.h>
#include <nn/util/util_IntUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_UniqueLock.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_TerminateResultApi.h>
#include <nn/am/am_Result.h>
#include <nn/lr/lr_Result.h>
#include <nn/svc/svc_Result.h>
#include <nn/fs/fs_Result.h>
#include <nn/am/service/am_StuckChecker.h>
#include <nn/am/service/am_ContentActionTable.h>

#include <mutex>
#include <list>
#include <cstring>

namespace nn { namespace am { namespace service {

IntegratedApplet::IntegratedApplet(AppletSystem* pAppletSystem, bool hasWindow) NN_NOEXCEPT
    : m_pAppletSystem(pAppletSystem)
    , m_WindowShouldCreatedAsActive(hasWindow)
    , m_pWindowGroup(pAppletSystem->GetWindowManager()->CreateWindowGroup())
    , m_pRightsEnvironment(MakeShared<rightsManagement::NoCheckRightsEnvironment>())
{
}

void IntegratedApplet::SetParent(std::shared_ptr<IntegratedApplet> pParent) NN_NOEXCEPT
{
    this->m_pParent = std::move(pParent);
    if (m_pParent)
    {
        m_pParent->AddChild(this->SharedFromThis());
        this->m_pWindowGroup = m_pParent->m_pWindowGroup;
        this->m_WindowSubOrder = m_pParent->m_WindowSubOrder + 1;
        if (m_pParent->IsExitRequested())
        {
            this->RequestExit();
        }
    }
}

void IntegratedApplet::SetFatalResultOnExit(Result result) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_StateMutex)> lk(m_StateMutex);
    this->m_FatalResultOnExit = result;
}

void IntegratedApplet::TerminateWithResult(Result terminateResult) NN_NOEXCEPT
{
    {
        std::unique_lock<decltype(m_StateMutex)> lk(m_StateMutex);
        if (!m_TerminateResult.IsSuccess())
        {
            // 既に terminate されている場合、その要因を保持し、後に来たものは無視
            return;
        }
        this->m_TerminateResult = terminateResult;
    }
    static_cast<core::Applet*>(this)->Terminate();
}

Result IntegratedApplet::GetIntegratedResult() NN_NOEXCEPT
{
    NN_RESULT_TRY(static_cast<core::Applet*>(this)->GetResult())
        NN_RESULT_CATCH(ResultAppletTerminatedInternal)
        {
            // nop
        }
    NN_RESULT_END_TRY
    {
        std::unique_lock<decltype(m_StateMutex)> lk(m_StateMutex);
        NN_RESULT_DO(m_TerminateResult);
        NN_RESULT_THROW_UNLESS(!m_ExitLocked, am::ResultAppletExitAbnormally());
    }
    NN_RESULT_SUCCESS;
}

void IntegratedApplet::UpdateExitImpl(std::unique_lock<decltype(m_StateMutex)> lk) NN_NOEXCEPT
{
    NN_SDK_ASSERT(lk.owns_lock());
    if (m_ExitRequested && !m_ExitLocked)
    {
        lk.unlock();
        KillProcess();
    }
}

void IntegratedApplet::LockExit() NN_NOEXCEPT
{
    std::unique_lock<decltype(m_StateMutex)> lk(m_StateMutex);
    UpdateExitImpl(std::move(lk));
    this->m_ExitLocked = true;
}

void IntegratedApplet::UnlockExit() NN_NOEXCEPT
{
    std::unique_lock<decltype(m_StateMutex)> lk(m_StateMutex);
    this->m_ExitLocked = false;
    UpdateExitImpl(std::move(lk));
}

void IntegratedApplet::UpdateExit() NN_NOEXCEPT
{
    std::unique_lock<decltype(m_StateMutex)> lk(m_StateMutex);
    UpdateExitImpl(std::move(lk));
}

bool IntegratedApplet::IsExitRequested() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_StateMutex)> lk(m_StateMutex);
    return m_ExitRequested;
}

bool IntegratedApplet::IsDormant() NN_NOEXCEPT
{
    auto pSelfProxy = m_pSelfProxy.lock();
    if (!pSelfProxy)
    {
        return true;
    }
    return pSelfProxy->IsSuspended();
}

bool IntegratedApplet::IsExitLocked() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_StateMutex)> lk(m_StateMutex);
    return m_ExitLocked;
}

inline void IntegratedApplet::AddAccessor(AppletAccessorImpl* p) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_StateMutex)> lk(m_StateMutex);
    m_List.push_back(*p);
    if (m_IsCompleted)
    {
        p->m_Event.Signal();
    }
}

inline void IntegratedApplet::RemoveAccessor(AppletAccessorImpl* p) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_StateMutex)> lk(m_StateMutex);
    m_List.erase(m_List.iterator_to(*p));
}

bool IntegratedApplet::IsCompleted() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_StateMutex)> lk(m_StateMutex);
    return m_IsCompleted;
}

void IntegratedApplet::SetDesirableKeyboardLayoutImpl(Bit32 layout) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_DesirableKeyboardLayoutMutex)> lk(m_DesirableKeyboardLayoutMutex);
    this->m_pDesirableKeyboardLayout = layout;
}

bool IntegratedApplet::GetDesirableKeyboardLayoutImpl(Bit32* pOutLayout) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_DesirableKeyboardLayoutMutex)> lk(m_DesirableKeyboardLayoutMutex);
    if (!m_pDesirableKeyboardLayout)
    {
        return false;
    }
    *pOutLayout = *m_pDesirableKeyboardLayout;
    return true;
}

Result IntegratedApplet::CreateAppletProxy(std::shared_ptr<AppletProxy>* pOut, os::ProcessId processId) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_StateMutex)> lk(m_StateMutex);
    if (m_pSelfProxy.lock())
    {
        NN_RESULT_THROW(ResultInvalidCall());
    }
    auto proxyInfo = DoCreateAppletProxy(processId);
    auto pSelfProxy = std::get<1>(proxyInfo);
    if (m_ExitRequested)
    {
        pSelfProxy->RequestExit();
    }
    this->m_pSelfProxy = std::move(pSelfProxy);
    *pOut = std::move(std::get<0>(proxyInfo));
    NN_RESULT_SUCCESS;
}

void IntegratedApplet::SetProcessHandle(sf::NativeHandle processHandle) NN_NOEXCEPT
{
    auto pSelfProxy = m_pSelfProxy.lock();
    pSelfProxy->SetProcessHandle(std::move(processHandle));
}

void IntegratedApplet::ActivateWindow(const ProxyOption& proxyOption) NN_NOEXCEPT
{
    auto pSelfProxy = m_pSelfProxy.lock();
    pSelfProxy->ActivateWindow(proxyOption);
}

IntegratedApplet::AppletAccessorImpl::AppletAccessorImpl(std::shared_ptr<IntegratedApplet> p) NN_NOEXCEPT
    : m_P(std::move(p))
{
    GetApplet()->AddAccessor(this);
}

IntegratedApplet::AppletAccessorImpl::~AppletAccessorImpl() NN_NOEXCEPT
{
    if (m_IndirectLayerConsumerHandle)
    {
        DestroyIndirectLayerConsumerEndPoint(GetApplet()->GetIndirectLayerHandle(), m_IndirectLayerConsumerHandle);
    }
    // アクセッサの解放時に Join を試行し、Join に成功した場合には、親の掃除をする
    auto parent = GetApplet()->GetParent();
    if (parent)
    {
        parent->CleanJoinableChildren();
    }
    GetApplet()->RemoveAccessor(this);
    // さらに自身の削除も試みる
    auto pAppletSystem = GetApplet()->GetAppletSystem();
    this->m_P.reset();
    pAppletSystem->CleanApplets();
}

Result IntegratedApplet::AppletAccessorImpl::GetAppletStateChangedEvent(sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT
{
    *pOut = sf::NativeHandle(m_Event.GetReadableHandle(), false);
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletAccessorImpl::IsCompleted(nn::sf::Out<bool> pOut) NN_NOEXCEPT
{
    *pOut = GetApplet()->IsCompleted();
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletAccessorImpl::Start() NN_NOEXCEPT
{
    OnStartByAccessor();
    static_cast<core::Applet*>(GetApplet())->Start();
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletAccessorImpl::RequestExit() NN_NOEXCEPT
{
    GetApplet()->RequestExit();
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletAccessorImpl::Terminate() NN_NOEXCEPT
{
    GetApplet()->TerminateWithResult(am::ResultAppletTerminatedManually());
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletAccessorImpl::GetResult() NN_NOEXCEPT
{
    return GetApplet()->GetIntegratedResult();
}

Result IntegratedApplet::AppletAccessorImpl::GetIntegratedAppletPointer(sf::Out<IntegratedAppletPointer> p) NN_NOEXCEPT
{
    *p = m_P;
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletAccessorImpl::GetIndirectLayerConsumerHandle(sf::Out<vi::IndirectConsumerHandleType> pOut, applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    auto layerHandle = GetApplet()->GetIndirectLayerHandle();
    NN_RESULT_THROW_UNLESS(layerHandle, am::ResultInvalidCall());
    if (m_IndirectLayerConsumerHandle)
    {
        DestroyIndirectLayerConsumerEndPoint(layerHandle, m_IndirectLayerConsumerHandle);
    }
    decltype(m_IndirectLayerConsumerHandle) ret;
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateIndirectLayerConsumerEndPoint(&ret, layerHandle, aruid));
    this->m_IndirectLayerConsumerHandle = ret;
    *pOut = ret;
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::Before() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(CallingWindableApplet::Before());
    m_pRightsEnvironment->Start();
    CreateWindow();
    SendPlayLog(pdm::AppletEventType::Launch);
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::After() NN_NOEXCEPT
{
    if (auto p = GetLastLibraryApplet())
    {
        // SIGLO-39918:
        // - 起動中の LA がいれば 1 秒 ExitRequest を送って待つ
        // SIGLO-74305:
        // - 待機時間を 1 秒から 10 秒に変更
        p->RequestExit();
        auto endTime = os::GetSystemTick().ToTimeSpan() + TimeSpan::FromMilliSeconds(10000);
        bool isSucceeded = false;
        while (os::GetSystemTick().ToTimeSpan() < endTime)
        {
            auto n = os::TimedWaitAny(TimeSpan::FromMilliSeconds(100), static_cast<core::Applet*>(p.get()));
            if (n == 0 && static_cast<core::Applet*>(p.get())->TryJoin())
            {
                isSucceeded = true;
                break;
            }
        }
        if (!isSucceeded)
        {
            ncm::ProgramId programId = {};
            p->MakeProgramIdFromAppletIdentityInfo(&programId, p->GetAppletIdentityInfo());
            NN_AM_SERVICE_APPLET_LOG(error, this, "Program(0x%016llx) does not exit for long time.", programId);
            // SIGLO-74305:
            // このまま放置しても復旧できる見込みが薄いのでアボートさせる。
            NN_ABORT_UNLESS_RESULT_SUCCESS(am::ResultDetectLibraryAppletNotExited());
        }
    }
    return CallingWindableApplet::After();
}


void IntegratedApplet::Cleanup() NN_NOEXCEPT
{
    std::unique_lock<decltype(m_StateMutex)> lk(m_StateMutex);
    if (m_IsFatalResultOnExitValid)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_FatalResultOnExit);
    }
    NN_SDK_ASSERT(!m_IsCompleted);
    this->m_IsCompleted = true;
    for (auto&& e : m_List)
    {
        e.m_Event.Signal();
    }
    lk.unlock();
    m_ReservedLibraryAppletAccessor.Reset();
    SendPlayLog(pdm::AppletEventType::Exit);
    DestroyWindow();
    m_pRightsEnvironment->Stop();
    m_pRightsEnvironment.reset();
    CallingWindableApplet::Cleanup();
}

void IntegratedApplet::RunBehindProcess(process::NsProcess* pProcess) NN_NOEXCEPT
{
    RunBehindProcessForRights(pProcess);
}

void IntegratedApplet::PushMessageToProcess(const am::AppletMessage& message) NN_NOEXCEPT
{
    std::unique_lock<decltype(m_StateMutex)> lk(m_StateMutex);
    auto pSelfProxy = m_pSelfProxy.lock();
    lk.unlock();
    if (pSelfProxy)
    {
        pSelfProxy->PushMessage(message);
    }
}

Result IntegratedApplet::ReserveToStartAndWaitAndUnwindThis(sf::SharedPointer<am::service::ILibraryAppletAccessor> p) NN_NOEXCEPT
{
    IntegratedAppletPointer a;
    NN_ABORT_UNLESS_RESULT_SUCCESS(p->GetIntegratedAppletPointer(&a));
    CallingWindableApplet::ReserveToStartAndWaitAndUnwindThis(a);
    // Winding で LA を起動する場合、IntegratedApplet::AccessorImpl::Start()
    // が呼ばれないため、最終 LA ハンドルの登録はここで行っておく。
    this->SetLastLibraryApplet(std::move(a));
    this->m_ReservedLibraryAppletAccessor = std::move(p);
    this->m_IsReservedWithoutUnwinding = false;
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::ReserveToStartAndWait(sf::SharedPointer<am::service::ILibraryAppletAccessor> p) NN_NOEXCEPT
{
    IntegratedAppletPointer a;
    NN_ABORT_UNLESS_RESULT_SUCCESS(p->GetIntegratedAppletPointer(&a));
    CallingWindableApplet::ReserveToStartAndWait(a);
    // 最終 LA ハンドルの登録はここで行っておく。
    this->SetLastLibraryApplet(std::move(a));
    this->m_ReservedLibraryAppletAccessor = std::move(p);
    this->m_IsReservedWithoutUnwinding = true;
    NN_RESULT_SUCCESS;
}

sf::SharedPointer<am::service::ILibraryAppletAccessor> IntegratedApplet::GetReserved() NN_NOEXCEPT
{
    return std::move(m_ReservedLibraryAppletAccessor);
}

void IntegratedApplet::OnProcessBegin(os::ProcessId processId) NN_NOEXCEPT
{
    CallingWindableApplet::OnProcessEnd(processId);
    UpdateExit();
    std::unique_lock<decltype(m_StateMutex)> lk(m_StateMutex);
    this->m_IsFatalResultOnExitValid = true;
}

void IntegratedApplet::OnProcessEnd(os::ProcessId processId) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, this, "unregister proxy process id = %lld", processId.value);
    GetAppletSystem()->UnregisterAppletProxy(processId.value);
    AfterDeactivateProxy(processId);
    {
        std::unique_lock<decltype(m_StateMutex)> lk(m_StateMutex);
        if (m_ExitRequested)
        {
            lk.unlock();
            CallingWindableApplet::ClearReservation();
        }
    }
    CallingWindableApplet::OnProcessEnd(processId);
}

void IntegratedApplet::RequestExit() NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, this, "IntegratedApplet::RequestExit()");
    auto pChild = GetLastCreatedLibraryApplet();
    if (pChild)
    {
        NN_AM_SERVICE_APPLET_LOG(seq, this, "RequestExit to child: " NN_AM_SERVICE_DIAGNOSTICS_APPLET_STRING_FORMAT, NN_AM_SERVICE_DIAGNOSTICS_APPLET_STRING_FORMAT_PARAMETER(pChild));
        pChild->RequestExit();
    }
    std::unique_lock<decltype(m_StateMutex)> lk(m_StateMutex);
    this->m_ExitRequested = true;
    if (!m_ExitLocked)
    {
        lk.unlock();
        NN_ABORT_UNLESS_RESULT_SUCCESS(KillProcess());
        return;
    }
    auto pSelfProxy = m_pSelfProxy.lock();
    lk.unlock();
    if (pSelfProxy)
    {
        pSelfProxy->RequestExit();
    }
}

bool IntegratedApplet::IsExitRequested() NN_NOEXCEPT
{
    std::unique_lock<decltype(m_StateMutex)> lk(m_StateMutex);
    return m_ExitRequested;
}

Result IntegratedApplet::AppletSelfProxyImpl::GetThisAppletKind(sf::Out<am::service::AppletKind> pOut) NN_NOEXCEPT
{
    NN_UNUSED(pOut);
    NN_AM_SERVICE_NOT_IMPLEMENTED("TODO: implement: IntegratedApplet::AppletSelfProxyImpl::GetThisAppletKind()");
}

Result IntegratedApplet::AppletSelfProxyImpl::AllowToEnterSleep() NN_NOEXCEPT
{
    NN_AM_SERVICE_NOT_IMPLEMENTED("TODO: implement: IntegratedApplet::AppletSelfProxyImpl::AllowToEnterSleep()");
}

Result IntegratedApplet::AppletSelfProxyImpl::DisallowToEnterSleep() NN_NOEXCEPT
{
    NN_AM_SERVICE_NOT_IMPLEMENTED("TODO: implement: IntegratedApplet::AppletSelfProxyImpl::DisallowToEnterSleep()");
}

Result IntegratedApplet::AppletSelfProxyImpl::GetOperationMode(sf::Out<Bit8> pOut) NN_NOEXCEPT
{
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(omm_Get, 60);
    *pOut = static_cast<Bit8>(nn::omm::GetOperationMode());
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::GetDefaultDisplayResolution(sf::Out<int> pOutWidth, sf::Out<int> pOutHeight) NN_NOEXCEPT
{
    nn::omm::GetDefaultDisplayResolution(pOutWidth.GetPointer(), pOutHeight.GetPointer());
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::GetDefaultDisplayResolutionChangeEvent(sf::Out<sf::NativeHandle> outValue) NN_NOEXCEPT
{
    os::SystemEvent event;
    nn::omm::GetDefaultDisplayResolutionChangeEvent(&event);

    // SIGLO-69853:
    // 第 2 引数は本来、event.DetachReadableHandle() する前の
    // event.GetBase()->_interProcessEvent._isReadableHandleManaged の値を
    // 取得して渡すのが正しいが、今はそういった関数も用意されていないので、
    // とりあえず true にしておく。omm が仮に HIPC から DFC に変更された場合、
    // 上記のように実装しないと、再び別の問題が起こる。
    *outValue = sf::NativeHandle(event.DetachReadableHandle(), true);
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::GetHdcpAuthenticationState(sf::Out<int> outValue) NN_NOEXCEPT
{
    omm::HdcpState hdcpState = omm::GetHdcpState();
    *outValue = oe::HdcpAuthenticationState_Unauthenticated;
    switch (hdcpState)
    {
    case omm::HdcpState::ProcessingForeground:
    case omm::HdcpState::ProcessingBackground:
        *outValue = oe::HdcpAuthenticationState_Processing;
        break;

    case omm::HdcpState::Authenticated:
        *outValue = oe::HdcpAuthenticationState_Authenticated;
        break;

    default:
        break;
    }
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::GetHdcpAuthenticationStateChangeEvent(sf::Out<sf::NativeHandle> outValue) NN_NOEXCEPT
{
    os::SystemEvent event;
    omm::GetHdcpStateChangeEvent(&event);
    // SIGLO-69853:
    //  IntegratedApplet::AppletSelfProxyImpl::GetDefaultDisplayResolutionChangeEvent() のコメントを参照
    *outValue = sf::NativeHandle(event.DetachReadableHandle(), true);
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::GetCradleStatus(sf::Out<Bit8> pOut) NN_NOEXCEPT
{
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(omm_Get, 60);
    *pOut = static_cast<Bit8>(nn::omm::GetCradleStatus());
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::GetCradleFwVersion(nn::sf::Out<std::uint32_t> pOutPdcH, nn::sf::Out<std::uint32_t> pOutPdcA, nn::sf::Out<std::uint32_t> pOutMcu, nn::sf::Out<std::uint32_t> pOutDp2Hdmi) NN_NOEXCEPT
{
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(omm_Get, 60);
    return nn::omm::GetCradleFwVersion(pOutPdcH.GetPointer(), pOutPdcA.GetPointer(), pOutMcu.GetPointer(), pOutDp2Hdmi.GetPointer());
}

Result IntegratedApplet::AppletSelfProxyImpl::GetBootMode(sf::Out<Bit8> pOut) NN_NOEXCEPT
{
    *pOut = static_cast<Bit8>(nn::pm::GetBootMode());
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::GetPerformanceMode(sf::Out<Bit32> pOut) NN_NOEXCEPT
{
    *pOut = static_cast<Bit32>(nn::apm::GetPerformanceMode());
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::Exit() NN_NOEXCEPT
{
    return GetApplet()->KillProcess();
}


Result IntegratedApplet::AppletSelfProxyImpl::LockExit() NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "LockExit()");
    GetApplet()->LockExit();
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::UnlockExit()NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "UnlockExit()");
    GetApplet()->UnlockExit();
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::SetScreenShotPermission(int permission) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(permission == applet::ScreenShotPermission_Inherit ||
                           permission == applet::ScreenShotPermission_Permit  ||
                           permission == applet::ScreenShotPermission_Forbid,
                           applet::ResultInvalidParameter());

    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "SetScreenShotPermission(%s)", permission == applet::ScreenShotPermission_Inherit ? "Inherit" : (permission == applet::ScreenShotPermission_Permit ? "Permit" : "Forbid"));
    auto result = GetApplet()->SetScreenShotPermissionImpl(static_cast<applet::ScreenShotPermission>(permission));
    this->OnGamePlayRecordingStateChangedImpl();
    return result;
}

Result IntegratedApplet::AppletSelfProxyImpl::CreateLibraryApplet(sf::Out<sf::SharedPointer<am::service::ILibraryAppletAccessor>> pOut, Bit32 appletId, uint32_t libraryAppletMode) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "LA Create Start: %lld ms, AppletId=0x%08x", os::GetSystemTick().ToTimeSpan().GetMilliSeconds(), appletId);
    sf::SharedPointer<am::service::ILibraryAppletAccessor> p;
    NN_RESULT_DO(GetApplet()->GetAppletSystem()->CreateLibraryApplet(&p, appletId, static_cast<applet::LibraryAppletMode>(libraryAppletMode), GetApplet()->SharedFromThis()));
    {
        IntegratedAppletPointer pApplet;
        NN_ABORT_UNLESS_RESULT_SUCCESS(p->GetIntegratedAppletPointer(&pApplet));
        GetApplet()->SetLastCreatedLibraryApplet(std::move(pApplet));
    }
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::TerminateAllLibraryApplets() NN_NOEXCEPT
{
    // Terminate 時に対象アプレットがキャプチャバッファを MapTransferMemory
    // していないことを保証するため、キャプチャバッファセマフォを取得する。
    // ResultCaptureBufferBusy 時は shim 側でリトライする。
    if (!am::service::TryAcquireCaptureBufferSemaphore())
    {
        return applet::ResultCaptureBufferBusy();
    }
    NN_UTIL_SCOPE_EXIT
    {
        am::service::ReleaseCaptureBufferSemaphore();
    };

    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "TerminateAllLibraryApplets()");
    GetApplet()->TerminateAllChildren();
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::AreAnyLibraryAppletsLeft(sf::Out<bool> pOut) NN_NOEXCEPT
{
    auto isLeft = GetApplet()->AreAnyChildrenLeft();
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "AreAnyLibraryAppletsLeft() = %s", (isLeft ? "true" : "false"));
    *pOut = isLeft;
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::CreateStorage(sf::Out<sf::SharedPointer<am::service::IStorage>> pOut, std::int64_t size) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(util::IsIntValueRepresentable<size_t>(size), am::ResultInvalidCall());
    return MakeSimpleMemoryStorage(pOut, static_cast<size_t>(size));
}

Result IntegratedApplet::AppletSelfProxyImpl::CreateTransferMemoryStorage(sf::Out<sf::SharedPointer<am::service::IStorage>> pOut, sf::NativeHandle transferMemory, int64_t size, bool isWritable) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(util::IsIntValueRepresentable<size_t>(size), am::ResultInvalidCall());
    return MakeLargeStorage(pOut, std::move(transferMemory), static_cast<size_t>(size), isWritable);
}

Result IntegratedApplet::AppletSelfProxyImpl::CreateHandleStorage(sf::Out<sf::SharedPointer<am::service::IStorage>> pOut, sf::NativeHandle transferMemory, int64_t size) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(util::IsIntValueRepresentable<size_t>(size), am::ResultInvalidCall());
    return MakeHandleStorage(pOut, std::move(transferMemory), static_cast<size_t>(size));
}

Result IntegratedApplet::AppletSelfProxyImpl::CreateWindow(sf::Out<sf::SharedPointer<am::service::IWindow>> pOut, am::service::WindowCreationOption option) NN_NOEXCEPT
{
    NN_UNUSED(pOut);
    NN_UNUSED(option);
    NN_AM_SERVICE_NOT_IMPLEMENTED("TODO: implement: IntegratedApplet::AppletSelfProxyImpl::CreateWindow()");
}

Result IntegratedApplet::AppletSelfProxyImpl::AcquireForegroundRights() NN_NOEXCEPT
{
    // TODO: implement: IntegratedApplet::AppletSelfProxyImpl::AcquireForegroundRights()
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::ReleaseForegroundRights() NN_NOEXCEPT
{
    //TODO: implement: IntegratedApplet::AppletSelfProxyImpl::ReleaseForegroundRights()
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::RejectToChangeIntoBackground() NN_NOEXCEPT
{
    // TODO: implement: IntegratedApplet::AppletSelfProxyImpl::RejectToChangeIntoBackground()
    NN_AM_SERVICE_APPLET_LOG(error, GetApplet(), "called unimplemented: IntegratedApplet::AppletSelfProxyImpl::RejectToChangeIntoBackground");
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::SetExpectedMasterVolume(float mainAppletVolume, float libraryAppletVolume) NN_NOEXCEPT
{
    auto p = GetApplet()->GetMainApplet();
    std::lock_guard<decltype(p->m_ExpectedVolumeMutex)> lk(p->m_ExpectedVolumeMutex);
    p->m_ExpectedVolumeForMainApplet = mainAppletVolume;
    p->m_ExpectedVolumeForLibraryApplet = libraryAppletVolume;
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::GetMainAppletExpectedMasterVolume(sf::Out<float> pOutVolume) NN_NOEXCEPT
{
    auto p = GetApplet()->GetMainApplet();
    std::lock_guard<decltype(p->m_ExpectedVolumeMutex)> lk(p->m_ExpectedVolumeMutex);
    *pOutVolume = p->m_ExpectedVolumeForMainApplet;
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::GetLibraryAppletExpectedMasterVolume(sf::Out<float> pOutVolume) NN_NOEXCEPT
{
    auto p = GetApplet()->GetMainApplet();
    std::lock_guard<decltype(p->m_ExpectedVolumeMutex)> lk(p->m_ExpectedVolumeMutex);
    *pOutVolume = p->m_ExpectedVolumeForLibraryApplet;
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::ChangeMainAppletMasterVolume(float volume, std::int64_t fadeTimeInNanoSeconds) NN_NOEXCEPT
{
    // 実際の変更動作を開始
    GetApplet()->GetMainApplet()->SetNextVolumeChangeFadeTime(volume, TimeSpan::FromNanoSeconds(fadeTimeInNanoSeconds));
    GetApplet()->SetMainAppletWindowVolume(volume);
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::SetTransparentVolumeRate(float transparentRate) NN_NOEXCEPT
{
    GetApplet()->SetTransparentVolumeRate(transparentRate);
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::GetLastApplicationCaptureImage(sf::OutBuffer pOutBuffer) NN_NOEXCEPT
{
    // 本関数を呼び出す shim も既に存在しないため ResultNotSupported を返す
    NN_UNUSED(pOutBuffer);
    return ResultNotSupported();
}

Result IntegratedApplet::AppletSelfProxyImpl::GetLastApplicationCaptureImageEx(sf::Out<bool> pOutIsScreenShotEnabled, sf::OutBuffer pOutBuffer) NN_NOEXCEPT
{
    auto dstAddr = pOutBuffer.GetPointerUnsafe();
    auto dstSize = pOutBuffer.GetSize();
    NN_RESULT_THROW_UNLESS(dstSize == ae::CaptureBufferSize, applet::ResultInvalidParameter());

    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "Invoke CopyFromCaptureBuffer(index=%d)", am::service::CaptureBufferIndex_LastApplication);
    return am::service::CopyImageFromCaptureBuffer(&*pOutIsScreenShotEnabled, dstAddr, dstSize, am::service::CaptureBufferIndex_LastApplication);
}

Result IntegratedApplet::AppletSelfProxyImpl::GetLastForegroundCaptureImage(sf::OutBuffer pOutBuffer) NN_NOEXCEPT
{
    // 本関数を呼び出す shim も既に存在しないため ResultNotSupported を返す
    NN_UNUSED(pOutBuffer);
    return ResultNotSupported();
}

Result IntegratedApplet::AppletSelfProxyImpl::GetLastForegroundCaptureImageEx(sf::Out<bool> pOutIsScreenShotEnabled, sf::OutBuffer pOutBuffer) NN_NOEXCEPT
{
    auto dstAddr = pOutBuffer.GetPointerUnsafe();
    auto dstSize = pOutBuffer.GetSize();
    NN_RESULT_THROW_UNLESS( dstSize == ae::CaptureBufferSize, applet::ResultInvalidParameter() );

    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "Invoke CopyFromCaptureBuffer(index=%d)", am::service::CaptureBufferIndex_LastForeground);
    return am::service::CopyImageFromCaptureBuffer(&*pOutIsScreenShotEnabled, dstAddr, dstSize, am::service::CaptureBufferIndex_LastForeground);
}

Result IntegratedApplet::AppletSelfProxyImpl::UpdateLastForegroundCaptureImage() NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "Invoke TryUpdateCaptureImage(index=%d)", am::service::CaptureBufferIndex_LastForeground);
    return am::service::TryCaptureCurrentForegroundImage(am::service::CaptureBufferIndex_LastForeground);
}

Result IntegratedApplet::AppletSelfProxyImpl::GetCallerAppletCaptureImage(sf::OutBuffer pOutBuffer) NN_NOEXCEPT
{
    // 本関数を呼び出す shim も既に存在しないため ResultNotSupported を返す
    NN_UNUSED(pOutBuffer);
    return ResultNotSupported();
}

Result IntegratedApplet::AppletSelfProxyImpl::GetCallerAppletCaptureImageEx(sf::Out<bool> pOutIsScreenShotEnabled, sf::OutBuffer pOutBuffer) NN_NOEXCEPT
{
    auto dstAddr = pOutBuffer.GetPointerUnsafe();
    auto dstSize = pOutBuffer.GetSize();
    NN_RESULT_THROW_UNLESS( dstSize == ae::CaptureBufferSize, applet::ResultInvalidParameter() );

    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "Invoke CopyFromCaptureBuffer(index=%d)", am::service::CaptureBufferIndex_CallerApplet);
    return am::service::CopyImageFromCaptureBuffer(&*pOutIsScreenShotEnabled, dstAddr, dstSize, am::service::CaptureBufferIndex_CallerApplet);
}

Result IntegratedApplet::AppletSelfProxyImpl::UpdateCallerAppletCaptureImage() NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "Invoke TryUpdateCaptureImage(index=%d)", am::service::CaptureBufferIndex_CallerApplet);
    return am::service::TryCaptureCurrentForegroundImage(am::service::CaptureBufferIndex_CallerApplet);
}

Result IntegratedApplet::AppletSelfProxyImpl::CopyBetweenCaptureBuffers(int dstIndex, int srcIndex) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "Invoke CopyBetweenCaptureBuffers(dstIndex=%d, srcIndex=%d)", dstIndex, srcIndex);
    NN_RESULT_THROW_UNLESS(dstIndex >= CaptureBufferIndex_LastApplication && dstIndex < CaptureBufferIndex_Max, am::ResultInvalidParameter());
    NN_RESULT_THROW_UNLESS(srcIndex >= CaptureBufferIndex_LastApplication && srcIndex < CaptureBufferIndex_Max, am::ResultInvalidParameter());

    return am::service::CopyBetweenCaptureBuffersImpl(static_cast<CaptureBufferIndex>(dstIndex), static_cast<CaptureBufferIndex>(srcIndex));
}

Result IntegratedApplet::AppletSelfProxyImpl::AcquireLastApplicationCaptureBuffer(sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT
{
    // 本関数を呼び出す shim も既に存在しないため ResultNotSupported を返す
    NN_UNUSED(pOut);
    return ResultNotSupported();
}

// キャプチャバッファの TransferMemory 対応
Result IntegratedApplet::AppletSelfProxyImpl::AcquireLastApplicationCaptureBufferEx(sf::Out<bool> pOutIsScreenShotEnabled, sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT
{
    // 自アプレットで既に Map 済みなら am::ResultInvalidCall 扱いとする。
    // 他アプレットで既に Map 済みならマップ失敗の Result を返す。
    NN_RESULT_THROW_UNLESS(!m_IsLastApplicationCaptureBufferOwner, am::ResultCaptureBufferAlreadyMapped());

    auto result = am::service::AcquireCaptureBufferTransferMemory(&*pOutIsScreenShotEnabled, pOut, am::service::CaptureBufferIndex_LastApplication);
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "MapCaptureBuffer(index=%d) result=0x%08x", am::service::CaptureBufferIndex_LastApplication, result.GetInnerValueForDebug());
    NN_RESULT_DO(result);

    m_IsLastApplicationCaptureBufferOwner = true;
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::ReleaseLastApplicationCaptureBuffer() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsLastApplicationCaptureBufferOwner, am::ResultCaptureBufferNotMapped());
    m_IsLastApplicationCaptureBufferOwner = false;

    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "UnmapCaptureBuffer(index=%d)", am::service::CaptureBufferIndex_LastApplication);
    am::service::ReleaseCaptureBufferTransferMemory(am::service::CaptureBufferIndex_LastApplication);
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::AcquireLastForegroundCaptureBuffer(sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT
{
    // 本関数を呼び出す shim も既に存在しないため ResultNotSupported を返す
    NN_UNUSED(pOut);
    return ResultNotSupported();
}

Result IntegratedApplet::AppletSelfProxyImpl::AcquireLastForegroundCaptureBufferEx(sf::Out<bool> pOutIsScreenShotEnabled, sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT
{
    // 自アプレットで既に Map 済みなら am::ResultInvalidCall 扱いとする。
    // 他アプレットで既に Map 済みならマップ失敗の Result を返す。
    NN_RESULT_THROW_UNLESS(!m_IsLastForegroundCaptureBufferOwner, am::ResultCaptureBufferAlreadyMapped());

    auto result = am::service::AcquireCaptureBufferTransferMemory(&*pOutIsScreenShotEnabled, pOut, am::service::CaptureBufferIndex_LastForeground);
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "MapCaptureBuffer(index=%d) result=0x%08x", am::service::CaptureBufferIndex_LastForeground, result.GetInnerValueForDebug());
    NN_RESULT_DO(result);

    m_IsLastForegroundCaptureBufferOwner = true;
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::ReleaseLastForegroundCaptureBuffer() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsLastForegroundCaptureBufferOwner, am::ResultCaptureBufferNotMapped());
    m_IsLastForegroundCaptureBufferOwner = false;

    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "UnmapCaptureBuffer(index=%d)", am::service::CaptureBufferIndex_LastForeground);
    am::service::ReleaseCaptureBufferTransferMemory(am::service::CaptureBufferIndex_LastForeground);
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::AcquireCallerAppletCaptureBuffer(sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT
{
    // 本関数を呼び出す shim も既に存在しないため ResultNotSupported を返す
    NN_UNUSED(pOut);
    return ResultNotSupported();
}

Result IntegratedApplet::AppletSelfProxyImpl::AcquireCallerAppletCaptureBufferEx(sf::Out<bool> pOutIsScreenShotEnabled, sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT
{
    // 自アプレットで既に Map 済みなら am::ResultInvalidCall 扱いとする。
    // 他アプレットで既に Map 済みならマップ失敗の Result を返す。
    NN_RESULT_THROW_UNLESS(!m_IsCallerAppletCaptureBufferOwner, am::ResultCaptureBufferAlreadyMapped());

    auto result = am::service::AcquireCaptureBufferTransferMemory(&*pOutIsScreenShotEnabled, pOut, am::service::CaptureBufferIndex_CallerApplet);
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "MapCaptureBuffer(index=%d) result=0x%08x", am::service::CaptureBufferIndex_CallerApplet, result.GetInnerValueForDebug());
    NN_RESULT_DO(result);

    m_IsCallerAppletCaptureBufferOwner = true;
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::ReleaseCallerAppletCaptureBuffer() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsCallerAppletCaptureBufferOwner, am::ResultCaptureBufferNotMapped());
    m_IsCallerAppletCaptureBufferOwner = false;

    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "UnmapCaptureBuffer(index=%d)", am::service::CaptureBufferIndex_CallerApplet);
    am::service::ReleaseCaptureBufferTransferMemory(am::service::CaptureBufferIndex_CallerApplet);
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::PushToGeneralChannel(sf::SharedPointer<am::service::IStorage> storage) NN_NOEXCEPT
{
    decltype(storage) other;
    NN_RESULT_DO(storage->GetAndInvalidate(&other));
    GetApplet()->GetAppletSystem()->GetGeneralChannel()->Push(std::move(other));
    NN_AM_SERVICE_APPLET_LOG(call, GetApplet(), "%s", NN_CURRENT_FUNCTION_NAME);
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::GetHomeButtonWriterLockAccessor(sf::Out<sf::SharedPointer<ILockAccessor>> pOut) NN_NOEXCEPT
{
    *pOut = MakeSfLock(GetApplet()->GetAppletSystem()->GetHomeButtonReaderWriterLock().GetWriterLockAccessor());
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::GetHomeButtonReaderLockAccessor(sf::Out<sf::SharedPointer<ILockAccessor>> pOut) NN_NOEXCEPT
{
    *pOut = MakeSfLock(GetApplet()->GetAppletSystem()->GetHomeButtonReaderWriterLock().GetReaderLockAccessor());
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::GetWriterLockAccessorEx(sf::Out<sf::SharedPointer<ILockAccessor>> pOut, int index) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(index >= 0 && index < applet::ReaderWriterLockIndex_Max, applet::ResultInvalidParameter());
    *pOut = MakeSfLock(GetApplet()->GetAppletSystem()->GetReaderWriterLockEx(index).GetWriterLockAccessor());
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::GetReaderLockAccessorEx(sf::Out<sf::SharedPointer<ILockAccessor>> pOut, int index) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(index >= 0 && index < applet::ReaderWriterLockIndex_Max, applet::ResultInvalidParameter());
    *pOut = MakeSfLock(GetApplet()->GetAppletSystem()->GetReaderWriterLockEx(index).GetReaderLockAccessor());
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::GetApplicationIdByContentActionName(sf::Out<Bit64> pOut, const sf::InArray<char>& name) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(name[name.GetLength() - 1] == 0, am::ResultInvalidParameter());
    ncm::ApplicationId id;
    NN_RESULT_DO(am::service::contentAction::GetApplicationId(&id, name.GetData()));
    *pOut = id.value;
    NN_RESULT_SUCCESS;
}

namespace {

struct SystemProgramIdTable
{
    ncm::SystemProgramId    programId;
    applet::AppletId        appletId;
};

const SystemProgramIdTable g_SystemProgramIdTable[] =
{
      // ProgramId                         // AppletId
    { ProgramId_LibraryAppletAuth,         applet::AppletId_LibraryAppletAuth         }, // 簡易認証 LA
    { ProgramId_LibraryAppletCabinet,      applet::AppletId_LibraryAppletCabinet      }, // amiibo 設定 LA
    { ProgramId_LibraryAppletController,   applet::AppletId_LibraryAppletController   }, // コントローラー LA
    { ProgramId_LibraryAppletDataErase,    applet::AppletId_LibraryAppletDataErase    }, // データ削除 LA
    { ProgramId_LibraryAppletError,        applet::AppletId_LibraryAppletError        }, // エラービューアー LA
    { ProgramId_LibraryAppletNetConnect,   applet::AppletId_LibraryAppletNetConnect   }, // ネット接続 LA
    { ProgramId_LibraryAppletPlayerSelect, applet::AppletId_LibraryAppletPlayerSelect }, // プレイヤー選択 LA
    { ProgramId_LibraryAppletSwkbd,        applet::AppletId_LibraryAppletSwkbd        }, // ソフトウェアキーボード LA
    { ProgramId_LibraryAppletMiiEdit,      applet::AppletId_LibraryAppletMiiEdit      }, // Mii 編集 LA
    { ProgramId_LibraryAppletWeb,          applet::AppletId_LibraryAppletWeb          }, // Web LA

    { ProgramId_LibraryAppletPhotoViewer,  applet::AppletId_LibraryAppletPhotoViewer  }, // フォトビューアー LA
    { ProgramId_LibraryAppletSet,          applet::AppletId_LibraryAppletSet          }, // 本体設定 LA
    { ProgramId_LibraryAppletOfflineWeb,   applet::AppletId_LibraryAppletOfflineWeb   }, // OfflineWeb LA
    { ProgramId_LibraryAppletLoginShare,   applet::AppletId_LibraryAppletLoginShare   }, // ログインシェア LA
    { ProgramId_LibraryAppletWifiWebAuth,  applet::AppletId_LibraryAppletWifiWebAuth  }, // WiFi WebAuth  LA
    { ProgramId_LibraryAppletMyPage,       applet::AppletId_LibraryAppletMyPage       }, // マイページ LA
    { ProgramId_LibraryAppletGift,         applet::AppletId_LibraryAppletGift         }, // おすそわけ通信 LA
    { ProgramId_LibraryAppletUserMigration, applet::AppletId_LibraryAppletUserMigration }, // ユーザ移行 LA
    { ProgramId_LibraryAppletEncounter,    applet::AppletId_LibraryAppletEncounter    }, // すれちがい LA
    { ProgramId_LibraryAppletStory,        applet::AppletId_LibraryAppletStory        }, // セーブデータ同期 LA

    // 以下のものは nn::ae::SetScreenShotAppletIdentityInfo() で
    // 使われることがあるため、削除しないで下さい。
    { ProgramId_SystemAppletMenu,          applet::AppletId_SystemAppletMenu          }, // システムアプレットメニュー
    { ProgramId_SystemApplication,         applet::AppletId_SystemApplication         }, // システムアプリケーション
};

void GetFirmwareDebugSettings(ncm::SystemProgramId* pId, const char* name, const char* key, const char* title)
{
    NN_UNUSED(title);
    char idStr[32];
    char* endPtr;
    auto size = settings::fwdbg::GetSettingsItemValue(idStr, sizeof(idStr), name, key);
    NN_ABORT_UNLESS(size < sizeof(idStr));
    idStr[size] = '\0';
    ncm::ProgramId programId = { std::strtoull(idStr, &endPtr, 16) };
    NN_ABORT_UNLESS(*endPtr == '\0');

    ncm::SystemProgramId id = { programId.value };
    NN_AM_SERVICE_LOG(seq, "%s programId=0x%016llx\n", title, id.value);
    *pId = id;
}

}

// AppletId から ProgramId を検索して返す。
// キャプチャ撮影時の jpg ファイル名の元となる ProgramId にも使用される。
Result IntegratedApplet::MakeSystemProgramIdFromAppletId(ncm::SystemProgramId* pOut, applet::AppletId appletId) NN_NOEXCEPT
{
    // OA の場合 settings::fwdbg を参照する
    if (appletId == applet::AppletId_OverlayApplet)
    {
        GetFirmwareDebugSettings(pOut, "ns.applet", "overlay_applet_id", "OverlayApplet");
        NN_RESULT_SUCCESS;
    }
    // Shop の場合も settings::fwdbg を参照する
    else if (appletId == applet::AppletId_LibraryAppletShop)
    {
        GetFirmwareDebugSettings(pOut, "ns.applet", "shop_applet_id", "ShopApplet");
        NN_RESULT_SUCCESS;
    }

    // それ以外はテーブル引きする
    for (auto i = 0u; i < sizeof(g_SystemProgramIdTable) / sizeof(*g_SystemProgramIdTable); ++i)
    {
        if (g_SystemProgramIdTable[i].appletId == appletId)
        {
            *pOut = g_SystemProgramIdTable[i].programId;
            NN_RESULT_SUCCESS;
        }
    }

    NN_AM_SERVICE_LOG(seq, "Not found programId for appletId=0x%08x\n", appletId);
    return applet::ResultInvalidAppletId();
}

Result IntegratedApplet::MakeProgramIdFromAppletIdentityInfo(ncm::ProgramId* pOut, const applet::AppletIdentityInfo& info) NN_NOEXCEPT
{
    if( info.IsApplication() )
    {
        *pOut = info.applicationId;
        NN_RESULT_SUCCESS;
    }
    else if( info.IsSystemApplication() )
    {
        pOut->value = info.systemApplicationId.value;
        NN_RESULT_SUCCESS;
    }
    else
    {
        ncm::SystemProgramId programId = ncm::SystemProgramId::GetInvalidId();
        NN_RESULT_DO(MakeSystemProgramIdFromAppletId(&programId, info.appletId));
        *pOut = programId;
        NN_RESULT_SUCCESS;
    }
}

Result IntegratedApplet::CreateAppletProcessImpl(std::shared_ptr<process::NsProcess>* pOut, applet::AppletId appletId) NN_NOEXCEPT
{
    if(appletId == applet::AppletId_SystemAppletMenu)
    {
        return m_pAppletSystem->GetNsProcessManager()->MakeSystemAppletProcess(pOut);
    }

    if (appletId == applet::AppletId_OverlayApplet)
    {
        return m_pAppletSystem->GetNsProcessManager()->MakeOverlayAppletProcess(pOut);
    }

    ncm::SystemProgramId programId;
    NN_RESULT_DO(MakeSystemProgramIdFromAppletId(&programId, appletId));
    return m_pAppletSystem->GetNsProcessManager()->MakeLibraryAppletProcess(pOut, programId);
}

Result IntegratedApplet::CreateApplicationProcessImpl(std::shared_ptr<process::NsProcess>* pOut, ncm::ApplicationId applicationId, uint8_t programIndex) NN_NOEXCEPT
{
    return m_pAppletSystem->GetNsProcessManager()->MakeApplicationProcess(pOut, applicationId, programIndex);
}

Result IntegratedApplet::CreateSystemApplicationProcessImpl(std::shared_ptr<process::NsProcess>* pOut, ncm::SystemApplicationId systemApplicationId) NN_NOEXCEPT
{
    return m_pAppletSystem->GetNsProcessManager()->MakeLibraryAppletProcess(pOut, {systemApplicationId.value});
}

Result IntegratedApplet::CreateProcess(std::shared_ptr<process::NsProcess>* pOut) NN_NOEXCEPT
{
    auto retryRestTime = TimeSpan::FromMilliSeconds(1000);
    auto retrySleepTime = TimeSpan::FromMilliSeconds(10);
    auto retryCount = 0;
retry:
    if (retryCount > 0)
    {
        os::SleepThread(retrySleepTime);
        retryRestTime -= retrySleepTime;
        retrySleepTime = retrySleepTime + retrySleepTime;
    }
    if (IsExitRequested())
    {
        NN_RESULT_THROW(am::ResultAppletCancelled());
    }
    {
        // プロセス作成時にリセット
        std::lock_guard<decltype(m_StateMutex)> lk(m_StateMutex);
        this->m_ExitLocked = false;
    }
    GetAppletSystem()->BeginProcessCreation();
    NN_UTIL_SCOPE_EXIT
    {
        GetAppletSystem()->EndProcessCreation();
    };
    std::shared_ptr<process::NsProcess> ret;
    if (m_pAttachedProcess)
    {
        ret = std::move(m_pAttachedProcess);
    }
    else
    {
        NN_RESULT_TRY(this->CreateProcessImpl(&ret))
            NN_RESULT_CATCH(ns::ResultApplicationRecordNotFound)
            {
                ReportProcessCreationFailure(NN_RESULT_CURRENT_RESULT);
                NN_RESULT_THROW(am::ResultApplicationRecordNotFound());
            }
            NN_RESULT_CATCH(lr::ResultProgramNotFound)
            {
                ReportProcessCreationFailure(NN_RESULT_CURRENT_RESULT);
                NN_RESULT_THROW(am::ResultAppletProgramNotFound());
            }
            NN_RESULT_CATCH(svc::ResultLimit)
            {
                if (!(retryRestTime >= 0))
                {
                    ReportProcessCreationFailure(NN_RESULT_CURRENT_RESULT);
                    NN_AM_SERVICE_APPLET_LOG(error, this, "failed to create process(svc::ResultLimit). give up.", static_cast<int>(retrySleepTime.GetMilliSeconds()), retryCount);
                    NN_RESULT_RETHROW;
                }
                NN_AM_SERVICE_APPLET_LOG(seq, this, "failed to create process(svc::ResultLimit). sleep %dms before retry[%d]", static_cast<int>(retrySleepTime.GetMilliSeconds()), retryCount);
                ++retryCount;
                goto retry;
            }
            NN_RESULT_CATCH(ns::ResultLaunchOverlayAppletDisabled)
            {
                // オーバーレイアプレットの起動無効化設定によるプロセス作成失敗は不具合ではないのでエラーレポートを作成しない。
                NN_AM_SERVICE_APPLET_LOG(seq, this, "launch overlay applet is disabled: %08x", NN_RESULT_CURRENT_RESULT.GetInnerValueForDebug());
                NN_RESULT_THROW(am::ResultAppletProcessCreationFailed());
            }
            NN_RESULT_CATCH(ns::ResultLaunchSystemAppletDisabled)
            {
                // システムアプレットの起動無効化設定によるプロセス作成失敗は不具合ではないのでエラーレポートを作成しない。
                NN_AM_SERVICE_APPLET_LOG(seq, this, "launch system applet is disabled: %08x", NN_RESULT_CURRENT_RESULT.GetInnerValueForDebug());
                NN_RESULT_THROW(am::ResultAppletProcessCreationFailed());
            }
            NN_RESULT_CATCH_ALL
            {
                ReportProcessCreationFailure(NN_RESULT_CURRENT_RESULT);
                NN_AM_SERVICE_APPLET_LOG(error, this, "failed to created process: %08x", NN_RESULT_CURRENT_RESULT.GetInnerValueForDebug());
                NN_RESULT_THROW(am::ResultAppletProcessCreationFailed());
            }
        NN_RESULT_END_TRY
    }
    BeforeActivateProxy(ret->GetProcessId());
    GetAppletSystem()->RegisterAppletProxy(ret->GetProcessId().value, this);
    NN_AM_SERVICE_APPLET_LOG(seq, this, "register proxy process id = %lld", ret->GetProcessId().value);
    *pOut = std::move(ret);
    NN_RESULT_SUCCESS;
}

void IntegratedApplet::AttachProcessAndStart(os::ProcessId processId) NN_NOEXCEPT
{
    auto pProcess = this->MakeAttachedProcess(processId);
    m_pAttachedProcess = std::move(pProcess);
    static_cast<core::Applet*>(this)->Start();
}

void IntegratedApplet::SendPlayLog( pdm::AppletEventType eventType ) NN_NOEXCEPT
{
    auto info = GetAppletIdentityInfo();
    ncm::SystemProgramId programId;
    if ( MakeSystemProgramIdFromAppletId( &programId, info.appletId ).IsFailure() )
    {
        programId = programId.GetInvalidId();
    }
    NotifyAppletEvent( eventType,  programId, 0, info.appletId,
        ncm::StorageId::BuildInSystem, ns::PlayLogPolicy::LogOnly );
}

Result IntegratedApplet::IndirectLayer::Create() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_Handle);
    decltype(m_Handle) handle;
    NN_RESULT_DO(am::service::CreateIndirectLayer(&handle));
    this->m_Handle = handle;
    NN_RESULT_SUCCESS;
}

IntegratedApplet::IndirectLayer::~IndirectLayer() NN_NOEXCEPT
{
    if (m_Handle)
    {
        DestroyIndirectLayer(m_Handle);
    }
}

void IntegratedApplet::ReportProcessCreationFailure(nn::Result result) NN_NOEXCEPT
{
    result = MakeCreateProcessFailureErrorReport(result);
    if (result.IsFailure())
    {
        NN_AM_SERVICE_APPLET_LOG(error, this, "failed to create error report: 0x%08x.\n", result.GetInnerValueForDebug());
    }
}

Result IntegratedApplet::MakeCreateProcessFailureErrorReport(nn::Result result) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(result.IsFailure());
    auto info = GetAppletIdentityInfo();
    if (info.IsApplication())
    {
        NN_RESULT_DO(ns::SetApplicationTerminateResult(info.applicationId, result));
        return am::service::MakeCreateProcessFailureErrorReport(result, info.applicationId.value);
    }
    else if (info.IsSystemApplication())
    {
        return am::service::MakeCreateProcessFailureErrorReport(result, info.systemApplicationId.value);
    }
    else
    {
        ncm::SystemProgramId id = ncm::SystemProgramId::GetInvalidId();
        NN_RESULT_DO(MakeSystemProgramIdFromAppletId(&id, info.appletId));
        return am::service::MakeCreateProcessFailureErrorReport(result, id.value);
    }
}

Result IntegratedApplet::SubmitApplicationInfo() NN_NOEXCEPT
{
    auto info = GetAppletIdentityInfo();
    ncm::SystemProgramId id = ncm::SystemProgramId::GetInvalidId();
    NN_RESULT_DO(MakeSystemProgramIdFromAppletId(&id, info.appletId));
    NN_RESULT_DO(am::service::SubmitApplicationInfo(id.value));
    NN_RESULT_SUCCESS;
}

void IntegratedApplet::OverrideRightsEnvironment(std::shared_ptr<service::rightsManagement::RightsEnvironment> p) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(p);
    this->m_pRightsEnvironment = std::move(p);
}

Result IntegratedApplet::AppletAccessorImpl::SetUsers(bool allUser, const sf::InArray<account::Uid>& uids) NN_NOEXCEPT
{
    auto& rightsEnvironment = *GetApplet()->m_pRightsEnvironment;
    return allUser
        ? rightsEnvironment.SetUsers(service::rightsManagement::AllUser)
        : rightsEnvironment.SetUsers(uids)
    ;
}

Result IntegratedApplet::AppletAccessorImpl::CheckRightsEnvironmentAvailable(sf::Out<bool> pOut) NN_NOEXCEPT
{
    *pOut = GetApplet()->CheckRightsEnvironmentAvailable();
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletAccessorImpl::GetNsRightsEnvironmentHandle(sf::Out<Bit64> pOut) NN_NOEXCEPT
{
    return GetApplet()->m_pRightsEnvironment->GetNsHandle(pOut.GetPointer());
}

void IntegratedApplet::RunBehindProcessForRights(process::NsProcess* pProcess) NN_NOEXCEPT
{
    os::TimerEvent timer{os::EventClearMode_ManualClear};
    timer.StartOneShot(0);
    for (;;)
    {
        switch (os::WaitAny(timer.GetBase(), pProcess))
        {
            case 0: // timer
            {
                timer.Clear();
                UpdateRightsEnvironmentStatus();
                // TORIAEZU: 定期的にポーリングしておく
                // TODO: 本実装: 権利状態の変更シグナル・時間経過によって必要な時に動かす
                timer.StartOneShot(m_pRightsEnvironment->GetPollingInterval());
                continue;
            }
            case 1: // pProcess
            {
                return;
            }
            default: NN_UNEXPECTED_DEFAULT;
        }
    }
}

void IntegratedApplet::UpdateRightsEnvironmentStatus() NN_NOEXCEPT
{
    auto newStatus = m_pRightsEnvironment->GetStatus();
    auto lk = util::MakeUniqueLock(m_RightsEnvironmentStatusMutex);
    auto oldStatus = std::exchange(m_RightsEnvironmentStatus, newStatus);
    auto setCanRun = [&](bool canRun)
    {
        auto updater = GetAppletSystem()->GetWindowManager()->BeginUpdate();
        updater.RefWindowProperty(GetWindow()).canRun = canRun;
    };
    if (oldStatus.isAvailable)
    {
        if (newStatus.isAvailable)
        {
            // 有効->有効
            OnRightsAvailableTimeChanged(newStatus.expirationTick);
        }
        else
        {
            // 有効->無効
            setCanRun(false);
            lk.unlock();
            OnRightsDeactivated();
        }
    }
    else
    {
        if (newStatus.isAvailable)
        {
            // 無効->有効
            setCanRun(true);
            OnRightsActivated(newStatus.expirationTick);
        }
        else
        {
            // 無効->無効
            // nop
        }
    }
}

bool IntegratedApplet::CheckRightsEnvironmentAvailable() NN_NOEXCEPT
{
    // 上位レイヤから「権利が使えるようになったかどうか」の問い合わせ

    // 権利失効→(上位レイヤによるに処理で)権利復帰→直後に権利失効などがあった場合に、
    // am が権利が一瞬復帰したことのエッジを検知できないと、失効通知を送ることができない。
    // これを避けるため、上位レイヤは権利復帰を行った後に本関数を呼び、
    // 権利が復帰した瞬間があるかどうかをチェックする。
    auto isRightsEnvironmentAvailable = [&]
    {
        auto lk = util::MakeUniqueLock(m_RightsEnvironmentStatusMutex);
        return m_RightsEnvironmentStatus.isAvailable;
    };
    if (isRightsEnvironmentAvailable())
    {
        // 現時点で復帰していると am が認識しているならば、すぐに true を返してよい
        return true;
    }
    // そうでない場合は、明示的に am の認識を最新にする
    UpdateRightsEnvironmentStatus();
    // その上で結果を返す
    return isRightsEnvironmentAvailable();
}

void IntegratedApplet::OnRightsActivated(os::Tick) NN_NOEXCEPT
{
    // nop
}

void IntegratedApplet::OnRightsDeactivated() NN_NOEXCEPT
{
    // nop
}

void IntegratedApplet::OnRightsAvailableTimeChanged(os::Tick) NN_NOEXCEPT
{
    // nop
}

}}}
