﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/am/service/am_GpuErrorControl.h>
#include <nn/am/service/am_StuckChecker.h>

#include "am_SpsmEvent.h"
#include "am_CaptureButton.h"
#include "am_ShellEvent.h"
#include "am_ApmEvent.h"
#include "am_OmmEvent.h"
#include "am_SdCardEvent.h"
#include "am_GameCardEvent.h"
#include "am_FatalEvent.h"
#include "am_ApplicationTimer.h"
#include <nn/am/service/am_ContentActionTable.h>

namespace nn { namespace am { namespace service {

//-----------------------------------------------------------------------------
//  イベントハンドラ用の自動ループ関数
//  - HOME ボタンの短押し／長押し検知
//  - POWER ボタンの短押し／中押し／長押し検知
//  - NuiShell からのイベント通知
//  - Omm からの動作モード変更通知
//  - SD カードが抜かれたことの検知
//  - その他（今後増える可能性あり）
//

void EventHandlerLoopAutoImpl(AppletSystem* pAppletSystem) NN_NOEXCEPT
{
    FatalEvent fatalEvent(pAppletSystem);
    SpsmEvent spsmEvent(pAppletSystem);
    CaptureButton& captureButton = GetCaptureButton();
    captureButton.SetWindowManager( pAppletSystem->GetWindowManager() );
    ShellEvent  shellEvent( pAppletSystem );
    ApmEvent apmEvent( pAppletSystem );
    OmmEvent& ommEvent = GetOmmEvent();
    ommEvent.SetupOmmEventHolder();
    SdCardEvent sdCardEvent( pAppletSystem );
    GameCardEvent gameCardEvent(pAppletSystem);
    GpuErrorEvent gpuErrorEvent;
    ApplicationTimer& applicationTimer = GetApplicationTimer();

    // 前処理
    os::MultiWaitType multi;
    os::InitializeMultiWait(&multi);

    // 関連ホルダのリンク
    fatalEvent.LinkMultiWaitHolders(&multi);
    spsmEvent.LinkMultiWaitHolders(&multi);
    captureButton.LinkMultiWaitHolders(&multi);
    shellEvent.LinkMultiWaitHolders(&multi);
    apmEvent.LinkMultiWaitHolders( &multi );
    ommEvent.LinkMultiWaitHolders( &multi );
    sdCardEvent.LinkMultiWaitHolders( &multi );
    gameCardEvent.LinkMultiWaitHolders(&multi);
    gpuErrorEvent.LinkMultiWaitHolders(&multi);
    applicationTimer.LinkMultiWaitHolders(&multi);

    os::MultiWaitHolderType contentActionTableUpdateEventHolder;
    am::service::contentAction::InitializeMultiWaitHolderForSystemDataUpdated(&contentActionTableUpdateEventHolder);
    os::LinkMultiWaitHolder(&multi, &contentActionTableUpdateEventHolder);

    // ここからループ処理
    for (;;)
    {
        auto* holder = os::WaitAny(&multi);
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(am_Event_Loop, 60);
        if (NN_STATIC_CONDITION(false))
        {
        }
        else if (holder->userData == reinterpret_cast<uintptr_t>(&fatalEvent))
        {
            fatalEvent.HandleEventIfHolderOwner(holder);
        }
        else if (holder->userData == reinterpret_cast<uintptr_t>(&spsmEvent))
        {
            spsmEvent.HandleEventIfHolderOwner(holder);
        }
        else if (holder->userData == reinterpret_cast<uintptr_t>(&captureButton))
        {
            captureButton.HandleEventIfHolderOwner(holder);
        }
        else if (holder->userData == reinterpret_cast<uintptr_t>(&shellEvent))
        {
            shellEvent.HandleEventIfHolderOwner(holder);
            gpuErrorEvent.OnMaybeProcessTerminated();
        }
        else if (holder->userData == reinterpret_cast<uintptr_t>( &apmEvent ))
        {
            apmEvent.HandleEventIfHolderOwner(holder);
        }
        else if (holder->userData == reinterpret_cast<uintptr_t>( &ommEvent ))
        {
            ommEvent.HandleEventIfHolderOwner(holder);
        }
        else if (holder->userData == reinterpret_cast<uintptr_t>( &sdCardEvent ))
        {
            sdCardEvent.HandleEventIfHolderOwner(holder);
        }
        else if (holder->userData == reinterpret_cast<uintptr_t>(&gameCardEvent))
        {
            gameCardEvent.HandleEventIfHolderOwner(holder);
        }
        else if (holder->userData == reinterpret_cast<uintptr_t>(&gpuErrorEvent))
        {
            gpuErrorEvent.HandleEventIfHolderOwner(holder, pAppletSystem);
        }
        else if (holder->userData == reinterpret_cast<uintptr_t>(&applicationTimer))
        {
            applicationTimer.HandleEventIfHolderOwner(holder);
        }
        else if (holder == &contentActionTableUpdateEventHolder)
        {
            am::service::contentAction::NotifySystemDataUpdated();
        }
    }

    // 後処理
#if 0
    // 警告防止のためコメントアウト
    shellEvent.UnlinkMultiWaitHolders(&multi);
    spsmEvent.UnlinkMultiWaitHolders(&multi);
    os::FinalizeMultiWait(&multi);
#endif
}


//-----------------------------------------------------------------------------

}}} // namespace nn::am::service

