﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ahid_Hdr.h"

namespace nn {
namespace ahid {
namespace hdr {

//////////////////////////////////////////////////////////////////////////////
//  constructor destructor
//////////////////////////////////////////////////////////////////////////////
NN_IMPLICIT HdrSession::HdrSession(HdrServer *pHdrServer, bool isDfcSession) NN_NOEXCEPT
{
    m_pHdrServer    = pHdrServer;
    m_IsDfcSession  = isDfcSession;
}


//////////////////////////////////////////////////////////////////////////////
HdrSession::~HdrSession() NN_NOEXCEPT
{
}


//////////////////////////////////////////////////////////////////////////////
//  public functions
//////////////////////////////////////////////////////////////////////////////
Result
HdrSession::GetDeviceEntries(
    nn::sf::Out<std::uint32_t> entriesBuffer
    ) NN_NOEXCEPT
{
    uint32_t *pEntries = entriesBuffer.GetPointer();
    uint32_t entries = 0;

    m_pHdrServer->m_HdrMutex.Lock();

    for (int i = 0; i < HdrEntriesCountMax; i++)
    {
        DeviceHandle handle;
        Result result = m_pHdrServer->m_HdrHandle.GetHandleForIndex(i, &handle);

        if (result.IsSuccess())
        {
            if (m_IsDfcSession == false)
            {
                if (DeviceIsFilteredForHipc(i) == false)
                {
                    entries++;
                }
            }
            else
            {
                entries++;
            }
        }
    }

    m_pHdrServer->m_HdrMutex.Unlock();

    *pEntries = entries;

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result
HdrSession::GetDeviceList(
    nn::sf::Out<std::uint32_t>  outEntries,
    uint32_t                    inEntries,
    nn::sf::OutBuffer           outDeviceHandles,
    nn::sf::InBuffer            inAttachFilter
    ) NN_NOEXCEPT
{
    uint32_t*           pOutEntries     = outEntries.GetPointer();
    const DeviceHandle* pHandlesConst   = reinterpret_cast<const DeviceHandle*>(outDeviceHandles.GetPointerUnsafe());
    const AttachFilter* pFilter         = reinterpret_cast<const AttachFilter*>(inAttachFilter.GetPointerUnsafe());

    DeviceHandle* pHandles = const_cast<DeviceHandle*>(pHandlesConst);

    uint32_t entries = 0;

    if (inEntries)
    {
        m_pHdrServer->m_HdrMutex.Lock();

        for (int i = 0; i < HdrEntriesCountMax; i++)
        {
            bool attachEntry = true;
            DeviceHandle handle;
            Result result = m_pHdrServer->m_HdrHandle.GetHandleForIndex(i, &handle);

            if (result.IsSuccess())
            {
                if (m_IsDfcSession == false)
                {
                    if (DeviceIsFilteredForHipc(i) == true)
                    {
                        continue;
                    }
                }

                if (pFilter)
                {
                    ServerDeviceParameters serverDeviceParameters;

                    m_pHdrServer->m_HdrDevice.GetServerDeviceParameters(i, &serverDeviceParameters);

                    if (pFilter->usagePage)
                    {
                        if (serverDeviceParameters.deviceParameters.usagePage != pFilter->usagePage)
                        {
                            attachEntry = false;
                        }
                    }

                    if (pFilter->usageId)
                    {
                        if (serverDeviceParameters.deviceParameters.usageId != pFilter->usageId)
                        {
                            attachEntry = false;
                        }
                    }

                    if (pFilter->busId)
                    {
                        if (serverDeviceParameters.deviceParameters.busId != pFilter->busId)
                        {
                            attachEntry = false;
                        }
                    }

                    if (pFilter->vendorId)
                    {
                        if (serverDeviceParameters.deviceParameters.vendorId != pFilter->vendorId)
                        {
                            attachEntry = false;
                        }
                    }

                    if (pFilter->productId)
                    {
                        if (serverDeviceParameters.deviceParameters.productId != pFilter->productId)
                        {
                            attachEntry = false;
                        }
                    }

                    if (pFilter->manufacturer[0])
                    {
                        if (memcmp(serverDeviceParameters.deviceParameters.manufacturer, pFilter->manufacturer, pFilter->manufacturer[0]))
                        {
                            attachEntry = false;
                        }
                    }

                    if (pFilter->product[0])
                    {
                        if (memcmp(serverDeviceParameters.deviceParameters.product, pFilter->product, pFilter->product[0]))
                        {
                            attachEntry = false;
                        }
                    }

                    if (pFilter->serialNumber[0])
                    {
                        if (memcmp(serverDeviceParameters.deviceParameters.serialNumber, pFilter->serialNumber, pFilter->serialNumber[0]))
                        {
                            attachEntry = false;
                        }
                    }

                    if (pFilter->attachedAfterTicks)
                    {
                        if (pFilter->attachedAfterTicks >= serverDeviceParameters.attachTime)
                        {
                            attachEntry = false;
                        }
                    }
                }

                if (attachEntry)
                {
                    *pHandles++ = handle;
                    entries++;
                }
            }

            if (entries == inEntries)
            {
                break;
            }
        }

        m_pHdrServer->m_HdrMutex.Unlock();
    }

    *pOutEntries = entries;

    return ResultSuccess();

} // NOLINT(impl/function_size)


//////////////////////////////////////////////////////////////////////////////
Result
HdrSession::GetDeviceParameters(
    DeviceHandle handle,
    nn::sf::OutBuffer outDeviceParameters
    ) NN_NOEXCEPT
{
    const DeviceParameters* pDeviceParamaters = reinterpret_cast<const DeviceParameters*>(outDeviceParameters.GetPointerUnsafe());
    Result result;

    m_pHdrServer->m_HdrMutex.Lock();

    int index;

    result = m_pHdrServer->m_HdrHandle.GetIndexForHandle(handle, &index);

    if (result.IsSuccess())
    {
        result = m_pHdrServer->m_HdrDevice.GetDeviceParameters(index, const_cast<DeviceParameters*>(pDeviceParamaters));
    }

    m_pHdrServer->m_HdrMutex.Unlock();

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result
HdrSession::AttachDevice(
    nn::sf::InBuffer inDeviceParameters,
    nn::sf::Out<DeviceHandle> outHandle
    ) NN_NOEXCEPT
{
    const DeviceParameters* pDeviceParamaters = reinterpret_cast<const DeviceParameters*>(inDeviceParameters.GetPointerUnsafe());
    DeviceHandle *pHandle = outHandle.GetPointer();
    Result result;

    m_pHdrServer->m_HdrMutex.Lock();

    result = m_pHdrServer->m_HdrHandle.AttachHandle(pHandle);

    if (result.IsSuccess())
    {
        int index;
        result = m_pHdrServer->m_HdrHandle.GetIndexForHandle(*pHandle, &index);

        if (result.IsSuccess())
        {
            result = m_pHdrServer->m_HdrDevice.Attach(index, const_cast<DeviceParameters*>(pDeviceParamaters));
        }
    }

    m_pHdrServer->m_HdrMutex.Unlock();

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result
HdrSession::DetachDevice(
    DeviceHandle handle
    ) NN_NOEXCEPT
{
    Result result;

    m_pHdrServer->m_HdrMutex.Lock();

    int index;

    result = m_pHdrServer->m_HdrHandle.GetIndexForHandle(handle, &index);

    if (result.IsSuccess())
    {
        result = m_pHdrServer->m_HdrDevice.Detach(index);

        if (result.IsSuccess())
        {
            result = m_pHdrServer->m_HdrHandle.DetachHandle(handle);
        }
    }

    m_pHdrServer->m_HdrMutex.Unlock();

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result
HdrSession::SetDeviceFilterForHipc(
    nn::sf::InBuffer    inDeviceFilter,
    uint32_t            inEntries
    ) NN_NOEXCEPT
{
    if (this->m_IsDfcSession)
    {
        // Make sure buffer is size of entries
        if (inDeviceFilter.GetSize() == (sizeof(DeviceFilter) * inEntries))
        {
            const DeviceFilter *pFilter = reinterpret_cast<const DeviceFilter*>(inDeviceFilter.GetPointerUnsafe());

            m_pHdrServer->m_HdrMutex.Lock();

            // Device filter can only be set by DFC, thus we can store and use pointer.
            m_pHdrServer->m_pDeviceFilter       = const_cast<DeviceFilter*>(pFilter);
            m_pHdrServer->m_DeviceFilterEntries = inEntries;

            m_pHdrServer->m_HdrMutex.Unlock();

            return ResultSuccess();
        }

        return ResultInvalidParameter();
    }

    return ResultFunctionNotSupported();
}


//////////////////////////////////////////////////////////////////////////////
bool HdrSession::DeviceIsFilteredForHipc(int deviceEntry) NN_NOEXCEPT
{
    // Compare device parameters against entries in the DeviceFilter
    if (m_pHdrServer->m_pDeviceFilter && m_pHdrServer->m_DeviceFilterEntries)
    {
        ServerDeviceParameters serverDeviceParameters;

        m_pHdrServer->m_HdrDevice.GetServerDeviceParameters(deviceEntry, &serverDeviceParameters);

        DeviceFilter *pDeviceFilter = m_pHdrServer->m_pDeviceFilter;

        for (uint32_t i = 0; i < m_pHdrServer->m_DeviceFilterEntries; i++)
        {
            // VID PID filter only valid if vendor ID is non-zero
            if (pDeviceFilter->vendorId)
            {
                if (pDeviceFilter->vendorId == serverDeviceParameters.deviceParameters.vendorId)
                {
                    if (pDeviceFilter->productId == serverDeviceParameters.deviceParameters.productId)
                    {
                        return true;
                    }
                }
            }

            // usage page and usage ID only valid if both is non-zero
            if ((pDeviceFilter->usagePage) && (pDeviceFilter->usageId))
            {
                if ((pDeviceFilter->usagePage == serverDeviceParameters.deviceParameters.usagePage) && (pDeviceFilter->usageId == serverDeviceParameters.deviceParameters.usageId))
                {
                    return true;
                }
            }

            pDeviceFilter++;
        }
    }

    return false;
}

}   // namespace hdr
}   // namespace ahid
}   // namespace nn
