﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/user/account_UserRegistry.h>

#include "../user/account_SaveDataAccessor.h"
#include <nn/account/detail/account_InternalConfig.h>
#include <nn/account/detail/account_Log.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_ResultForAdministrators.h>
#include <nn/account/account_ResultPrivate.h>

#include <mutex>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_SystemEventApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

namespace nn { namespace account { namespace user {

/** -------------------------------------------------------------------------------------------
    内部で使う
 */

bool UserRegistry::DeletePendingUserUnsafe(const Uid& uid) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    for (auto& pu: m_PendingUsers)
    {
        if (pu == uid)
        {
            // ユーザーの追加処理の取り消し
            pu = InvalidUid;
            -- m_PendingCount;
            return true;
        }
    }
    return false;
}

void UserRegistry::RelocateUserUnsafe(int fromIndex, int toIndex) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(fromIndex >= 0 && fromIndex < UserCountMax);
    NN_SDK_ASSERT(toIndex >= 0 && toIndex < UserCountMax);
    NN_SDK_ASSERT(m_Users[toIndex].IsValid());
    NN_SDK_ASSERT(m_Users[fromIndex].IsValid());

    if (fromIndex == toIndex)
    {
        return;
    }

    auto tmp = m_Users[fromIndex];
    if (fromIndex < toIndex)
    {
        // 対象の位置まで前詰め
        for (auto i = fromIndex + 1; i <= toIndex; ++ i)
        {
            m_Users[i - 1] = std::move(m_Users[i]);
            NN_SDK_ASSERT(m_Users[i - 1].IsValid());
        }
        m_Users[toIndex] = std::move(tmp);
    }
    else
    {
        // 対象の位置まで後ろ詰め
        for (auto i = fromIndex; i >= toIndex + 1; -- i)
        {
            m_Users[i] = std::move(m_Users[i - 1]);
            NN_SDK_ASSERT(m_Users[i].IsValid());
        }
        m_Users[toIndex] = std::move(tmp);
    }
    NN_SDK_ASSERT(m_Users[toIndex].IsValid());
}

void UserRegistry::StoreSaveData() const NN_NOEXCEPT
{
    detail::SerializedUuid serialized;
    std::memcpy(&serialized, &m_InstanceId, sizeof(serialized));
    auto instanceId = serialized.Deserialize();
    Uid users[sizeof(m_Users) / sizeof(m_Users[0])];
    for (auto i = 0u; i < sizeof(users) / sizeof(users[0]); ++ i)
    {
        users[i] = m_Users[i];
    }

    auto lock = m_pStorage->AcquireWriterLock();
    NN_ABORT_UNLESS_RESULT_SUCCESS(SaveDataAccessor::Store(
        instanceId,
        users, sizeof(users) / sizeof(users[0]),
        m_PendingUsers, sizeof(m_PendingUsers) / sizeof(m_PendingUsers[0]),
        *m_pStorage));
}

/** -------------------------------------------------------------------------------------------
    サービス起動, 終了時の処理
*/
Result UserRegistry::Initialize(const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_Initialized);

    // セーブデータの読み込み
    detail::Uuid instanceId;
    Uid users[sizeof(m_Users) / sizeof(m_Users[0])];
    NN_RESULT_DO(SaveDataAccessor::Load(
        &instanceId,
        &m_RegisteredCount, users, sizeof(users) / sizeof(users[0]),
        &m_PendingCount, m_PendingUsers, sizeof(m_PendingUsers) / sizeof(m_PendingUsers[0]),
        storage));

    static_assert(sizeof(util::Uuid) == sizeof(detail::SerializedUuid), "sizeof(util::Uuid) != sizeof(detail::Uuid)");
    auto serialized = instanceId.Serialize();
    std::memcpy(&m_InstanceId, &serialized, sizeof(m_InstanceId));
    for (auto i = 0u; i < sizeof(users) / sizeof(users[0]) && users[i]; ++ i)
    {
        m_Users[i] = UserRef(users[i], m_Counters[i]);
    }
    m_pStorage = &storage;
    m_Initialized = true;
    NN_RESULT_SUCCESS;
}
void UserRegistry::ListPendingUsers(nn::account::Uid* outUsers, int count) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    NN_SDK_ASSERT(count == sizeof(m_PendingUsers) / sizeof(m_PendingUsers[0]));

    int ct = 0;
    for (auto& u: m_PendingUsers)
    {
        outUsers[ct ++] = u;
    }
    for (auto i = ct; i < count; ++ i)
    {
        outUsers[i] = InvalidUid;
    }
}

/** -------------------------------------------------------------------------------------------
    公開 API
 */

int UserRegistry::GetUserCount() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    return m_RegisteredCount;
}

nn::Result UserRegistry::GetUserExistence(bool* pOutExistence, const Uid& uid) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    NN_RESULT_THROW_UNLESS(uid, ResultInvalidUserId());

    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    for (auto& u: m_Users)
    {
        if (u == uid)
        {
            *pOutExistence = true;
            NN_RESULT_SUCCESS;
        }
    }
    *pOutExistence = false;
    NN_RESULT_SUCCESS;
}

nn::Result UserRegistry::ListAllUsers(Uid* outUsers, int count) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    NN_RESULT_THROW_UNLESS(count == UserCountMax, ResultInvalidArrayLength());

    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    for (auto i = 0; i < m_RegisteredCount; ++ i)
    {
        NN_SDK_ASSERT(m_Users[i].IsValid());
        outUsers[i] = m_Users[i];
    }
    for (auto i = m_RegisteredCount; i < UserCountMax; ++ i)
    {
        NN_SDK_ASSERT(!m_Users[i].IsValid());
        outUsers[i] = InvalidUid;
    }
    NN_RESULT_SUCCESS;
}

nn::Result UserRegistry::GetUserRegistrationEvent(os::SystemEventType** pOutEvent) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    return Notifiable::AcquireSystemEvent(pOutEvent, UserRegistryTag_Registration);
}

nn::Result UserRegistry::BeginUserRegistration(Uid* pOutUid) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    auto uid = detail::ConvertToUid(m_pStorage->GenerateUuidWithContext());
    NN_RESULT_DO(BeginUserRegistrationAs(uid));
    *pOutUid = uid;
    NN_RESULT_SUCCESS;
}

nn::Result UserRegistry::BeginUserRegistrationAs(const Uid& uid) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    NN_SDK_ASSERT(uid);
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    NN_RESULT_THROW_UNLESS(m_PendingCount == 0, ResultRegistrationAlreadyBegins());

    // 登録済み Uid との重複チェック
    for (auto i = 0; i < m_RegisteredCount; ++ i)
    {
        NN_RESULT_THROW_UNLESS(static_cast<Uid>(m_Users[i]) != uid, ResultUidAlreadyRegistered());
    }
    for (const auto& pu: m_PendingUsers)
    {
        NN_RESULT_THROW_UNLESS(pu != uid, ResultUidAlreadyRegistered());
    }

    // 登録上限チェック
    NN_RESULT_THROW_UNLESS(m_RegisteredCount + m_PendingCount + 1 <= UserCountMax, ResultUserRegistryFull());

    // 登録処理
    bool found = false;
    for (auto& pu: m_PendingUsers)
    {
        if (!pu)
        {
            pu = uid;
            NN_SDK_ASSERT(pu);
            ++ m_PendingCount;

            found = true;
            break;
        }
    }
    NN_ABORT_UNLESS(
        found,
        "[nn::account] -----------------------------------------------\n"
        "  ABORT: Pending user list broken (internal)\n");
    StoreSaveData();
    NN_RESULT_SUCCESS;
}

nn::Result UserRegistry::CancelUserRegistration(const Uid& uid) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    NN_RESULT_THROW_UNLESS(uid, ResultInvalidUserId());

    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    auto deleted = DeletePendingUserUnsafe(uid);
    NN_RESULT_THROW_UNLESS(deleted, ResultUserNotExist());

    StoreSaveData();
    NN_RESULT_SUCCESS;
}

nn::Result UserRegistry::CompleteUserRegistration(const Uid& uid) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    NN_RESULT_THROW_UNLESS(uid, ResultInvalidUserId());

    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    auto deleted = DeletePendingUserUnsafe(uid);
    NN_RESULT_THROW_UNLESS(deleted, ResultUserNotExist());

    bool registered = false;
    for (auto& u: m_Users)
    {
        if (!u.IsValid())
        {
            // 参照カウンタの初期化
            UserRef ref;
            for (auto& ct: m_Counters)
            {
                if (!ct)
                {
                    ref = UserRef(uid, ct);
                    NN_SDK_ASSERT(ref.IsValid());
                }
            }
            // 削除済みのユーザーに関してまだ参照が残っている場合は登録に失敗する
            NN_RESULT_THROW_UNLESS(ref.IsValid(), ResultUserRegistryFull());

            u = std::move(ref);
            ++ m_RegisteredCount;
            registered = true;
            break;
        }
    }
    NN_ABORT_UNLESS(
        registered,
        "[nn::account] -----------------------------------------------\n"
        "  ABORT: User list broken (internal)\n");

    // セーブデータの更新 + 通知
    StoreSaveData();
    Notifiable::SignalEvents(UserRegistryTag_Registration);
    NN_RESULT_SUCCESS;
}

nn::Result UserRegistry::DeleteUser(const Uid& uid) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    NN_RESULT_THROW_UNLESS(uid, ResultInvalidUserId());

    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    for (auto i = 0; i < m_RegisteredCount; ++ i)
    {
        auto& user = m_Users[i];
        if (static_cast<Uid>(m_Users[i]) == uid)
        {
            // 参照が 1 より大きいユーザーに関しては警告する (削除不可とはしない)
            if (!(user.GetCount() <= 1))
            {
                NN_DETAIL_ACCOUNT_WARN("User account is referred by %d extra objects\n", user.GetCount() - 1);
                // NN_RESULT_THROW_UNLESS(user.GetCount() <= 1, ResultUserReferred());
            }

            // ユーザーの削除
            auto tail = m_RegisteredCount - 1;
            RelocateUserUnsafe(i, tail); // 削除するものは最後尾へ
            m_Users[tail].Release(); // Registry からユーザーへの参照を削除
            NN_SDK_ASSERT(!m_Users[tail].IsValid()); // 参照してないはず
            -- m_RegisteredCount;

            if (m_LastOpenedUser == uid)
            {
                // 最後にオープンされたユーザーが削除された場合、履歴を削除する
                m_LastOpenedUser = InvalidUid;
            }

            // セーブデータの更新 + 通知
            StoreSaveData();
            Notifiable::SignalEvents(UserRegistryTag_Registration);
            NN_RESULT_SUCCESS;
        }
    }
    NN_RESULT_THROW(ResultUserNotExist());
}

Result UserRegistry::GetUserRef(UserRef* pOut, const Uid& uid) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    NN_SDK_ASSERT(uid);

    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    for (auto i = 0; i < m_RegisteredCount; ++ i)
    {
        auto& u = m_Users[i];
        if (u == uid)
        {
            NN_SDK_ASSERT(u.IsValid()); // 参照カウンタが有効なこと。
            *pOut = u;
            NN_RESULT_SUCCESS;
        }
    }
    NN_RESULT_THROW(ResultUserNotExist());
}

nn::Result UserRegistry::SetUserPosition(const Uid& uid, int position) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    NN_RESULT_THROW_UNLESS(position >= 0, ResultInvalidUserId());
    NN_RESULT_THROW_UNLESS(uid, ResultInvalidUserId());

    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    NN_RESULT_THROW_UNLESS(position < m_RegisteredCount, ResultInvalidUserPosition());
    for (auto i = 0; i < m_RegisteredCount; ++ i)
    {
        auto& u = m_Users[i];
        if (u == uid)
        {
            RelocateUserUnsafe(i, position);

            // セーブデータの更新
            StoreSaveData();
            NN_RESULT_SUCCESS;
        }
    }
    NN_RESULT_THROW(ResultUserNotExist());
}

}}} // ~namespace nn::account::user
