﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/user/account_UserRegistry.h>

#include "../detail/account_PathUtil.h"
#include "../detail/account_UuidUtil.h"
#include <nn/account/detail/account_InternalConfig.h>
#include <nn/account/detail/account_LocalStorage.h>
#include <nn/account/account_ResultPrivate.h>

namespace nn { namespace account { namespace user {

struct SaveData
{
    static const uint32_t Magic     = 0x290185bc; // NOTE: 不変
    static const uint32_t Version   = 0x01601160; // NOTE: yyymmdd#
    static const uint64_t Header    = static_cast<uint64_t>(Magic) << 32 | Version;

    uint64_t header;
    uint64_t zero;
    detail::SerializedUuid instanceId;
    detail::SerializedUuid users[UserCountMax];
    detail::SerializedUuid pendings[1];
}; // 176 バイト
static_assert(std::alignment_of<SaveData>::value <= std::alignment_of<std::max_align_t>::value, "alignof(SaveData) > alignof(max_align_t)");


class SaveDataAccessor
{
private:
    static Result LoadImpl(
        detail::Uuid* pOutInstanceId,
        Uid* outUsers, int userCount,
        Uid* outPendings, int pendingCount,
        const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(userCount == UserCountMax);
        NN_SDK_ASSERT(pendingCount == 1);

        auto& fs = storage.GetFileSystem();
        char path[64];
        detail::PathUtil::GetUserListPath(path, sizeof(path), storage.GetRootPath());

        size_t size;
        NN_RESULT_DO(fs.GetSize(&size, path));
        NN_RESULT_THROW_UNLESS(size == sizeof(SaveData), ResultSaveDataBroken());

        SaveData save;
        NN_RESULT_DO(fs.Read(&size, &save, sizeof(save), path));
        NN_RESULT_THROW_UNLESS(detail::ConvertNetworkToHost(save.header) == SaveData::Header, ResultSaveDataBroken());
        NN_RESULT_THROW_UNLESS(save.zero == 0x00ull, ResultSaveDataBroken());

        *pOutInstanceId = save.instanceId.Deserialize();
        for (auto i = 0; i < userCount; ++ i)
        {
            auto uuid = save.users[i].Deserialize();
            outUsers[i] = detail::ConvertToUid(uuid);
        }
        for (auto i = 0; i < pendingCount; ++ i)
        {
            auto uuid = save.pendings[i].Deserialize();
            outPendings[i] = detail::ConvertToUid(uuid);
        }
        NN_RESULT_SUCCESS;
    }

public:
    static Result Load(
        detail::Uuid* pOutInstanceId,
        int* pOutActualUserCount, Uid* outUsers, int userCount,
        int* pOutActualPendingCount, Uid* outPendings, int pendingCount,
        const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(userCount == UserCountMax);
        NN_SDK_ASSERT(pendingCount == 1);

        auto r = LoadImpl(pOutInstanceId, outUsers, userCount, outPendings, pendingCount, storage);
        if (fs::ResultPathNotFound::Includes(r))
        {
            // ユーザー数 0 で初期化
            *pOutInstanceId = storage.GenerateUuidWithContext();
            for (auto i = 0; i < userCount; ++ i)
            {
                outUsers[i] = InvalidUid;
            }
            *pOutActualUserCount = 0;
            for (auto i = 0; i < pendingCount; ++ i)
            {
                outPendings[i] = InvalidUid;
            }
            *pOutActualPendingCount = 0;
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_DO(r);

        *pOutActualUserCount = 0;
        for (auto i = 0; i < userCount && outUsers[i]; ++ i)
        {
            ++ *pOutActualUserCount;
        }
        *pOutActualPendingCount = 0;
        for (auto i = 0; i < pendingCount && outPendings[i]; ++ i)
        {
            ++ *pOutActualPendingCount;
        }
        NN_RESULT_SUCCESS;
    }
    static Result Store(
        const detail::Uuid& instanceId,
        const Uid* users, int userCount,
        const Uid* pendings, int pendingCount,
        const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(userCount == UserCountMax);
        NN_SDK_ASSERT(pendingCount == 1);

        SaveData save = {
            detail::ConvertHostToNetwork(SaveData::Header),
            0x00ull,
            instanceId.Serialize()};
        for (auto i = 0; i < userCount; ++ i)
        {
            save.users[i] = detail::ConvertToUuid(users[i]).Serialize();
        }
        for (auto i = 0; i < pendingCount; ++ i)
        {
            save.pendings[i] = detail::ConvertToUuid(pendings[i]).Serialize();
        }

        char path[64];
        detail::PathUtil::GetUserListPath(path, sizeof(path), storage.GetRootPath());
        return storage.GetFileSystem().Write(path, &save, sizeof(save));
    }
};

}}} // ~namespace nn::account::user
