﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "account_ProfileAdaptor.h"

#include "../detail/account_ByteUtil.h"
#include "../detail/account_CacheUtil.h"
#include "../detail/account_PathUtil.h"
#include <nn/account/baas/account_BaasTypes.h>
#include <nn/account/profile/account_ProfileStorage.h>
#include <nn/account/account_Config.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_ResultPrivate.h>

#include <algorithm>
#include <limits>
#include <cstring>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Base64.h>
#include <nn/util/util_BitPack.h>
#include <nn/util/util_CharacterEncoding.h>

namespace nn { namespace account { namespace profile {

namespace
{

/** -------------------------------------------------------------------------------------------
    BaaS が識別しない NX 固有のプロフィール情報
 */

struct ExtraDataSerializer
{
private:
    struct PackedProfile
    {
        detail::SerializedUuid author;
        uint64_t timeStamp;
        char _padding[16];
        char userData[UserDataBytesMax];

        static PackedProfile Pack(const ProfileBase& profile, const UserData& userData) NN_NOEXCEPT
        {
            PackedProfile p;
            p.author = detail::ConvertToUuid(profile.author).Serialize();
            p.timeStamp = detail::ConvertHostToNetwork(profile.timeStamp);
            std::memcpy(p.userData, userData.data, sizeof(userData.data));
            // padding
            std::memset(p._padding, 0x00, sizeof(p._padding));
            return p;
        }
        static void Unpack(ProfileBase* pOutProfile, UserData* pOutUserData, const PackedProfile& profile) NN_NOEXCEPT
        {
            ProfileBase base = DefaultProfileBase;
            base.author = detail::ConvertToUid(profile.author.Deserialize());
            base.timeStamp = detail::ConvertNetworkToHost(profile.timeStamp);
            base.nickname[0] = '\0';
            *pOutProfile = base;
            std::memcpy(pOutUserData->data, profile.userData, sizeof(pOutUserData->data));
        }
    };
    NN_STATIC_ASSERT(sizeof(PackedProfile) == 168);
    NN_STATIC_ASSERT((sizeof(PackedProfile) * 4) / 3 == 224);
    NN_STATIC_ASSERT((sizeof(PackedProfile) * 4) % 3 == 0);
    static const size_t CodedPackedProfileSize = sizeof(PackedProfile) * 4 / 3;

    static const char Header[9];
    static const size_t TotalCodedLength = (sizeof(Header) - 1) + CodedPackedProfileSize;
    NN_STATIC_ASSERT(TotalCodedLength <= baas::UserProfile::ExtraDataBytesForTransfer);


public:
    static Result Serialize(
        char* outString, size_t outStringBytes,
        const ProfileBase& profile, const UserData& userData) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(outStringBytes > TotalCodedLength);
        NN_SDK_REQUIRES(profile);

        std::strncpy(outString, Header, sizeof(Header));
        auto offset = sizeof(Header) - 1;

        auto p = PackedProfile::Pack(profile, userData);
        auto profileString = outString + offset;
        auto profileStringBytes = outStringBytes - offset;
        auto rv = util::Base64::ToBase64String(
            profileString, profileStringBytes,
            &p, sizeof(p), util::Base64::Mode_NormalNoLinefeed);
        NN_RESULT_THROW_UNLESS(rv == util::Base64::Status_Success, ResultLocalDataBroken());
        NN_RESULT_THROW_UNLESS(strnlen(profileString, profileStringBytes) == CodedPackedProfileSize, ResultLocalDataBroken());
        offset += CodedPackedProfileSize;

        NN_SDK_ASSERT(offset == TotalCodedLength);
        NN_SDK_ASSERT(strnlen(outString, outStringBytes) == offset);
        NN_RESULT_SUCCESS;
    }
    static Result Deserialize(
        ProfileBase* pOutProfile, UserData* pOutUserData,
        const char* string, size_t stringBytes) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(stringBytes > TotalCodedLength, ResultBaasDataBroken());
        NN_RESULT_THROW_UNLESS(std::memcmp(string, Header, sizeof(Header) - 1) == 0, ResultBaasDataBroken());
        auto offset = sizeof(Header) - 1;

        PackedProfile packed;
        const auto profileString = string + offset;
        size_t bytes;
        auto rv = util::Base64::FromBase64String(
            &bytes, &packed, sizeof(packed),
            profileString, util::Base64::Mode_NormalNoLinefeed);
        NN_RESULT_THROW_UNLESS(rv == util::Base64::Status_Success, ResultBaasDataBroken());
        NN_RESULT_THROW_UNLESS(bytes == sizeof(packed), ResultBaasDataBroken());
        offset += CodedPackedProfileSize;

        NN_SDK_ASSERT(offset == TotalCodedLength);
        NN_SDK_ASSERT(string[offset] == '\0');

        PackedProfile::Unpack(pOutProfile, pOutUserData, packed);
        NN_RESULT_THROW_UNLESS(*pOutProfile, ResultBaasDataBroken());
        NN_RESULT_SUCCESS;
    }
};
const char ExtraDataSerializer::Header[] = "01606150";

inline bool operator ==(const ProfileBase& lhs, const ProfileBase& rhs) NN_NOEXCEPT
{
    return true
        && lhs.timeStamp == rhs.timeStamp
        && lhs.author == rhs.author
        && std::strncmp(lhs.nickname, rhs.nickname, sizeof(lhs.nickname)) == 0;
}
} // ~namespace nn::account::profile::<anonymous>

/** -------------------------------------------------------------------------------------------
    ProfileAdaptor の実装
 */
ProfileAdaptor::ProfileAdaptor(ProfileStorage* pStorage, const Uid& uid) NN_NOEXCEPT
    : m_pStorage(pStorage)
    , m_Uid(uid)
    , m_ForceImport(false)
{
}

ProfileAdaptor::ProfileAdaptor(ProfileStorage* pStorage, const Uid& uid, bool forceImport) NN_NOEXCEPT
    : m_pStorage(pStorage)
    , m_Uid(uid)
    , m_ForceImport(forceImport)
{
}

Result ProfileAdaptor::Decode(ProfileBase* pOut, UserData* pOutUserData, const baas::UserProfileBase& remote) const NN_NOEXCEPT
{
    ProfileBase profile;
    NN_RESULT_DO(ExtraDataSerializer::Deserialize(&profile, pOutUserData, remote.extraData, sizeof(remote.extraData)));
    std::strncpy(profile.nickname, remote.nickname, sizeof(profile.nickname));
    NN_RESULT_THROW_UNLESS(profile, ResultBaasDataBroken());
    *pOut = profile;
    NN_RESULT_SUCCESS;
}

Result ProfileAdaptor::Evaluate(EvaluationResult* pOut, const ProfileBase& remote) const NN_NOEXCEPT
{
    ProfileBase local = DefaultProfileBase;
    NN_RESULT_DO(m_pStorage->GetProfile(&local, nullptr, m_Uid));

    if (m_ForceImport)
    {
        *pOut = EvaluationResult_Import;
        NN_RESULT_SUCCESS;
    }

    if (local.author == remote.author)
    {
        if (true
            && local == remote)
        {
            // 完全に一致
            *pOut = EvaluationResult_Stay;
            NN_RESULT_SUCCESS;
        }

        // 同一UA由来のプロフィールであれば、内容が違えば時刻関係なく更新する。
        // (時刻の巻き戻しがありえるので)
        *pOut = (local.author == m_Uid)
            ? EvaluationResult_Export // 双方自分自身のプロフィールであれば、サーバー側を更新
            : EvaluationResult_Import; // 双方同一の他人のプロフィールであれば、ローカル側を更新
        NN_RESULT_SUCCESS;
    }

    // タイムスタンプが等しい場合は、リモートの更新合戦になるのでアップロードしない。
    *pOut = (local.timeStamp > remote.timeStamp)
        ? EvaluationResult_Export
        : EvaluationResult_Import;
    NN_RESULT_SUCCESS;
}

Result ProfileAdaptor::Import(
    const ProfileBase& remote, const UserData& remoteUserData, const void* image, size_t imageSize) const NN_NOEXCEPT
{
    NN_RESULT_DO(m_pStorage->ApplyExternalData(m_Uid, remote, remoteUserData, image, imageSize, m_ForceImport));
    NN_RESULT_SUCCESS;
}

Result ProfileAdaptor::AcquireProfile(baas::UserProfileBase* pOut) const NN_NOEXCEPT
{
    ProfileBase local;
    UserData userData;
    NN_RESULT_DO(m_pStorage->GetProfile(&local, &userData, m_Uid));
    NN_SDK_ASSERT(local);

    auto length = GetNicknameLength(local.nickname, sizeof(local.nickname));
    NN_SDK_ASSERT(length < sizeof(pOut->nickname));
    std::strncpy(pOut->nickname, local.nickname, sizeof(pOut->nickname));
    pOut->nickname[length] = '\0';

    NN_RESULT_DO(ExtraDataSerializer::Serialize(pOut->extraData, sizeof(pOut->extraData), local, userData));
    NN_RESULT_SUCCESS;
}

Result ProfileAdaptor::CreateProfileImageCache(detail::Uuid* pOut, void* workBuffer, size_t workBufferSize) const NN_NOEXCEPT
{
    struct ImageIoBuffer
    {
        char input[ProfileImageBytesMax];
        char output[static_cast<size_t>((ProfileImageBytesMax * 7) / 5.0 + 0.5)];
    };

    NN_SDK_ASSERT(reinterpret_cast<uintptr_t>(workBuffer) % std::alignment_of<std::max_align_t>::value == 0);
    NN_SDK_ASSERT(workBufferSize >= sizeof(ImageIoBuffer));
    NN_UNUSED(workBufferSize);
    ImageIoBuffer* buffer = reinterpret_cast<ImageIoBuffer*>(workBuffer);

    size_t inSize;
    NN_RESULT_DO(m_pStorage->LoadImage(&inSize, buffer->input, sizeof(buffer->input), m_Uid));
    NN_SDK_ASSERT(inSize <= sizeof(buffer->input));

    auto status = util::Base64::ToBase64String(buffer->output, sizeof(buffer->output), buffer->input, inSize, util::Base64::Mode_NormalNoLinefeed);
    NN_ABORT_UNLESS(
        status == util::Base64::Status_Success,
        "[nn::account] -----------------------------------------------\n"
        "  ABORT: Base64 encoding failed: %d\n", status);
    auto length = strnlen(buffer->output, sizeof(buffer->output));

    NN_RESULT_DO(detail::CacheUtil::StoreCacheFile(pOut, buffer->output, length, m_pStorage->GetStorageRef()));
    NN_RESULT_SUCCESS;
}

}}}
