﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "account_AuthenticationAdaptor.h"

#include "../detail/account_ByteUtil.h"
#include "../detail/account_CacheUtil.h"
#include "../detail/account_PathUtil.h"
#include "../detail/account_TimeUtil.h"
#include "../http/account_HttpUtil.h"
#include <nn/account/ndas/account_ResultForNdas.h>
#include <nn/account/account_ResultForAdministrators.h>
#include <nn/account/account_ResultPrivate.h>

#include <utility>

#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace account { namespace ndas {

#define NN_ACCOUNT_PRINT_ENTRY_STATE(e) \
        NN_SDK_LOG("    %s : %s\n", (e)? "+": "-", (e).path)

namespace
{
const int NdasErrorCodeLength = 4;
const int NdasAppAuthErrorCodeOffset = 100;

inline bool ExtractErrorCodeFromString(uint16_t* pOut, const char* value) NN_NOEXCEPT
{
    if (value[NdasErrorCodeLength] != '\0')
    {
        return false;
    }
    int16_t code = 0;
    for (auto i = 0; i < NdasErrorCodeLength; ++ i)
    {
        auto c = value[i];
        if (!std::isdigit(c))
        {
            return false;
        }
        code = (code * 10) + c - '0';
    }
    *pOut = code;
    return true;
}

inline std::pair<bool, Result> ConvertAppAuthErrorCodeToResult(uint16_t errorCode) NN_NOEXCEPT
{
    const Result NdasAppAuthResultTable[] = {
        ResultNdasRemovedStatus(), // 廃止 SIGLO-68316
        ResultNdasRemovedStatus(), // 廃止 SIGLO-68316
        ResultNdasRemovedStatus(), // 廃止 SIGLO-68316
        ResultNdasStatusNo0103(),
        ResultNdasStatusNo0104(),
        ResultNdasStatusNo0105(),
        ResultNdasStatusNo0106(),
        ResultNdasStatusNo0107(),
        ResultNdasStatusNo0108(),
        ResultNdasStatusNo0109(),
        ResultNdasStatusNo0110(),
        ResultUnacceptableApplicationVersion(),
        ResultNdasStatusNo0112(),
        ResultNdasStatusNo0113(),
        ResultNdasRemovedStatus(), // 廃止 SIGLO-68316
        ResultNdasRemovedStatus(), // 廃止 SIGLO-68316
        ResultNdasRemovedStatus(), // 廃止 SIGLO-68316
        ResultNdasStatusNo0117(),
        ResultNdasStatusNo0118(),
        ResultNdasStatusNo0119(),
    };
    static_assert(
        std::extent<decltype(NdasAppAuthResultTable)>::value == 20,
        "NDAS ApplicationAuth Result definition");

    const auto Length = static_cast<int>(std::extent<decltype(NdasAppAuthResultTable)>::value);
    if (errorCode >= NdasAppAuthErrorCodeOffset
        && errorCode - NdasAppAuthErrorCodeOffset < Length)
    {
        return std::pair<bool, Result>(true, NdasAppAuthResultTable[errorCode - NdasAppAuthErrorCodeOffset]);
    }
    return std::pair<bool, Result>(false, ResultNotImplemented());
}

} // ~namespace nn::account::ndas::<anonymous>

/* --------------------------------------------------------------------------------------------
    アプリケーション認証
 */

ApplicationAuthenticationAdaptor::ApplicationAuthenticationAdaptor(
    ApplicationAuthenticationCache& appAuthCache,
    const detail::ApplicationInfo& appInfo) NN_NOEXCEPT
    : m_AppAuthCache(appAuthCache)
    , m_AppInfo(appInfo)
    // JSON から抽出するフィールド値
    , m_ExpirationEntry("$.expires_in")
    , m_TokenEntry("$.application_auth_token")
    , m_ErrorCodeEntry("$.errors[0].code")
    , m_ErrorMsgEntry("$.errors[0].message")
    , m_NasClientIdEntry("$.settings[0].nas:client_id")
    , m_NasRedirectUriEntry("$.settings[0].nas:redirect_uri")
    // Driver にセットする値
    , m_ExpirationAbs(-1)
    , m_TokenCacheId(detail::InvalidUuid)
    // エラー処理用
    , m_IoResult(ResultSuccess())
    , m_ErrorCode(0u)
{
    m_NasClientInfo.clientId = 0x00ull;
    m_NasClientInfo.redirectUri[0] = '\0';
}
ApplicationAuthenticationAdaptor::~ApplicationAuthenticationAdaptor() NN_NOEXCEPT
{
    if (m_TokenCacheId)
    {
        detail::CacheUtil::DeleteCacheFile(m_TokenCacheId, m_AppAuthCache.GetStorageRef());
    }
}
Result ApplicationAuthenticationAdaptor::Adapt(int32_t httpCode) NN_NOEXCEPT
{
    NN_RESULT_DO(m_IoResult);
    if (m_ErrorCodeEntry || !(m_ExpirationEntry && m_TokenEntry))
    {
        NN_SDK_LOG(
            "[nn::account] -----------------------------------------------\n"
            "  Error: NDAS ApplicationAuthentication Failed: %04d\n", m_ErrorCode);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_ExpirationEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_TokenEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_ErrorCodeEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_ErrorMsgEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_NasClientIdEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_NasRedirectUriEntry);

        if (m_ErrorCodeEntry)
        {
            auto r = ConvertAppAuthErrorCodeToResult(m_ErrorCode);
            if (r.first)
            {
                NN_RESULT_THROW(r.second);
            }
        }
        NN_RESULT_DO(http::HandleHttpStatusCode(httpCode));
        NN_RESULT_THROW(ResultNdasDataBroken());
    }

    ApplicationMeta meta = { nas::InvalidNasClientInfo };
    if (m_NasClientIdEntry && m_NasRedirectUriEntry)
    {
        meta.nasClientInfo = m_NasClientInfo;
    }
    m_AppAuthCache.Store(m_AppInfo, m_ExpirationAbs, m_TokenCacheId, meta);
    m_TokenCacheId = detail::InvalidUuid;
    NN_RESULT_SUCCESS;
}
void ApplicationAuthenticationAdaptor::Update(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT
{
    if (!CanContinue())
    {
        return;
    }

    if (m_ExpirationEntry.CanAccept(jsonPath))
    {
        m_ExpirationAbs = value + detail::GetUptimeInSeconds();
        m_ExpirationEntry.MarkAccepted();
    }
}
void ApplicationAuthenticationAdaptor::Update(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
{
    if (!CanContinue())
    {
        return;
    }

    if (m_TokenEntry.CanAccept(jsonPath))
    {
        if (valueLength > static_cast<int>(detail::NdasAppAuthTokenSizeMax))
        {
            return;
        }

        detail::Uuid cacheId;
        m_IoResult = detail::CacheUtil::StoreCacheFile(&cacheId, value, valueLength, m_AppAuthCache.GetStorageRef());
        if (!m_IoResult.IsSuccess())
        {
            return;
        }

        m_TokenCacheId = cacheId;
        m_TokenEntry.MarkAccepted();
    }
    else if (m_ErrorCodeEntry.CanAccept(jsonPath))
    {
        uint16_t code;
        if (!ExtractErrorCodeFromString(&code, value))
        {
            return;
        }

        m_ErrorCode = code;
        m_ErrorCodeEntry.MarkAccepted();
    }
    else if (m_ErrorMsgEntry.CanAccept(jsonPath))
    {
        NN_SDK_LOG(
            "[nn::account] NDAS ApplicationAuthentication ----------------\n"
            "  Error: \"%s\"\n", value);
        m_ErrorMsgEntry.MarkAccepted();
    }
    else if (m_NasClientIdEntry.CanAccept(jsonPath))
    {
        if (static_cast<uint32_t>(valueLength) <= sizeof(uint64_t) * 2)
        {
            m_NasClientInfo.clientId = detail::ExtractHexadecimal<uint64_t>(value, static_cast<size_t>(valueLength));
            m_NasClientIdEntry.MarkAccepted();
        }
    }
    else if (m_NasRedirectUriEntry.CanAccept(jsonPath))
    {
        if (valueLength < sizeof(m_NasClientInfo.redirectUri))
        {
            std::strncpy(m_NasClientInfo.redirectUri, value, sizeof(m_NasClientInfo.redirectUri));
            m_NasClientInfo.redirectUri[valueLength] = '\0';
            m_NasRedirectUriEntry.MarkAccepted();
        }
    }
}

#undef NN_ACCOUNT_PRINT_ENTRY_STATE

}}} // ~namespace nn::account::ndas
