﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "account_NasUserAdaptor.h"
#include "../detail/account_CacheUtil.h"
#include "../detail/account_PathUtil.h"

namespace nn {
namespace account {
namespace nas {

#define NN_ACCOUNT_PRINT_ENTRY_STATE(e) \
        NN_SDK_LOG("    %s : %s\n", (e)? "+": "-", (e).path)

NasUserAdaptor::NasUserAdaptor(
    const detail::AbstractLocalStorage& storage,
    char* buffer, size_t bufferSize) NN_NOEXCEPT
    : m_Storage(storage)
    , m_IdEntry("$.id")
    , m_IsChildEntry("$.isChild")
    , m_ScreenNameEntry("$.screenName")
    , m_EmailEntry("$.email")
    , m_LoginIdEntry("$.loginId")
    , m_NicknameEntry("$.nickname")
    , m_BirthdayEntry("$.birthday")
    , m_GenderEntry("$.gender")
    , m_AnalyticsPermissionsForTargetMarketingEntry("$.analyticsPermissions.targetMarketing.permitted")
    , m_AnalyticsPermissionsForInternalAnalysisEntry("$.analyticsPermissions.internalAnalysis.permitted")
    , m_LanguageEntry("$.language")
    , m_CountryEntry("$.country")
    , m_RegionEntry("$.region")
    , m_TimezoneEntry("$.timezone.id")
    , m_NnIdEntry("$.links.nintendoNetwork.id")
    , m_NnCreatedAtEntry("$.links.nintendoNetwork.createdAt")
    , m_TwitterIdEntry("$.links.twitter.id")
    , m_TwitterCreatedAtEntry("$.links.twitter.createdAt")
    , m_FacebookIdEntry("$.links.facebook.id")
    , m_FacebookCreatedAtEntry("$.links.facebook.createdAt")
    , m_GoogleIdEntry("$.links.google.id")
    , m_GoogleCreatedAtEntry("$.links.google.createdAt")
    , m_IsAdapted(false)
    , m_CacheId(detail::InvalidUuid)
    , m_Stream(m_Path, buffer, bufferSize, storage)
    , m_StackAllocator(m_StackBuffer, sizeof(m_StackBuffer))
    , m_Writer(m_Stream, &m_StackAllocator, 8)
{
    // ファイル初期化
    if (SetIoResult(detail::CacheUtil::ReserveCacheFile(&m_CacheId, SizeToReserve, m_Storage)).IsSuccess())
    {
        detail::PathUtil::GetCachePath(m_Path, sizeof(m_Path), m_CacheId, m_Storage.GetRootPath());
        if (!m_Writer.StartObject())
        {
            auto r = SetIoResult(ResultInsufficientBuffer());
            NN_UNUSED(r);
        }
    }
}
NasUserAdaptor::~NasUserAdaptor() NN_NOEXCEPT
{
    if (m_CacheId)
    {
        detail::CacheUtil::DeleteCacheFile(m_CacheId, m_Storage);
        m_CacheId = detail::InvalidUuid;
    }
}
Result NasUserAdaptor::AdaptImpl() NN_NOEXCEPT
{
    if (!(true
        && m_IdEntry
        && m_IsChildEntry
        && m_ScreenNameEntry
        && m_EmailEntry
        && m_LoginIdEntry
        && m_NicknameEntry
        && m_BirthdayEntry
        && m_GenderEntry
        && m_AnalyticsPermissionsForTargetMarketingEntry
        && m_AnalyticsPermissionsForInternalAnalysisEntry
        && m_LanguageEntry
        && m_CountryEntry
        && m_RegionEntry
        && m_TimezoneEntry))
    {
        NN_SDK_LOG("[nn::account] -----------------------------------------------\n");
        NN_SDK_LOG("  Error: NAS User Resource Download Failed\n");
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_IdEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_IsChildEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_ScreenNameEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_EmailEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_LoginIdEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_NicknameEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_BirthdayEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_GenderEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_AnalyticsPermissionsForTargetMarketingEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_AnalyticsPermissionsForInternalAnalysisEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_LanguageEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_CountryEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_RegionEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_NnIdEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_NnCreatedAtEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_TwitterIdEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_TwitterCreatedAtEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_FacebookIdEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_FacebookCreatedAtEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_GoogleIdEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_GoogleCreatedAtEntry);

        NN_RESULT_THROW(ResultNasDataBroken());
    }

    NN_RESULT_THROW_UNLESS(
        m_Writer.Key("isNnLinked") && m_Writer.Bool(m_NnIdEntry && m_NnCreatedAtEntry),
        ResultNotSupported());
    NN_RESULT_THROW_UNLESS(
        m_Writer.Key("isTwitterLinked") && m_Writer.Bool(m_TwitterIdEntry && m_TwitterCreatedAtEntry),
        ResultNotSupported());
    NN_RESULT_THROW_UNLESS(
        m_Writer.Key("isFacebookLinked") && m_Writer.Bool(m_FacebookIdEntry && m_FacebookCreatedAtEntry),
        ResultNotSupported());
    NN_RESULT_THROW_UNLESS(
        m_Writer.Key("isGoogleLinked") && m_Writer.Bool(m_GoogleIdEntry && m_GoogleCreatedAtEntry),
        ResultNotSupported());

    NN_RESULT_THROW_UNLESS(m_Writer.EndObject(), ResultNasDataBroken());
    NN_RESULT_THROW_UNLESS(m_Writer.IsComplete(), ResultNasDataBroken());

    size_t size;
    NN_RESULT_DO(m_Stream.Finalize(&size));
    auto lock = m_Storage.AcquireWriterLock();
    if (size < SizeToReserve)
    {
        NN_RESULT_DO(m_Storage.GetFileSystem().SetSize(m_Path, size));
    }
    m_IsAdapted = true;
    NN_RESULT_SUCCESS;
}
Result NasUserAdaptor::PullUserResourceCache(NintendoAccountId* pOutNaId, NasUserResourceCache& userResourceCache) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsAdapted);
    NN_RESULT_DO(userResourceCache.Store(m_Id, std::move(m_CacheId)));
    NN_SDK_ASSERT(!m_CacheId);
    *pOutNaId = m_Id;
    NN_RESULT_SUCCESS;
}
bool NasUserAdaptor::UpdateImpl(const JsonPathType& jsonPath, std::nullptr_t) NN_NOEXCEPT
{
    if (m_EmailEntry.CanAccept(jsonPath))
    {
        if (m_Writer.Key("email") && m_Writer.String(""))
        {
            m_EmailEntry.MarkAccepted();
        }
        return true;
    }
    else if (m_LoginIdEntry.CanAccept(jsonPath))
    {
        if (m_Writer.Key("loginId") && m_Writer.String(""))
        {
            m_LoginIdEntry.MarkAccepted();
        }
        return true;
    }
    else if (m_RegionEntry.CanAccept(jsonPath))
    {
        if (m_Writer.Key("region") && m_Writer.String(""))
        {
            m_RegionEntry.MarkAccepted();
        }
        return true;
    }
    return false;
}
bool NasUserAdaptor::UpdateImpl(const JsonPathType& jsonPath, bool value) NN_NOEXCEPT
{
    if (m_IsChildEntry.CanAccept(jsonPath))
    {
        if (m_Writer.Key("isChild") && m_Writer.Bool(value))
        {
            m_IsChildEntry.MarkAccepted();
        }
        return true;
    }
    else if (m_AnalyticsPermissionsForTargetMarketingEntry.CanAccept(jsonPath))
    {
        if (m_Writer.Key("analyticsForTargetMarketingPermitted") && m_Writer.Bool(value))
        {
            m_AnalyticsPermissionsForTargetMarketingEntry.MarkAccepted();
        }
        return true;
    }
    else if (m_AnalyticsPermissionsForInternalAnalysisEntry.CanAccept(jsonPath))
    {
        if (m_Writer.Key("analyticsForInternalAnalysisPermitted") && m_Writer.Bool(value))
        {
            m_AnalyticsPermissionsForInternalAnalysisEntry.MarkAccepted();
        }
        return true;
    }
    return false;
}
bool NasUserAdaptor::UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
{
#define SizeOfMember(type, member) sizeof(((type*)(nullptr))->member)

    NN_SDK_ASSERT(strlen(value) == static_cast<size_t>(valueLength));

    if (m_IdEntry.CanAccept(jsonPath))
    {
        if (valueLength == sizeof(uint64_t) * 2)
        {
            auto id = detail::ExtractHexadecimal<uint64_t>(value, valueLength);
            if (id != 0x00ull && m_Writer.Key("id") && m_Writer.String(value))
            {
                m_Id.id = id;
                m_IdEntry.MarkAccepted();
            }
        }
        return true;
    }
    else if (m_ScreenNameEntry.CanAccept(jsonPath))
    {
        if (valueLength < SizeOfMember(NasUserBase, screenName))
        {
            if (m_Writer.Key("screenName") && m_Writer.String(value))
            {
                m_ScreenNameEntry.MarkAccepted();
            }
        }
        return true;
    }
    else if (m_EmailEntry.CanAccept(jsonPath))
    {
        if (valueLength < SizeOfMember(NasUserBase, email))
        {
            if (m_Writer.Key("email") && m_Writer.String(value))
            {
                m_EmailEntry.MarkAccepted();
            }
        }
        return true;
    }
    else if (m_LoginIdEntry.CanAccept(jsonPath))
    {
        if (valueLength < SizeOfMember(NasUserBase, loginId))
        {
            if (m_Writer.Key("loginId") && m_Writer.String(value))
            {
                m_LoginIdEntry.MarkAccepted();
            }
        }
        return true;
    }
    else if (m_NicknameEntry.CanAccept(jsonPath))
    {
        if (valueLength < SizeOfMember(NasUserBase, nickname))
        {
            if (m_Writer.Key("nickname") && m_Writer.String(value))
            {
                m_NicknameEntry.MarkAccepted();
            }
        }
        return true;
    }
    else if (m_BirthdayEntry.CanAccept(jsonPath))
    {
        if (valueLength < SizeOfMember(NasUserBase, birthday))
        {
            if (m_Writer.Key("birthday") && m_Writer.String(value))
            {
                m_BirthdayEntry.MarkAccepted();
            }
        }
        return true;
    }
    else if (m_GenderEntry.CanAccept(jsonPath))
    {
        if (valueLength < SizeOfMember(NasUserBase, gender))
        {
            if (m_Writer.Key("gender") && m_Writer.String(value))
            {
                m_GenderEntry.MarkAccepted();
            }
        }
        return true;
    }
    else if (m_LanguageEntry.CanAccept(jsonPath))
    {
        if (valueLength < SizeOfMember(NasUserBase, language))
        {
            if (m_Writer.Key("language") && m_Writer.String(value))
            {
                m_LanguageEntry.MarkAccepted();
            }
        }
        return true;
    }
    else if (m_CountryEntry.CanAccept(jsonPath))
    {
        if (valueLength < SizeOfMember(NasUserBase, country))
        {
            if (m_Writer.Key("country") && m_Writer.String(value))
            {
                m_CountryEntry.MarkAccepted();
            }
        }
        return true;
    }
    else if (m_RegionEntry.CanAccept(jsonPath))
    {
        if (valueLength < SizeOfMember(NasUserBase, region))
        {
            if (m_Writer.Key("region") && m_Writer.String(value))
            {
                m_RegionEntry.MarkAccepted();
            }
        }
        return true;
    }
    else if (m_TimezoneEntry.CanAccept(jsonPath))
    {
        if (valueLength < SizeOfMember(NasUserBase, timezone))
        {
            if (m_Writer.Key("timezone") && m_Writer.String(value))
            {
                m_TimezoneEntry.MarkAccepted();
            }
        }
        return true;
    }
    else if (m_NnIdEntry.CanAccept(jsonPath))
    {
        m_NnIdEntry.MarkAccepted();
        return true;
    }
    else if (m_TwitterIdEntry.CanAccept(jsonPath))
    {
        m_TwitterIdEntry.MarkAccepted();
        return true;
    }
    else if (m_FacebookIdEntry.CanAccept(jsonPath))
    {
        m_FacebookIdEntry.MarkAccepted();
        return true;
    }
    else if (m_GoogleIdEntry.CanAccept(jsonPath))
    {
        m_GoogleIdEntry.MarkAccepted();
        return true;
    }
    return false;

#undef SizeOfMember
} // NOLINT(readability/fn_size)
bool NasUserAdaptor::UpdateImpl(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT
{
    NN_UNUSED(value);
    if (m_NnCreatedAtEntry.CanAccept(jsonPath))
    {
        m_NnCreatedAtEntry.MarkAccepted();
        return true;
    }
    else if (m_TwitterCreatedAtEntry.CanAccept(jsonPath))
    {
        m_TwitterCreatedAtEntry.MarkAccepted();
        return true;
    }
    else if (m_FacebookCreatedAtEntry.CanAccept(jsonPath))
    {
        m_FacebookCreatedAtEntry.MarkAccepted();
        return true;
    }
    else if (m_GoogleCreatedAtEntry.CanAccept(jsonPath))
    {
        m_GoogleCreatedAtEntry.MarkAccepted();
        return true;
    }
    return false;

}

NasUserCacheAdaptor::NasUserCacheAdaptor(NasUserBase* pOut) NN_NOEXCEPT
    : m_IsChildEntry("$.isChild")
    , m_ScreenNameEntry("$.screenName")
    , m_EmailEntry("$.email")
    , m_LoginIdEntry("$.loginId")
    , m_NicknameEntry("$.nickname")
    , m_BirthdayEntry("$.birthday")
    , m_GenderEntry("$.gender")
    , m_AnalyticsOptedInEntry("$.analyticsOptedIn")
    , m_AnalyticsForTargetMarketingPermittedEntry("$.analyticsForTargetMarketingPermitted")
    , m_AnalyticsForInternalAnalysisPermittedEntry("$.analyticsForInternalAnalysisPermitted")
    , m_LanguageEntry("$.language")
    , m_CountryEntry("$.country")
    , m_RegionEntry("$.region")
    , m_TimezoneEntry("$.timezone")
    , m_IsNnLinkedEntry("$.isNnLinked")
    , m_IsTwitterLinkedEntry("$.isTwitterLinked")
    , m_IsFacebookLinkedEntry("$.isFacebookLinked")
    , m_IsGoogleLinkedEntry("$.isGoogleLinked")
    , m_pOutNasUserBase(pOut)
{
    // デフォルト値
    m_pOutNasUserBase->isChild = false;
    m_pOutNasUserBase->screenName[0] = '\0';
    m_pOutNasUserBase->email[0] = '\0';
    m_pOutNasUserBase->loginId[0] = '\0';
    m_pOutNasUserBase->nickname[0] = '\0';
    m_pOutNasUserBase->birthday[0] = '\0';
    m_pOutNasUserBase->gender[0] = '\0';
    m_pOutNasUserBase->analyticsOptedIn = false;
    m_pOutNasUserBase->analyticsForTargetMarketingPermitted = false;
    m_pOutNasUserBase->analyticsForInternalAnalysisPermitted = false;
    m_pOutNasUserBase->language[0] = '\0';
    m_pOutNasUserBase->country[0] = '\0';
    m_pOutNasUserBase->region[0] = '\0';
    m_pOutNasUserBase->timezone[0] = '\0';

    m_pOutNasUserBase->isNnLinked = false;
    m_pOutNasUserBase->isTwitterLinked = false;
    m_pOutNasUserBase->isFacebookLinked = false;
    m_pOutNasUserBase->isGoogleLinked = false;
}
Result NasUserCacheAdaptor::Adapt() NN_NOEXCEPT
{
    NN_SDK_ASSERT(false
        || m_AnalyticsOptedInEntry
        || (m_AnalyticsForTargetMarketingPermittedEntry && m_AnalyticsForInternalAnalysisPermittedEntry));
    if (!m_AnalyticsOptedInEntry)
    {
        m_pOutNasUserBase->analyticsOptedIn = false
            || m_pOutNasUserBase->analyticsForTargetMarketingPermitted
            || m_pOutNasUserBase->analyticsForInternalAnalysisPermitted;
    }
    NN_RESULT_SUCCESS;
}
void NasUserCacheAdaptor::Update(const JsonPathType& jsonPath, bool value) NN_NOEXCEPT
{
    if (m_IsChildEntry.CanAccept(jsonPath))
    {
        m_pOutNasUserBase->isChild = value;
        m_IsChildEntry.MarkAccepted();
    }
    else if (m_AnalyticsOptedInEntry.CanAccept(jsonPath))
    {
        m_pOutNasUserBase->analyticsOptedIn = value;
        m_AnalyticsOptedInEntry.MarkAccepted();
    }
    else if (m_AnalyticsForTargetMarketingPermittedEntry.CanAccept(jsonPath))
    {
        m_pOutNasUserBase->analyticsForTargetMarketingPermitted = value;
        m_AnalyticsForTargetMarketingPermittedEntry.MarkAccepted();
    }
    else if (m_AnalyticsForInternalAnalysisPermittedEntry.CanAccept(jsonPath))
    {
        m_pOutNasUserBase->analyticsForInternalAnalysisPermitted = value;
        m_AnalyticsForInternalAnalysisPermittedEntry.MarkAccepted();
    }
    else if (m_IsNnLinkedEntry.CanAccept(jsonPath))
    {
        m_pOutNasUserBase->isNnLinked = value;
        m_IsNnLinkedEntry.MarkAccepted();
    }
    else if (m_IsTwitterLinkedEntry.CanAccept(jsonPath))
    {
        m_pOutNasUserBase->isTwitterLinked = value;
        m_IsTwitterLinkedEntry.MarkAccepted();
    }
    else if (m_IsFacebookLinkedEntry.CanAccept(jsonPath))
    {
        m_pOutNasUserBase->isFacebookLinked = value;
        m_IsFacebookLinkedEntry.MarkAccepted();
    }
    else if (m_IsGoogleLinkedEntry.CanAccept(jsonPath))
    {
        m_pOutNasUserBase->isGoogleLinked = value;
        m_IsGoogleLinkedEntry.MarkAccepted();
    }
}
void NasUserCacheAdaptor::Update(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
{
    NN_SDK_ASSERT(strlen(value) == static_cast<size_t>(valueLength));

    if (m_ScreenNameEntry.CanAccept(jsonPath))
    {
        if (valueLength < sizeof(m_pOutNasUserBase->screenName))
        {
            std::strncpy(m_pOutNasUserBase->screenName, value, valueLength);
            m_pOutNasUserBase->screenName[valueLength] = '\0';
            m_ScreenNameEntry.MarkAccepted();
        }
    }
    else if (m_EmailEntry.CanAccept(jsonPath))
    {
        if (valueLength < sizeof(m_pOutNasUserBase->email))
        {
            std::strncpy(m_pOutNasUserBase->email, value, valueLength);
            m_pOutNasUserBase->email[valueLength] = '\0';
            m_EmailEntry.MarkAccepted();
        }
    }
    else if (m_LoginIdEntry.CanAccept(jsonPath))
    {
        if (valueLength < sizeof(m_pOutNasUserBase->loginId))
        {
            std::strncpy(m_pOutNasUserBase->loginId, value, valueLength);
            m_pOutNasUserBase->loginId[valueLength] = '\0';
            m_LoginIdEntry.MarkAccepted();
        }
    }
    else if (m_NicknameEntry.CanAccept(jsonPath))
    {
        if (valueLength < sizeof(m_pOutNasUserBase->nickname))
        {
            std::strncpy(m_pOutNasUserBase->nickname, value, valueLength);
            m_pOutNasUserBase->nickname[valueLength] = '\0';
            m_NicknameEntry.MarkAccepted();
        }
    }
    else if (m_BirthdayEntry.CanAccept(jsonPath))
    {
        if (valueLength < sizeof(m_pOutNasUserBase->birthday))
        {
            std::strncpy(m_pOutNasUserBase->birthday, value, valueLength);
            m_pOutNasUserBase->birthday[valueLength] = '\0';
            m_BirthdayEntry.MarkAccepted();
        }
    }
    else if (m_GenderEntry.CanAccept(jsonPath))
    {
        if (valueLength < sizeof(m_pOutNasUserBase->gender))
        {
            std::strncpy(m_pOutNasUserBase->gender, value, valueLength);
            m_pOutNasUserBase->gender[valueLength] = '\0';
            m_GenderEntry.MarkAccepted();
        }
    }
    else if (m_LanguageEntry.CanAccept(jsonPath))
    {
        if (valueLength < sizeof(m_pOutNasUserBase->language))
        {
            std::strncpy(m_pOutNasUserBase->language, value, valueLength);
            m_pOutNasUserBase->language[valueLength] = '\0';
            m_LanguageEntry.MarkAccepted();
        }
    }
    else if (m_CountryEntry.CanAccept(jsonPath))
    {
        if (valueLength < sizeof(m_pOutNasUserBase->country))
        {
            std::strncpy(m_pOutNasUserBase->country, value, valueLength);
            m_pOutNasUserBase->country[valueLength] = '\0';
            m_CountryEntry.MarkAccepted();
        }
    }
    else if (m_RegionEntry.CanAccept(jsonPath))
    {
        if (valueLength < sizeof(m_pOutNasUserBase->region))
        {
            std::strncpy(m_pOutNasUserBase->region, value, valueLength);
            m_pOutNasUserBase->region[valueLength] = '\0';
            m_RegionEntry.MarkAccepted();
        }
    }
    else if (m_TimezoneEntry.CanAccept(jsonPath))
    {
        if (valueLength < sizeof(m_pOutNasUserBase->timezone))
        {
            std::strncpy(m_pOutNasUserBase->timezone, value, valueLength);
            m_pOutNasUserBase->timezone[valueLength] = '\0';
            m_TimezoneEntry.MarkAccepted();
        }
    }
} // NOLINT(readability/fn_size)

#undef NN_ACCOUNT_PRINT_ENTRY_STATE

}
}
} // ~namespace nn::account::nas
