﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/nas/account_NasOp2Driver.h>

#include <nn/account/http/account_CurlInputStream.h>
#include <nn/account/http/account_ResultForHttp.h>
#include <nn/account/json/account_RapidJsonApi.h>

#include "account_NasAuthorizationHeader.h"
#include "account_NasOp2Adaptor.h"

namespace nn { namespace account { namespace nas {
namespace {
const char Op2Fqdn[] = "capi.%.op2.nintendo.net";
int GetUrlToGetMembership(char* url, size_t bufferSize, const NintendoAccountId& naId) NN_NOEXCEPT
{
    auto l = nn::util::SNPrintf(url, bufferSize, "https://%s/v1/users/%016llx/membership", Op2Fqdn, naId.id);
    NN_ABORT_UNLESS(l < static_cast<int>(bufferSize)); // 終端チェック
    return l;
}

Result SetupStreamForGetMembership(
    http::CurlInputStream& stream,
    struct curl_slist** headers,
    const NintendoAccountId& naId,
    const detail::Uuid& accessTokenCacheId,
    const detail::AbstractLocalStorage& storage,
    char* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(workBufferSize >= RequiredBufferSizeForAuthorizationHeader);

    // Authorization ヘッダの準備
    NN_RESULT_DO(CreateAuthorizationHeader(workBuffer, workBufferSize, accessTokenCacheId, storage));
    *headers = curl_slist_append(*headers, workBuffer);
    NN_RESULT_THROW_UNLESS(*headers != nullptr, http::ResultCurlSlistAppendFailure());

    // リクエスト生成
    NN_RESULT_DO(stream.SetHeaders(*headers));

    // URL の作成
    char url[256];
    auto l = GetUrlToGetMembership(url, sizeof(url), naId);
    NN_SDK_ASSERT(static_cast<uint32_t>(l)  < sizeof(url));
    NN_UNUSED(l);
    NN_RESULT_DO(stream.SetUrl(url));

    NN_RESULT_SUCCESS;
}

} // ~namesapce nn::account::nas::<anonymous>

const size_t NasOp2Driver::RequiredBufferSize = 8 * 1024u;

Result NasOp2Driver::GetMembership(
    NetworkServiceLicense* pOutLicense, time::PosixTime* pOutExpiration,
    const NintendoAccountId& id, const detail::Uuid& accessTokenCacheId,
    CURL* curlHandle, void* rawBuffer, size_t bufferSize,
    const detail::Cancellable* pCancellable) const NN_NOEXCEPT
{
    // バッファ定義
    struct Buffer
    {
        union
        {
            char workBuffer[RequiredBufferSizeForAuthorizationHeader];
            struct
            {
                char stringBuffer[128]; // User リソースのうち、欲しいフィールドの最大長
                char inputBuffer[detail::IoBufferSizeMin];
            } io;
        } u;
    };;
    NN_STATIC_ASSERT(sizeof(Buffer) <= RequiredBufferSize);
    NN_STATIC_ASSERT(std::alignment_of<Buffer>::value <= std::alignment_of<std::max_align_t>::value);

    // 事前条件
    NN_SDK_REQUIRES_NOT_NULL(pOutLicense);
    NN_SDK_REQUIRES_NOT_NULL(pOutExpiration);
    NN_SDK_REQUIRES(id);
    NN_SDK_REQUIRES(accessTokenCacheId);
    NN_SDK_REQUIRES(curlHandle != nullptr);
    NN_SDK_REQUIRES(rawBuffer != nullptr);
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(rawBuffer) % std::alignment_of<std::max_align_t>::value == 0);
    NN_SDK_REQUIRES(bufferSize >= sizeof(Buffer));

    // リクエストの作成
    auto* buffer = reinterpret_cast<Buffer*>(rawBuffer);

    struct curl_slist *headers = nullptr;
    NN_UTIL_SCOPE_EXIT
    {
        curl_slist_free_all(headers);
    };

    http::CurlInputStream stream(curlHandle, pCancellable);
    NN_RESULT_DO(stream.Initialize());
    stream.SetStringBuffer(buffer->u.io.stringBuffer, sizeof(buffer->u.io.stringBuffer));
    stream.SetInputBuffer(buffer->u.io.inputBuffer, bufferSize - offsetof(Buffer, u.io.inputBuffer));

    NN_RESULT_DO(SetupStreamForGetMembership(
        stream, &headers, id, accessTokenCacheId, m_Storage,
        buffer->u.workBuffer, sizeof(buffer->u.workBuffer)));

    // 通信の実行と結果の適用
    NasOp2MembershipAdaptor adaptor;
    NN_RESULT_DO(stream.Open());
    NN_RESULT_DO(json::ImportJsonByRapidJson(adaptor, stream, pCancellable));
    stream.Close();
    NN_RESULT_DO(adaptor.Adapt(stream.GetHttpCode()));

    // 結果の取得
    adaptor.GetMembership(pOutLicense, pOutExpiration);
    NN_RESULT_SUCCESS;
}

}}} // ~namespace nn::account::nas
