﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "account_NasProblemDetails.h"
#include "../http/account_RedirectUriParser.h"
#include <nn/account/detail/account_InternalTypes.h>
#include <nn/account/detail/account_LocalStorage.h>
#include <nn/account/json/account_JsonAdaptor.h>
#include <nn/account/nas/account_NasInternalTypes.h>
#include <nn/account/nas/account_NasTypes.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_TypesForSystemServices.h>

#include <cstddef>
#include <cstring>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn {
namespace account {
namespace nas {

class NasPrimaryTokenAdaptor final
    : public NasTokenRequestAdaptorBase<8, 32>
{
    NN_DISALLOW_COPY(NasPrimaryTokenAdaptor);

private:
    const detail::AbstractLocalStorage& m_Storage;

    json::LookupEntry m_AccessTokenEntry;
    json::LookupEntry m_IdTokenEntry;
    json::LookupEntry m_RefreshTokenEntry;
    json::LookupEntry m_ExpirationEntry;

    detail::Uuid m_AccessTokenCacheId;
    detail::Uuid m_IdTokenCacheId;
    detail::Uuid m_RefreshTokenCacheId;
    int64_t m_ExpirationAbs;

    bool m_IsAdapted;

public:
    typedef json::JsonPath<8, 32> JsonPathType;

public:
    explicit NasPrimaryTokenAdaptor(const detail::AbstractLocalStorage& storage) NN_NOEXCEPT;
    ~NasPrimaryTokenAdaptor() NN_NOEXCEPT;

    Result PullPrimaryTokens(NasCredentialCache* pOutCredentialCache, NasAccessTokenCache* pOutAccessTokenCache) NN_NOEXCEPT;

    virtual Result AdaptImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual bool UpdateImpl(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT NN_OVERRIDE;
    virtual bool UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT NN_OVERRIDE;
};

class NasAccessTokenAdaptor final
    : public NasTokenRequestAdaptorBase<8, 32>
{
    NN_DISALLOW_COPY(NasAccessTokenAdaptor);

private:
    const detail::AbstractLocalStorage& m_Storage;

    json::LookupEntry m_AccessTokenEntry;
    json::LookupEntry m_ExpirationEntry;

    detail::Uuid m_AccessTokenCacheId;
    int64_t m_ExpirationAbs;

    bool m_IsAdapted;

public:
    typedef json::JsonPath<8, 32> JsonPathType;

public:
    explicit NasAccessTokenAdaptor(const detail::AbstractLocalStorage& storage) NN_NOEXCEPT;
    ~NasAccessTokenAdaptor() NN_NOEXCEPT;

    Result PullAccessToken(NasAccessTokenCache* pOutAccessTokenCache) NN_NOEXCEPT;

    virtual Result AdaptImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual bool UpdateImpl(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT NN_OVERRIDE;
    virtual bool UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT NN_OVERRIDE;
};

class NasAuthorizationErrorAdaptor
    : public http::RedirectUriParser<detail::NasCallbackUriLengthMax + 1>
{
    // このアダプタは application/json 形式の応答を受け取るのではなく、
    // コールバック URI のクエリパラメータや URI フラグメントを解釈して応答を取得する。

    NN_DISALLOW_COPY(NasAuthorizationErrorAdaptor);

protected:
    typedef http::RedirectUriParser<detail::NasCallbackUriLengthMax + 1> ParserBase;

private:
    const State& m_State;

    http::UriLookupEntry m_StateEntry;
    http::UriLookupEntry m_ErrorEntry;
    http::UriLookupEntry m_ErrorDetailEntry;

    NasProblemWithDetail m_Problem;

    virtual void UpdateImplImpl(const char* key, size_t keyLength, const char* value, size_t valueLength) NN_NOEXCEPT = 0;
    bool UpdateError(const char* key, size_t keyLength, const char* value, size_t valueLength) NN_NOEXCEPT;
    virtual void UpdateImpl(const char* key, size_t keyLength, const char* value, size_t valueLength) NN_NOEXCEPT final NN_OVERRIDE
    {
        if (UpdateError(key, keyLength, value, valueLength))
        {
            return;
        }
        UpdateImplImpl(key, keyLength, value, valueLength);
    }

    Result AdaptError() NN_NOEXCEPT;
    virtual Result AdaptImpl() NN_NOEXCEPT = 0;

public:
    NasAuthorizationErrorAdaptor(
        const char (&redirectUri)[detail::NasCallbackUriLengthMax + 1], const State& state) NN_NOEXCEPT;
    Result Adapt() NN_NOEXCEPT
    {
        NN_RESULT_DO(AdaptError());
        NN_RESULT_DO(AdaptImpl());
        NN_RESULT_SUCCESS;
    }
};

class NasServiceAuthorizationAdaptor final
    : public NasAuthorizationErrorAdaptor
{
    NN_DISALLOW_COPY(NasServiceAuthorizationAdaptor);

private:
    const detail::AbstractLocalStorage& m_Storage;

    http::UriLookupEntry m_CodeEntry;
    detail::Uuid m_CodeCacheId;

    Result m_IoResult;
    bool m_IsAdapted;

    bool CanContinue() const NN_NOEXCEPT
    {
        return m_IoResult.IsSuccess();
    }
    virtual void UpdateImplImpl(const char* key, size_t keyLength, const char* value, size_t valueLength) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result AdaptImpl() NN_NOEXCEPT NN_OVERRIDE;

public:
    NasServiceAuthorizationAdaptor(
        const char (&redirectUri)[detail::NasCallbackUriLengthMax + 1], const State& state,
        const detail::AbstractLocalStorage& storage) NN_NOEXCEPT;
    ~NasServiceAuthorizationAdaptor() NN_NOEXCEPT;
    Result PullAuthorizationCode(detail::Uuid* pOut) NN_NOEXCEPT;
};

class NasApplicationAuthorizationAdaptor final
    : public NasAuthorizationErrorAdaptor
{
    NN_DISALLOW_COPY(NasApplicationAuthorizationAdaptor);

private:
    http::UriLookupEntry m_CodeEntry;
    http::UriLookupEntry m_IdTokenEntry;
    NasApplicationAuthorization* m_pOutAuthorization;

    virtual void UpdateImplImpl(const char* key, size_t keyLength, const char* value, size_t valueLength) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result AdaptImpl() NN_NOEXCEPT NN_OVERRIDE;

public:
    NasApplicationAuthorizationAdaptor(
        NasApplicationAuthorization* pOutAuthorization,
        const char (&redirectUri)[detail::NasCallbackUriLengthMax + 1], const State& state) NN_NOEXCEPT;
};

class NasIdTokenAdaptor final
    : public NasAuthorizationErrorAdaptor
{
    NN_DISALLOW_COPY(NasIdTokenAdaptor);

private:
    const detail::AbstractLocalStorage& m_Storage;

    http::UriLookupEntry m_IdTokenEntry;
    char (&m_IdToken)[detail::NasIdTokenSizeMax];

    virtual void UpdateImplImpl(const char* key, size_t keyLength, const char* value, size_t valueLength) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result AdaptImpl() NN_NOEXCEPT NN_OVERRIDE;

public:
    NasIdTokenAdaptor(
        char (&idToken)[detail::NasIdTokenSizeMax],
        const char (&redirectUri)[detail::NasCallbackUriLengthMax + 1], const State& state,
        const detail::AbstractLocalStorage& storage) NN_NOEXCEPT;
};

} // ~namespace nn::account::nas
}
}
