﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/http/account_RedirectionCaptor.h>

#include <nn/nn_SdkLog.h>
#include <nn/account/account_ResultPrivate.h>
#include <nn/account/http/account_ResultForHttp.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include "account_HttpUtil.h"

namespace nn { namespace account { namespace http {
namespace {
inline bool IsSpace(char c) NN_NOEXCEPT
{
    return c == ' ' || c == '\r' || c == '\n' || c == '\t';
}
const char* TrimSpace(size_t* pOutActualLength, const char* ptr, size_t length) NN_NOEXCEPT
{
    auto actualLength = length;

    size_t leadingNbspCount = 0;
    while (leadingNbspCount < actualLength && IsSpace(ptr[leadingNbspCount]))
    {
        ++ leadingNbspCount;
    }
    ptr += leadingNbspCount;
    actualLength -= leadingNbspCount;

    size_t followingNbspCount = 0;
    while (followingNbspCount < actualLength && IsSpace(ptr[actualLength - followingNbspCount - 1]))
    {
        ++ followingNbspCount;
    }
    *pOutActualLength = actualLength - followingNbspCount;
    return ptr;
}
} // ~namespace nn::account::http::<anonymous>

size_t RedirectionCaptor::HeaderFunction(char* ptr, size_t unitBytes, size_t count, void* pContext) NN_NOEXCEPT
{
    const auto InputBytes = unitBytes * count;
#if 0
    if (true
        && InputBytes > 0
        && ptr[0] != '\n'
        && ptr[0] != '\r'
        && ptr[0] != ' ')
    {
        NN_SDK_LOG("HEADER ------------------------\n");
        NN_SDK_LOG("%.*s\n", InputBytes, ptr);
    }
#endif
    auto& obj = *reinterpret_cast<RedirectionCaptor*>(pContext);
    if (!(obj.m_IoResult.IsSuccess() && obj.m_BufferInfo.used == 0))
    {
        // すでに失敗している
        return nn::http::stream::CallbackErrorValue;
    }

    if (!obj.m_InProcess)
    {
        // "HTTP/x.x 302 "ヘッダを受け取っていない or Continue だった
        long statusCode;
        if (nn::http::stream::TryExtractStatusCodeFromHeader(&statusCode, ptr, InputBytes))
        {
            if (!(statusCode >= 0))
            {
                // 未知のステータスコードの場合はエラー
                obj.m_IoResult = ResultHttpUnsupportedStatusCode();
                return nn::http::stream::CallbackErrorValue;
            }

            if (statusCode == 302)
            {
                // リダイレクトを検出した
                obj.m_InProcess = true;
                return InputBytes;
            }

            // リダイレクト以外
            switch (statusCode / 100)
            {
            case 1:
            case 2:
                // 1xx, 2xx の場合は何もしない
                obj.m_InProcess = false;
                return InputBytes;
            default:
                // それ以外の場合は予期しないステータスコードなのでエラーにする。
                // (302 ではない 3xx 系は http::ResultHttpUnsupportedStatusCode になる。)
                obj.m_IoResult = HandleHttpStatusCode(statusCode);
                return nn::http::stream::CallbackErrorValue;
            }
        }
        // 302 のヘッダでもなく、ステータスコードでもない場合は読み飛ばす
        return InputBytes;
    }

    static const char Location[] = "Location: ";
    if (!(InputBytes >= sizeof(Location) && util::Strnicmp(ptr, Location, sizeof(Location) - 1) == 0))
    {
        // リダイレクト先の情報ではない
        return InputBytes;
    }

    // リダイレクト先
    size_t length;
    auto location = TrimSpace(&length, ptr + sizeof(Location) - 1, InputBytes - (sizeof(Location) - 1));
    if (!(length < obj.m_BufferInfo.total))
    {
        obj.m_IoResult = ResultInsufficientBuffer();
        return nn::http::stream::CallbackErrorValue;
    }
    strncpy(obj.m_BufferInfo.address, location, length);
    obj.m_BufferInfo.address[length] = '\0';
    obj.m_BufferInfo.used = length + 1;
    return nn::http::stream::CallbackErrorValue; // これ以上続ける必要はない
}

size_t RedirectionCaptor::WriteFunction(char*, size_t, size_t, void*) NN_NOEXCEPT
{
    // コンテントには用がないので即時中断する
    return static_cast<size_t>(-1);
}

RedirectionCaptor::RedirectionCaptor(CURL* curlHandle, const util::Cancelable* pCancellable) NN_NOEXCEPT
    : WebApiAccessorBase(curlHandle, pCancellable)
    , m_InProcess(false)
    , m_IoResult(ResultSuccess())
{
}

Result RedirectionCaptor::Initialize(char* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(buffer != nullptr);
    NN_SDK_ASSERT(bufferSize > 0);
    NN_RESULT_DO(WebApiAccessorBase::SetDefault());
    NN_RESULT_DO(WebApiAccessorBase::SetHeaderFunction(HeaderFunction, this));
    NN_RESULT_DO(WebApiAccessorBase::SetWriteFunction(WriteFunction, this));
    NN_RESULT_DO(WebApiAccessorBase::SetUserAgent(GetUserAgent()));
    NN_RESULT_DO(WebApiAccessorBase::SetErrorHandler(DefaultErrorHandler));
    m_BufferInfo.address = buffer;
    m_BufferInfo.total = bufferSize;
    m_BufferInfo.used = 0;
    NN_RESULT_SUCCESS;
}

Result RedirectionCaptor::Invoke() NN_NOEXCEPT
{
    NN_UTIL_SCOPE_EXIT
    {
        WebApiAccessorBase::Reset();
    };
    NN_SDK_ASSERT(m_BufferInfo.address != nullptr);

    NN_RESULT_TRY(WebApiAccessorBase::Perform())
    NN_RESULT_CATCH(ResultCurlErrorWriteError)
    {
        NN_RESULT_DO(m_IoResult);
        if (m_BufferInfo.used <= 0)
        {
            NN_RESULT_RETHROW;
        }
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_END_TRY

    NN_RESULT_DO(WebApiAccessorBase::GetResult());
    NN_RESULT_THROW(ResultHttpUnsupportedStatusCode());
}
const char* RedirectionCaptor::GetLocation() const NN_NOEXCEPT
{
    return m_BufferInfo.address;
}

}}} // ~namespace nn::account::http
