﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "../detail/account_ByteUtil.h"
#include <nn/account/account_TypesForSystemServices.h>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>

#if !defined(NN_SDK_BUILD_RELEASE)
#include <cstdlib>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#endif

namespace nn {
namespace account {
namespace http {

struct UriLookupEntry
{
    const char* path;
    bool found;

    explicit UriLookupEntry(const char* p) NN_NOEXCEPT
        : path(p)
        , found(false)
    {
        NN_SDK_ASSERT(p);
    }
    NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return found;
    }
    bool CanAccept(const char* key, size_t keyLength) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(keyLength > 0);
        NN_SDK_ASSERT(key[keyLength] == '=');
        NN_SDK_ASSERT(strnlen(key, keyLength) == keyLength);

        return !(*this) && (std::strncmp(path, key, keyLength) == 0);
    }
    void MarkAccepted() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!found);
        found = true;
    }
};

#define NN_ACCOUNT_RETURN_FALSE_UNLESS(cond) \
    do \
    { \
        if (!NN_STATIC_CONDITION(cond)) \
        { \
            return false; \
        }\
    } while(NN_STATIC_CONDITION(false))

template <size_t UriBaseSize>
class RedirectUriParser
{
private:
    const char (&m_UriBase)[UriBaseSize];

protected:
    virtual void UpdateImpl(const char* key, size_t keyLength, const char* value, size_t valueLength) NN_NOEXCEPT = 0;

    void Update(const char* key, size_t keyLength, const char* value, size_t valueLength) NN_NOEXCEPT
    {
#if 0
        // デバッグ用のログ出力
        size_t bufSize = 2048u;
        char* buf = reinterpret_cast<char*>(std::malloc(bufSize));
        NN_UTIL_SCOPE_EXIT
        {
            std::free(buf);
        };

        util::SNPrintf(buf, bufSize, "%.*s : %.*s", keyLength, key, valueLength, value);
        NN_SDK_LOG("Update: %s\n", buf);

        for (int i = 0; i < static_cast<int>(keyLength); ++ i)
        {
            NN_SDK_ASSERT(key[i] != '=');
            NN_SDK_ASSERT(key[i] != '\0');
        }
        NN_SDK_ASSERT(key[keyLength] == '=');
        for (int i = 0; i < static_cast<int>(valueLength); ++ i)
        {
            NN_SDK_ASSERT(value[i] != '&');
            NN_SDK_ASSERT(value[i] != '\0');
        }
        NN_SDK_ASSERT(value[valueLength] == '&' || value[valueLength] == '\0');
#endif // ~0
        UpdateImpl(key, keyLength, value, valueLength);
    }

public:
    explicit RedirectUriParser(const char (&uriBase)[UriBaseSize]) NN_NOEXCEPT
        : m_UriBase(uriBase)
    {
    }

    bool Parse(const char* document, size_t bufferSize) NN_NOEXCEPT
    {
        const auto BaseLength = strnlen(m_UriBase, UriBaseSize);
        NN_ACCOUNT_RETURN_FALSE_UNLESS(BaseLength < UriBaseSize);

        const auto DocumentLength = strnlen(document, bufferSize);
        NN_ACCOUNT_RETURN_FALSE_UNLESS(DocumentLength < bufferSize); // NULL 終端されている
        NN_ACCOUNT_RETURN_FALSE_UNLESS(strncmp(m_UriBase, document, BaseLength) == 0); // URI のベース部分の一致

        if (!(DocumentLength + 1 > BaseLength))
        {
            // 入力はないが、エラーではない
            return true;
        }

        // URI フラグメントとクエリパラメータの検査 (特に区別しない)
        NN_ACCOUNT_RETURN_FALSE_UNLESS(document[BaseLength] == '?' || document[BaseLength] == '#');
        auto key = &document[BaseLength + 1];
        while (key && key - document <= static_cast<int>(DocumentLength))
        {
            // 次のトークンとの区切り文字を探す
            auto p = detail::FindChar(key, DocumentLength - (key - document), '&');
            // length(token) = 次の区切り文字があればそれまでの長さ、そうでなければ終端までの長さ
            auto tokenLength = (p? p - key : DocumentLength - (key - document));
            NN_ACCOUNT_RETURN_FALSE_UNLESS(tokenLength > 0);
            // token = key[0, tokenLength)
            auto tokenEnd = key + tokenLength;

            // 値との区切り文字を探す
            auto keyEnd = detail::FindChar(key, tokenLength, '=');
            // なければエラー
            NN_ACCOUNT_RETURN_FALSE_UNLESS(keyEnd);
            // length(key) = 値との区切り文字までの長さ
            auto keylength = keyEnd - key;
            // length(key) <= 0 ならエラー
            NN_ACCOUNT_RETURN_FALSE_UNLESS(keylength > 0);
            // value = token[keyLength + 1, tokenLength)
            auto value = keyEnd + 1;

            // パーサ実装に pair<key, value> を渡す
            Update(key, keylength, value, tokenEnd - value);

            // 次のトークンがなければ終了
            key = (p ? p + 1 : nullptr);
        }
        return true;
    }
};

#undef NN_ACCOUNT_RETURN_FALSE_UNLESS

}
}
} // ~namespace nn::account::http
