﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/detail/account_Settings.h>

#include <mutex>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

namespace nn { namespace account { namespace detail {

namespace
{
/* --------------------------------------------------------------------------------------------
    開発用
 */
#if !defined(NN_SDK_BUILD_RELEASE)
bool g_IsInitialized = false;
#endif

/* --------------------------------------------------------------------------------------------
    設定の定義 (Processes/account/!.nact, account.settings.fwdbg-*.yml 準拠)
 */
const char SettingName[] = "account";
const char NaRequiredForNetworkServiceName[] = "na_required_for_network_service"; // bool
const char NaLicenseVerificationEnabledName[] = "na_license_verification_enabled"; // bool

#if defined(NN_BUILD_CONFIG_SPEC_NX)
const bool NaRequiredForNetworkServiceDefault = true;
const bool NaLicenseVerificationEnabledDefault = true;

#else
const bool NaRequiredForNetworkServiceDefault = false;
const bool NaLicenseVerificationEnabledDefault = false;
#endif

/* --------------------------------------------------------------------------------------------
    nn::settings アクセッサ
 */
inline bool LoadSettingValue(const char* key) NN_NOEXCEPT
{
    bool value;
    auto s = settings::fwdbg::GetSettingsItemValue(&value, sizeof(value), SettingName, key);
    NN_ABORT_UNLESS(s == sizeof(value));
    return value;
}
} // ~namespace nn::account::detail::<anonymous>

/* --------------------------------------------------------------------------------------------
    メンバ変数
 */
os::SdkMutex FirmwareSettings::s_Lock;

#define NN_ACCOUNT_DEFINE_VALUE(type, key) \
    type FirmwareSettings::g_ ## key = key ## Default

NN_ACCOUNT_DEFINE_VALUE(bool, NaRequiredForNetworkService);
NN_ACCOUNT_DEFINE_VALUE(bool, NaLicenseVerificationEnabled);

#undef NN_ACCOUNT_DEFINE_VALUE

/* --------------------------------------------------------------------------------------------
    設定値の再読み込み
 */
void FirmwareSettings::Refresh() NN_NOEXCEPT
{
    std::lock_guard<decltype(s_Lock)> lock(s_Lock);
#define NN_ACCOUNT_LOAD_SETTING_VALUE(key) \
    g_ ## key = LoadSettingValue(key ## Name)

    NN_ACCOUNT_LOAD_SETTING_VALUE(NaRequiredForNetworkService);
    NN_ACCOUNT_LOAD_SETTING_VALUE(NaLicenseVerificationEnabled);

#undef NN_ACCOUNT_LOAD_SETTING_VALUE

#if !defined(NN_SDK_BUILD_RELEASE)
    g_IsInitialized = true;
#endif
}

/* --------------------------------------------------------------------------------------------
    読み取りメソッド定義
    */
#define NN_ACCOUNT_DEFINE_BOOL_METHOD(key) \
    bool FirmwareSettings::Is ## key () NN_NOEXCEPT \
    { \
        std::lock_guard<decltype(s_Lock)> lock(s_Lock); \
        NN_SDK_ASSERT(g_IsInitialized); \
        return g_ ## key; \
    }

NN_ACCOUNT_DEFINE_BOOL_METHOD(NaRequiredForNetworkService)
NN_ACCOUNT_DEFINE_BOOL_METHOD(NaLicenseVerificationEnabled)

#undef NN_ACCOUNT_DEFINE_BOOL_METHOD

}}} // ~namespace nn::account::detail
