﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/detail/account_OAuthProcedureBase.h>

#include <nn/account/account_Types.h>
#include <nn/account/account_ResultPrivate.h>
#include <nn/account/http/account_Interface.sfdl.h>

#include <cstring>
#include <utility>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include "account_ShimLibraryUtility.h"

namespace nn {
namespace account {
namespace detail {

/* --------------------------------------------------------------------------------------------
    OAuthProcedureBase
*/
OAuthProcedureBase::OAuthProcedureBase() NN_NOEXCEPT
    : m_User(InvalidUid)
    , m_Ptr(nullptr)
{
}
OAuthProcedureBase::OAuthProcedureBase(const Uid& user, http::IOAuthProcedure* ptr) NN_NOEXCEPT
    : m_User(user)
    , m_Ptr(ptr)
{
}
OAuthProcedureBase::OAuthProcedureBase(OAuthProcedureBase&& rhs) NN_NOEXCEPT
    : m_User(rhs.m_User)
    , m_Ptr(rhs.m_Ptr)
{
    rhs.m_User = InvalidUid;
    rhs.m_Ptr = nullptr;
}
OAuthProcedureBase::~OAuthProcedureBase() NN_NOEXCEPT
{
    if (m_Ptr != nullptr)
    {
        sf::ReleaseSharedObject(m_Ptr);
    }
}
OAuthProcedureBase& OAuthProcedureBase::operator=(OAuthProcedureBase&& rhs) NN_NOEXCEPT
{
    OAuthProcedureBase tmp(std::move(rhs));
    this->Swap(tmp);
    return *this;
}
OAuthProcedureBase& OAuthProcedureBase::Swap(OAuthProcedureBase& rhs) NN_NOEXCEPT
{
    std::swap(m_User, rhs.m_User);
    std::swap(m_Ptr, rhs.m_Ptr);
    return *this;
}
http::IOAuthProcedure* OAuthProcedureBase::GetPtr() const NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, OAuthProcedureBase);
    return m_Ptr;
}
Result OAuthProcedureBase::PrepareAsync(AsyncContext* pOut)
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, OAuthProcedureBase);
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    sf::SharedPointer<detail::IAsyncContext> p;
    NN_RESULT_DO(m_Ptr->PrepareAsync(&p));
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    *pOut = AsyncContext(p.Detach());
    NN_RESULT_SUCCESS;
}
Result OAuthProcedureBase::GetRequest(RequestUrl* pOutRequestUrl, CallbackUri* pOutCallbackUri) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, OAuthProcedureBase);
    NN_SDK_REQUIRES_NOT_NULL(pOutRequestUrl);
    NN_SDK_REQUIRES_NOT_NULL(pOutCallbackUri);

    return m_Ptr->GetRequest(pOutRequestUrl, pOutCallbackUri);
}
Result OAuthProcedureBase::ApplyResponse(const char* response, size_t responseBufferSize) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, OAuthProcedureBase);
    NN_SDK_REQUIRES_NOT_NULL(response);
    NN_SDK_REQUIRES_GREATER(responseBufferSize, 0u);

    return m_Ptr->ApplyResponse(sf::InArray<char>(response, responseBufferSize));
}
Result OAuthProcedureBase::ApplyResponseAsync(AsyncContext* pOutContext, const char* response, size_t responseBufferSize) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, OAuthProcedureBase);
    NN_SDK_REQUIRES_NOT_NULL(pOutContext);
    NN_SDK_REQUIRES_NOT_NULL(response);
    NN_SDK_REQUIRES_GREATER(responseBufferSize, 0u);

    sf::SharedPointer<detail::IAsyncContext> p;
    NN_RESULT_DO(m_Ptr->ApplyResponseAsync(&p, sf::InArray<char>(response, responseBufferSize)));
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    *pOutContext = AsyncContext(p.Detach());
    NN_RESULT_SUCCESS;
}
Result OAuthProcedureBase::Suspend(SessionId* pOut) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, OAuthProcedureBase);
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    detail::Uuid sessionId;
    NN_RESULT_DO(m_Ptr->Suspend(&sessionId));
    std::memcpy(pOut, &sessionId, sizeof(*pOut));
    NN_RESULT_SUCCESS;
}


/* --------------------------------------------------------------------------------------------
    SimpleOAuthProcedureBase
*/
SimpleOAuthProcedureBase::SimpleOAuthProcedureBase() NN_NOEXCEPT
{
}
SimpleOAuthProcedureBase::SimpleOAuthProcedureBase(const Uid& user, http::IOAuthProcedure* ptr) NN_NOEXCEPT
    : OAuthProcedureBase(user, reinterpret_cast<http::IOAuthProcedure*>(ptr))
{
}
SimpleOAuthProcedureBase::SimpleOAuthProcedureBase(SimpleOAuthProcedureBase&& rhs) NN_NOEXCEPT
    : OAuthProcedureBase(std::move(rhs))
{
}
SimpleOAuthProcedureBase& SimpleOAuthProcedureBase::operator=(SimpleOAuthProcedureBase&& rhs) NN_NOEXCEPT
{
    SimpleOAuthProcedureBase tmp(std::move(rhs));
    this->Swap(tmp);
    return *this;
}
SimpleOAuthProcedureBase& SimpleOAuthProcedureBase::Swap(SimpleOAuthProcedureBase& rhs) NN_NOEXCEPT
{
    OAuthProcedureBase::Swap(rhs);
    return *this;
}

} // ~namespace nn::account::detail
}
}
