﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "../http/account_HttpUtil.h"
#include <nn/account/json/account_JsonAdaptor.h>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

namespace nn { namespace account { namespace baas {

const int ProblemDetailsErrorCodeLengthMax = 64;

// BaaS API が定めるエラーコードの表現形式
// 参照: http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=112892437
struct BaasProblemDetails
{
    uint16_t status;
    char errorCode[ProblemDetailsErrorCodeLengthMax];

    bool operator==(const BaasProblemDetails& rts) const NN_NOEXCEPT;
    bool operator!=(const BaasProblemDetails& rts) const NN_NOEXCEPT;
};

template <int NodeDepthMax, int PathLengthMax>
class BaasProblemDetailsAdaptor
{
    NN_STATIC_ASSERT(NodeDepthMax >= 4 && PathLengthMax >= 16);

    NN_DISALLOW_COPY(BaasProblemDetailsAdaptor);

public:
    typedef json::JsonPath<NodeDepthMax, PathLengthMax> JsonPathType;

private:
    json::LookupEntry m_StatusEntry;
    json::LookupEntry m_ErrorCodeEntry;
    json::LookupEntry m_TitleEntry; // 表示のみ
    json::LookupEntry m_TypeEntry; // 表示のみ
    json::LookupEntry m_DetailEntry; // 表示のみ

    BaasProblemDetails m_Details;

public:
    explicit BaasProblemDetailsAdaptor() NN_NOEXCEPT;

    Result Adapt(int32_t httpCode) const NN_NOEXCEPT;

    void Update(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT;
    void Update(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT;

    void Update(const JsonPathType&, std::nullptr_t) NN_NOEXCEPT
    { /*NOP*/ }
    void Update(const JsonPathType&, bool) NN_NOEXCEPT
    { /*NOP*/ }
    void Update(const JsonPathType&, uint64_t) NN_NOEXCEPT
    { /*NOP*/ }
    void Update(const JsonPathType&, double) NN_NOEXCEPT
    { /*NOP*/ }
};

// BaaS のエラー処理を行う基底のアダプタ
template <int NodeDepthMax, int PathLengthMax>
class BaasCommonAdaptor
    : public json::ExtensibleJsonAdaptorBase<BaasProblemDetailsAdaptor<NodeDepthMax, PathLengthMax>>
{
};

// BaaS の汎用 DELETE リクエスト用アダプタ
class BaasDeleteAdaptor
    : public BaasCommonAdaptor<4, 16>
{
    NN_DISALLOW_COPY(BaasDeleteAdaptor);
private:
    typedef BaasCommonAdaptor<4, 16> Base;

public:
    typedef Base::JsonPathType JsonPathType;

protected:
    virtual Result AdaptImpl() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_SUCCESS;
    }

public:
    BaasDeleteAdaptor() NN_NOEXCEPT
    {
    }
};

}}} // ~namespace nn::account::baas

/* --------------------------------------------------------------------------------------------
    実装
*/

#include <cstring>
#include <limits>
#include <utility>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>

namespace nn { namespace account { namespace baas {

std::pair<bool, Result> FindResult(int32_t httpCode, const BaasProblemDetails& problemDetails) NN_NOEXCEPT;

/* --------------------------------------------------------------------------------------------
    BaasProblemDetails
*/

inline bool BaasProblemDetails::operator==(const BaasProblemDetails& rhs) const NN_NOEXCEPT
{
    return true
        && status == rhs.status
        && strncmp(errorCode, rhs.errorCode, sizeof(errorCode)) == 0;
}
inline bool BaasProblemDetails::operator!=(const BaasProblemDetails& rhs) const NN_NOEXCEPT
{
    return !(*this == rhs);
}

template <int NodeDepthMax, int PathLengthMax>
inline BaasProblemDetailsAdaptor<NodeDepthMax, PathLengthMax>::BaasProblemDetailsAdaptor() NN_NOEXCEPT
    : m_StatusEntry("$.status")
    , m_ErrorCodeEntry("$.errorCode")
    , m_TitleEntry("$.title")
    , m_TypeEntry("$.type")
    , m_DetailEntry("$.detail")
{
}
template <int NodeDepthMax, int PathLengthMax>
inline Result BaasProblemDetailsAdaptor<NodeDepthMax, PathLengthMax>::Adapt(int32_t httpCode) const NN_NOEXCEPT
{
    if (httpCode / 100 == 2)
    {
        // HttpStatus が 2xx なら ResultSuccess
        NN_RESULT_SUCCESS;
    }

#define NN_ACCOUNT_PRINT_ENTRY_STATE(e) \
    NN_SDK_LOG("    %s : %s\n", (e)? "+": "-", (e).path)

    NN_SDK_LOG("[nn::account] -----------------------------------------------\n");
    NN_SDK_LOG("  Error: BaaS ProblemDetails for HTTP Status %d\n", httpCode);
    NN_ACCOUNT_PRINT_ENTRY_STATE(m_StatusEntry);
    NN_ACCOUNT_PRINT_ENTRY_STATE(m_ErrorCodeEntry);
    NN_ACCOUNT_PRINT_ENTRY_STATE(m_TypeEntry);
    NN_ACCOUNT_PRINT_ENTRY_STATE(m_TitleEntry);
    NN_ACCOUNT_PRINT_ENTRY_STATE(m_DetailEntry);

#undef NN_ACCOUNT_PRINT_ENTRY_STATE

    if (m_StatusEntry && m_ErrorCodeEntry)
    {
        auto r = FindResult(httpCode, m_Details);
        if (r.first)
        {
            NN_RESULT_THROW(r.second);
        }
    }
    NN_RESULT_THROW(http::HandleHttpStatusCode(httpCode));
}
template <int NodeDepthMax, int PathLengthMax>
inline void BaasProblemDetailsAdaptor<NodeDepthMax, PathLengthMax>::Update(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT
{
    if (m_StatusEntry.CanAccept(jsonPath))
    {
        if (!(value >= 0 && value <= std::numeric_limits<uint16_t>::max()))
        {
            // HTTP ステータスコードが大きすぎる
            return;
        }
        m_Details.status = static_cast<uint16_t>(value);
        m_StatusEntry.MarkAccepted();
    }
}
template <int NodeDepthMax, int PathLengthMax>
inline void BaasProblemDetailsAdaptor<NodeDepthMax, PathLengthMax>::Update(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
{
    NN_SDK_ASSERT(valueLength >= 0);
    if (m_ErrorCodeEntry.CanAccept(jsonPath))
    {
        if (!(static_cast<uint32_t>(valueLength) <= sizeof(m_Details.errorCode)))
        {
            // エラーコードが長すぎる
            return;
        }
        std::strncpy(m_Details.errorCode, value, sizeof(m_Details.errorCode));
        m_ErrorCodeEntry.MarkAccepted();
    }
    else if (m_TitleEntry.CanAccept(jsonPath))
    {
        NN_SDK_LOG("[nn::account] BaaS BaasProblemDetails::Title  : %s\n", value);
        m_TitleEntry.MarkAccepted();
    }
    else if (m_TypeEntry.CanAccept(jsonPath))
    {
        NN_SDK_LOG("[nn::account] BaaS BaasProblemDetails::Type   : %s\n", value);
        m_TypeEntry.MarkAccepted();
    }
    else if (m_DetailEntry.CanAccept(jsonPath))
    {
        NN_SDK_LOG("[nn::account] BaaS BaasProblemDetails::Detail : %s\n", value);
        m_DetailEntry.MarkAccepted();
    }
}

}}} // ~namespace nn::account::baas
