﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/baas/account_BaasLoginDriver.h>
#include <nn/account/baas/account_ResultForBaas.h>

#include "account_BaasAuthorizationHeader.h"
#include "account_BaasLoginAdaptor.h"
#include "account_BaasResourceResolver.h"
#include "../detail/account_ByteUtil.h"
#include "../detail/account_CacheUtil.h"
#include <nn/account/baas/account_BaasTypes.h>
#include <nn/account/detail/account_Settings.h>
#include <nn/account/http/account_CurlInputStream.h>
#include <nn/account/http/account_ResultForHttp.h>
#include <nn/account/json/account_RapidJsonApi.h>

#include <cstddef>
#include <type_traits>

#include <nn/nn_StaticAssert.h>
#include <nn/util/util_ScopeExit.h>

namespace nn { namespace account { namespace baas {
namespace
{
static const char Op2VerificationSkipOption[] = "&skipOp2Verification=1";

bool IsOp2VerificationRequired() NN_NOEXCEPT
{
    // 以下のそれぞれを満たす場合、 OP2 の資格情報チェックをする必要がある。
    // - ネットワークサービス利用に NA が必要
    // - NA の資格情報検査が有効
    return true
        && detail::FirmwareSettings::IsNaRequiredForNetworkService()
        && detail::FirmwareSettings::IsNaLicenseVerificationEnabled();
}

nn::Result SetupStreamForClientLogin(
    http::CurlInputStream& stream,
    char* postDataBuffer, size_t postDataBufferSize,
    const ndas::NdasOperator& ndasOperator) NN_NOEXCEPT
{
    // POST データの作成
    static const char PostDataFormat[] = "grantType=public_client&assertion=";
    NN_SDK_ASSERT(postDataBufferSize > sizeof(PostDataFormat) + detail::NdasDevAuthTokenSizeMax);
    std::strncpy(postDataBuffer, PostDataFormat, sizeof(PostDataFormat));
    auto l = sizeof(PostDataFormat) - 1;

    auto tokenBuffer = postDataBuffer + l;
    auto tokenBufferSize = postDataBufferSize - l;
    NN_RESULT_THROW_UNLESS(tokenBufferSize > detail::NdasDevAuthTokenSizeMax, ResultInsufficientBuffer());

    // デバイス認証トークンの取得
    size_t devAuthSizeActual;
    NN_RESULT_DO(ndasOperator.LoadServiceAuthenticationTokenCacheForApplications(&devAuthSizeActual, tokenBuffer, tokenBufferSize));
    NN_SDK_ASSERT(devAuthSizeActual <= detail::NdasDevAuthTokenSizeMax);
    tokenBuffer[devAuthSizeActual] = '\0';
    NN_SDK_ASSERT(strnlen(postDataBuffer, postDataBufferSize) == l + devAuthSizeActual);

    // リクエスト生成
    NN_RESULT_DO(stream.SetHttpPost(postDataBuffer, false));

    // URL の作成
    char url[256];
    l = BaasResourceResolver::GetUrlForClientLogin(url, sizeof(url));
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);
    NN_RESULT_DO(stream.SetUrl(url));

    NN_RESULT_SUCCESS;
}
nn::Result SetupStreamForUserLogin(
    http::CurlInputStream& stream,
    struct curl_slist** headers,
    char* postDataBuffer, size_t postDataBufferSize,
    const ClientAccessTokenCache& clientAccessTokenCache,
    const BaasCredential& credential,
    char* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(workBufferSize >= RequiredBufferSizeForAuthorizationHeader);

    // POST データ作成
    static const char PostDataFormat[] = "id=%016llx&password=%.*s";
    auto l = nn::util::SNPrintf(postDataBuffer, postDataBufferSize, PostDataFormat, credential.loginId, BaasLoginPasswordLength, credential.loginPassword);
    NN_SDK_ASSERT(static_cast<uint32_t>(l)  < postDataBufferSize);
    NN_UNUSED(l);

    // Authorization ヘッダの準備
    NN_RESULT_DO(CreateAuthorizationHeaderWithClientAccessToken(workBuffer, workBufferSize, clientAccessTokenCache));
    *headers = curl_slist_append(*headers, workBuffer);
    NN_RESULT_THROW_UNLESS(*headers != nullptr, http::ResultCurlSlistAppendFailure());

    // リクエスト生成
    NN_RESULT_DO(stream.SetHeaders(*headers));
    NN_RESULT_DO(stream.SetHttpPost(postDataBuffer, false));

    // URL の作成
    char url[256];
    l = BaasResourceResolver::GetUrlForUserLogin(url, sizeof(url));
    NN_SDK_ASSERT(static_cast<uint32_t>(l)  < sizeof(url));
    NN_UNUSED(l);
    NN_RESULT_DO(stream.SetUrl(url));

    NN_RESULT_SUCCESS;
}

nn::Result SetupStreamForUserLoginWithAppInfo(
    http::CurlInputStream& stream,
    struct curl_slist** headers,
    char* postDataBuffer, size_t postDataBufferSize,
    const ClientAccessTokenCache& clientAccessTokenCache,
    const BaasCredential& credential,
    const ndas::NdasOperator& ndasOperator,
    const detail::ApplicationInfo& appInfo,
    char* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(workBufferSize >= detail::NdasAppAuthTokenSizeMax);
    NN_SDK_REQUIRES(workBufferSize >= RequiredBufferSizeForAuthorizationHeader);

    // アプリケーション認証トークンの取得
    size_t appAuthSizeActual;
    NN_RESULT_DO(ndasOperator.LoadApplicationAuthenticationTokenCache(&appAuthSizeActual, workBuffer, workBufferSize, appInfo));
    NN_SDK_ASSERT(appAuthSizeActual <= detail::NdasAppAuthTokenSizeMax);

    // POST データ作成
    static const char PostDataFormat[] = "id=%016llx&password=%.*s&appAuthNToken=%.*s";
    auto l = nn::util::SNPrintf(
        postDataBuffer, postDataBufferSize,
        PostDataFormat,
        credential.loginId, BaasLoginPasswordLength, credential.loginPassword, appAuthSizeActual, workBuffer);
    NN_SDK_ASSERT(static_cast<uint32_t>(l)  < postDataBufferSize);

    // OP2 検証スキップ
    if (!IsOp2VerificationRequired())
    {
        NN_ABORT_UNLESS(postDataBufferSize - sizeof(Op2VerificationSkipOption) >= static_cast<uint32_t>(l));
        std::strncpy(postDataBuffer + l, Op2VerificationSkipOption, sizeof(Op2VerificationSkipOption));
    }

    // Authorization ヘッダの準備
    char* authHeader = reinterpret_cast<char*>(workBuffer);
    NN_RESULT_DO(CreateAuthorizationHeaderWithClientAccessToken(authHeader, workBufferSize, clientAccessTokenCache));
    *headers = curl_slist_append(*headers, authHeader);
    NN_RESULT_THROW_UNLESS(*headers != nullptr, http::ResultCurlSlistAppendFailure());

    // リクエスト生成
    NN_RESULT_DO(stream.SetHeaders(*headers));
    NN_RESULT_DO(stream.SetHttpPost(postDataBuffer, false));

    // URL の作成
    char url[256];
    l = BaasResourceResolver::GetUrlForUserLogin(url, sizeof(url));
    NN_SDK_ASSERT(static_cast<uint32_t>(l)  < sizeof(url));
    NN_UNUSED(l);
    NN_RESULT_DO(stream.SetUrl(url));

    NN_RESULT_SUCCESS;
}

nn::Result SetupStreamForFederationLogin(
    http::CurlInputStream& stream,
    struct curl_slist** headers,
    char* postDataBuffer, size_t postDataBufferSize,
    const ClientAccessTokenCache& clientAccessTokenCache,
    const BaasCredential& credential,
    const detail::Uuid& naIdTokenCacheId,
    const detail::AbstractLocalStorage& storage,
    char* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(workBufferSize >= RequiredBufferSizeForAuthorizationHeader);

    // POST データ作成
    static const char PostDataFormat[] = "id=%016llx&password=%.*s&idp=nintendoAccount&idToken=";
    auto l = nn::util::SNPrintf(
        postDataBuffer, postDataBufferSize,
        PostDataFormat, credential.loginId, BaasLoginPasswordLength, credential.loginPassword);
    NN_SDK_ASSERT(static_cast<uint32_t>(l)  < postDataBufferSize);
    NN_UNUSED(l);

    // NintendoAccount ID トークンの取得
    auto idTokenBuffer = postDataBuffer + l;
    auto idTokenBufferSize = postDataBufferSize - l;
    NN_SDK_ASSERT(idTokenBufferSize > detail::NasIdTokenSizeMax);
    size_t sizeActual;
    NN_RESULT_DO((detail::CacheUtil::LoadCacheFile<detail::NasIdTokenSizeMax, ResultNasTokenLengthUnacceptable>(
        &sizeActual, idTokenBuffer, idTokenBufferSize, naIdTokenCacheId, storage)));
    NN_SDK_ASSERT(sizeActual <= detail::NasIdTokenSizeMax);
    NN_SDK_ASSERT(idTokenBufferSize - sizeActual > 0);
    idTokenBuffer[sizeActual] = '\0';

    // Authorization ヘッダの準備
    NN_RESULT_DO(CreateAuthorizationHeaderWithClientAccessToken(workBuffer, workBufferSize, clientAccessTokenCache));
    *headers = curl_slist_append(*headers, workBuffer);
    NN_RESULT_THROW_UNLESS(*headers != nullptr, http::ResultCurlSlistAppendFailure());

    // リクエスト生成
    NN_RESULT_DO(stream.SetHeaders(*headers));
    NN_RESULT_DO(stream.SetHttpPost(postDataBuffer, false));

    // URL の作成
    char url[256];
    l = BaasResourceResolver::GetUrlForFederationLogin(url, sizeof(url));
    NN_SDK_ASSERT(static_cast<uint32_t>(l)  < sizeof(url));
    NN_UNUSED(l);
    NN_RESULT_DO(stream.SetUrl(url));

    NN_RESULT_SUCCESS;
}

nn::Result SetupStreamForFederationLoginForGuest(
    http::CurlInputStream& stream,
    struct curl_slist** headers,
    char* postDataBuffer, size_t postDataBufferSize,
    const ClientAccessTokenCache& clientAccessTokenCache,
    const char (&nasIdToken)[detail::NasIdTokenSizeMax],
    const ndas::NdasOperator& ndasOperator,
    const detail::ApplicationInfo& appInfo,
    char* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(workBufferSize >= detail::NdasAppAuthTokenSizeMax);
    NN_SDK_REQUIRES(workBufferSize >= RequiredBufferSizeForAuthorizationHeader);

    // アプリケーション認証トークンの取得
    size_t appAuthSizeActual;
    NN_RESULT_DO(ndasOperator.LoadApplicationAuthenticationTokenCache(&appAuthSizeActual, workBuffer, workBufferSize, appInfo));
    NN_SDK_ASSERT(appAuthSizeActual <= detail::NdasAppAuthTokenSizeMax);

    // POST データ作成
    static const char PostDataFormat[] = "idp=nintendoAccount&idToken=%.*s&appAuthNToken=%.*s";
    auto l = nn::util::SNPrintf(
        postDataBuffer, postDataBufferSize,
        PostDataFormat,
        sizeof(nasIdToken), nasIdToken,
        appAuthSizeActual, workBuffer);
    NN_SDK_ASSERT(static_cast<uint32_t>(l)  < postDataBufferSize);

    // OP2 検証スキップ
    if (!IsOp2VerificationRequired())
    {
        NN_ABORT_UNLESS(postDataBufferSize - sizeof(Op2VerificationSkipOption) >= static_cast<uint32_t>(l));
        std::strncpy(postDataBuffer + l, Op2VerificationSkipOption, sizeof(Op2VerificationSkipOption));
    }

    // Authorization ヘッダの準備
    NN_RESULT_DO(CreateAuthorizationHeaderWithClientAccessToken(workBuffer, workBufferSize, clientAccessTokenCache));
    *headers = curl_slist_append(*headers, workBuffer);
    NN_RESULT_THROW_UNLESS(*headers != nullptr, http::ResultCurlSlistAppendFailure());

    // リクエスト生成
    NN_RESULT_DO(stream.SetHeaders(*headers));
    NN_RESULT_DO(stream.SetHttpPost(postDataBuffer, false));

    // URL の作成
    char url[256];
    l = BaasResourceResolver::GetUrlForFederationLogin(url, sizeof(url));
    NN_SDK_ASSERT(static_cast<uint32_t>(l)  < sizeof(url));
    NN_UNUSED(l);
    NN_RESULT_DO(stream.SetUrl(url));

    NN_RESULT_SUCCESS;
}

} // ~namespace nn::account::baas::<anonymous>

const size_t BaasLoginDriver::RequiredBufferSize = 8 * 1024u;

BaasLoginDriver::BaasLoginDriver(
    ClientAccessTokenCache& clientAccessTokenCache,
    UserAccessTokenCache& userAccessTokenCache,
    UserIdTokenCache& userIdTokenCache,
    const ndas::NdasOperator& ndasOperator,
    const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
    : m_ClientAccessTokenCache(clientAccessTokenCache)
    , m_UserAccessTokenCache(userAccessTokenCache)
    , m_UserIdTokenCache(userIdTokenCache)
    , m_NdasOperator(ndasOperator)
    , m_Storage(storage)
{
}

Result BaasLoginDriver::EnsureClientAccessTokenCache(
    CURL* curlHandle, void* rawBuffer, size_t bufferSize,
    const detail::Cancellable* pCancellable) const NN_NOEXCEPT
{
    // バッファ定義
    struct Buffer
    {
        char postDataBuffer[detail::NdasDevAuthTokenSizeMax + 1024];
        union
        {
            struct
            {
                char stringBuffer[detail::BaasAccessTokenSizeMax + 8];
                char inputBuffer[detail::IoBufferSizeMin];
            } io;
        } u;
    };
    NN_STATIC_ASSERT(sizeof(Buffer) <= RequiredBufferSize);
    NN_STATIC_ASSERT(std::alignment_of<Buffer>::value <= std::alignment_of<std::max_align_t>::value);

    // 事前条件
    NN_SDK_REQUIRES(curlHandle != nullptr);
    NN_SDK_REQUIRES(rawBuffer != nullptr);
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(rawBuffer) % std::alignment_of<std::max_align_t>::value == 0);
    NN_SDK_REQUIRES(bufferSize >= sizeof(Buffer));

    // 既に有効な認証トークンがあるなら、即時返る。
    NN_RESULT_THROW_UNLESS(!m_ClientAccessTokenCache.IsAvailable(), ResultSuccess());

    // リクエストの作成
    auto* buffer = reinterpret_cast<Buffer*>(rawBuffer);

    http::CurlInputStream stream(curlHandle, pCancellable);
    NN_RESULT_DO(stream.Initialize());
    stream.SetStringBuffer(buffer->u.io.stringBuffer, sizeof(buffer->u.io.stringBuffer));
    stream.SetInputBuffer(buffer->u.io.inputBuffer, bufferSize - offsetof(Buffer, u.io.inputBuffer));

    NN_RESULT_DO(SetupStreamForClientLogin(
        stream, buffer->postDataBuffer, sizeof(buffer->postDataBuffer), m_NdasOperator));

    // 通信の実行と結果の適用
    ClientLoginAdaptor adaptor(m_ClientAccessTokenCache);
    NN_RESULT_DO(stream.Open());
    NN_RESULT_DO(json::ImportJsonByRapidJson(adaptor, stream, pCancellable));
    stream.Close();
    return adaptor.Adapt(stream.GetHttpCode());
}
Result BaasLoginDriver::IsUserAccessTokenCacheAvailable(bool* pOut, const NetworkServiceAccountId& id) const NN_NOEXCEPT
{
    *pOut = m_UserAccessTokenCache.IsAvailable(id);
    NN_RESULT_SUCCESS;
}
Result BaasLoginDriver::LoadUserAccessTokenCache(size_t* pOutSizeActual, char* buffer, size_t bufferSize, const NetworkServiceAccountId& id) const NN_NOEXCEPT
{
    return m_UserAccessTokenCache.Load(pOutSizeActual, buffer, bufferSize, id);
}

Result BaasLoginDriver::ExecuteUserLogin(
    const NetworkServiceAccountId& id, const util::optional<NintendoAccountId>& pLinkedNa,
    UserLoginAdaptor& adaptor, http::CurlInputStream& stream, const detail::Cancellable* pCancellable) const NN_NOEXCEPT
{
    NN_RESULT_DO(stream.Open());
    NN_RESULT_DO(json::ImportJsonByRapidJson(adaptor, stream, pCancellable));
    stream.Close();

    // DA が削除されたり BAN されているエラー
    NN_RESULT_DO(adaptor.Adapt(stream.GetHttpCode()));

    // DA の所属がどこかで変更されている
    NN_RESULT_THROW_UNLESS(adaptor.GetUserId() == id, ResultBaasUserMismatch());

    NintendoAccountId serverLinkedNa;
    if (adaptor.TryGetNintendoAccountId(&serverLinkedNa))
    {
        // ローカル: 未リンク, リモート: リンク
        NN_RESULT_THROW_UNLESS(pLinkedNa, ResultBaasUserLinkageStateMismatch());
        // NA との連携関係がサーバー上で変更されている
        NN_RESULT_THROW_UNLESS(serverLinkedNa == *pLinkedNa, ResultBaasUserLinkedAccountMismatch());
    }
    else
    {
        // ローカル: リンク, リモート: 未リンク
        NN_RESULT_THROW_UNLESS(!pLinkedNa, ResultBaasUserLinkageStateMismatch());;
    }
    NN_RESULT_SUCCESS;
}

Result BaasLoginDriver::EnsureUserAccessTokenCache(
    const NetworkServiceAccountId& id, const BaasCredential& credential,
    const util::optional<NintendoAccountId>& pLinkedNa,
    CURL* curlHandle, void* rawBuffer, size_t bufferSize,
    const detail::Cancellable* pCancellable) const NN_NOEXCEPT
{
    // バッファ定義
    struct Buffer
    {
        char postDataBuffer[128];
        union
        {
            char workBuffer[RequiredBufferSizeForAuthorizationHeader];
            struct
            {
                char stringBuffer[detail::BaasIdTokenSizeMax + 8];
                char inputBuffer[detail::IoBufferSizeMin];
            } io;
        } u;
    };
    NN_STATIC_ASSERT(sizeof(Buffer) <= RequiredBufferSize);
    NN_STATIC_ASSERT(std::alignment_of<Buffer>::value <= std::alignment_of<std::max_align_t>::value);

    // 事前条件
    NN_SDK_REQUIRES(curlHandle != nullptr);
    NN_SDK_REQUIRES(rawBuffer != nullptr);
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(rawBuffer) % std::alignment_of<std::max_align_t>::value == 0);
    NN_SDK_REQUIRES(bufferSize >= sizeof(Buffer));

    // 既に有効な認証トークンがあるなら、即時返る。
    NN_RESULT_THROW_UNLESS(!m_UserAccessTokenCache.IsAvailable(id), ResultSuccess());

    // リクエストの作成
    auto* buffer = reinterpret_cast<Buffer*>(rawBuffer);

    struct curl_slist *headers = nullptr;
    NN_UTIL_SCOPE_EXIT
    {
        curl_slist_free_all(headers);
    };

    http::CurlInputStream stream(curlHandle, pCancellable);
    NN_RESULT_DO(stream.Initialize());
    stream.SetStringBuffer(buffer->u.io.stringBuffer, sizeof(buffer->u.io.stringBuffer));
    stream.SetInputBuffer(buffer->u.io.inputBuffer, bufferSize - offsetof(Buffer, u.io.inputBuffer));

    NN_RESULT_DO(SetupStreamForUserLogin(
        stream, &headers, buffer->postDataBuffer, sizeof(buffer->postDataBuffer), m_ClientAccessTokenCache, credential,
        buffer->u.workBuffer, sizeof(buffer->u.workBuffer)));

    // 通信の実行と結果の適用
    UserLoginAdaptor adaptor(m_UserAccessTokenCache, m_UserIdTokenCache, detail::InvalidApplicationInfo);
    NN_RESULT_DO(ExecuteUserLogin(id, pLinkedNa, adaptor, stream, pCancellable));

    NN_RESULT_SUCCESS;
}

Result BaasLoginDriver::IsUserIdTokenCacheAvailable(bool* pOut, const NetworkServiceAccountId& id, const detail::ApplicationInfo& appInfo) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(appInfo);
    *pOut = m_UserIdTokenCache.IsAvailable(std::pair<NetworkServiceAccountId, detail::ApplicationInfo>(id, appInfo));
    NN_RESULT_SUCCESS;
}
Result BaasLoginDriver::LoadUserIdTokenCache(size_t* pOutSizeActual, char* buffer, size_t bufferSize, const NetworkServiceAccountId& id, const detail::ApplicationInfo& appInfo) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(appInfo);
    return m_UserIdTokenCache.Load(pOutSizeActual, buffer, bufferSize, id, appInfo);
}
Result BaasLoginDriver::EnsureUserIdTokenCache(
    const NetworkServiceAccountId& id, const BaasCredential& credential,
    const util::optional<NintendoAccountId>& pLinkedNa,
    const detail::ApplicationInfo& appInfo,
    CURL* curlHandle, void* rawBuffer, size_t bufferSize,
    const detail::Cancellable* pCancellable) const NN_NOEXCEPT
{
    // バッファ定義
    struct Buffer
    {
        char postDataBuffer[detail::NdasAppAuthTokenSizeMax + 1024];
        union
        {
            char workBuffer[NN_ACCOUNT_MAX(RequiredBufferSizeForAuthorizationHeader, detail::NdasAppAuthTokenSizeMax + 1)];
            struct
            {
                char stringBuffer[detail::BaasIdTokenSizeMax + 8];
                char inputBuffer[detail::IoBufferSizeMin];
            } io;
        } u;
    };
    NN_STATIC_ASSERT(sizeof(Buffer) <= RequiredBufferSize);
    NN_STATIC_ASSERT(std::alignment_of<Buffer>::value <= std::alignment_of<std::max_align_t>::value);

    // 事前条件
    NN_SDK_REQUIRES(appInfo);
    NN_SDK_REQUIRES(curlHandle != nullptr);
    NN_SDK_REQUIRES(rawBuffer != nullptr);
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(rawBuffer) % std::alignment_of<std::max_align_t>::value == 0);
    NN_SDK_REQUIRES(bufferSize >= sizeof(Buffer));

    // 既に有効な認証トークンがあるなら、即時返る。
    NN_RESULT_THROW_UNLESS(!m_UserIdTokenCache.IsAvailable(
        std::pair<NetworkServiceAccountId, detail::ApplicationInfo>(id, appInfo)), ResultSuccess());

    // リクエストの作成
    auto* buffer = reinterpret_cast<Buffer*>(rawBuffer);

    struct curl_slist *headers = nullptr;
    NN_UTIL_SCOPE_EXIT
    {
        curl_slist_free_all(headers);
    };

    http::CurlInputStream stream(curlHandle, pCancellable);
    NN_RESULT_DO(stream.Initialize());
    stream.SetStringBuffer(buffer->u.io.stringBuffer, sizeof(buffer->u.io.stringBuffer));
    stream.SetInputBuffer(buffer->u.io.inputBuffer, bufferSize - offsetof(Buffer, u.io.inputBuffer));

    NN_RESULT_DO(SetupStreamForUserLoginWithAppInfo(
        stream, &headers, buffer->postDataBuffer, sizeof(buffer->postDataBuffer),
        m_ClientAccessTokenCache, credential, m_NdasOperator, appInfo,
        buffer->u.workBuffer, sizeof(buffer->u.workBuffer)));

    // 通信の実行と結果の適用
    UserLoginAdaptor adaptor(m_UserAccessTokenCache, m_UserIdTokenCache, appInfo);
    NN_RESULT_DO(ExecuteUserLogin(id, pLinkedNa, adaptor, stream, pCancellable));

    NN_RESULT_SUCCESS;
}

/* --------------------------------------------------------
    Federation Login
*/
Result BaasLoginDriver::ExecuteFederationLoginWithNintendoAccount(
    NetworkServiceAccountId* pOutLoggedInAs, UserProfile* pOutProfile,
    const BaasCredential& credential, const NintendoAccountId& naId, const detail::Uuid& naIdTokenCacheId,
    CURL* curlHandle, void* rawBuffer, size_t bufferSize,
    const detail::Cancellable* pCancellable) const NN_NOEXCEPT
{
    // バッファ定義
    struct Buffer
    {
        char postDataBuffer[detail::NasIdTokenSizeMax + 128];
        union
        {
            char workBuffer[RequiredBufferSizeForAuthorizationHeader];
            struct
            {
                char stringBuffer[detail::BaasIdTokenSizeMax + 8];
                char inputBuffer[detail::IoBufferSizeMin];
            } io;
        } u;
    };
    NN_STATIC_ASSERT(sizeof(Buffer) <= RequiredBufferSize);
    NN_STATIC_ASSERT(std::alignment_of<Buffer>::value <= std::alignment_of<std::max_align_t>::value);

    // 事前条件
    NN_SDK_REQUIRES_NOT_NULL(pOutLoggedInAs);
    NN_SDK_REQUIRES(naIdTokenCacheId);
    NN_SDK_REQUIRES(curlHandle != nullptr);
    NN_SDK_REQUIRES(rawBuffer != nullptr);
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(rawBuffer) % std::alignment_of<std::max_align_t>::value == 0);
    NN_SDK_REQUIRES(bufferSize >= sizeof(Buffer));

    // リクエストの作成
    auto* buffer = reinterpret_cast<Buffer*>(rawBuffer);

    struct curl_slist *headers = nullptr;
    NN_UTIL_SCOPE_EXIT
    {
        curl_slist_free_all(headers);
    };

    http::CurlInputStream stream(curlHandle, pCancellable);
    NN_RESULT_DO(stream.Initialize());
    stream.SetStringBuffer(buffer->u.io.stringBuffer, sizeof(buffer->u.io.stringBuffer));
    stream.SetInputBuffer(buffer->u.io.inputBuffer, bufferSize - offsetof(Buffer, u.io.inputBuffer));

    NN_RESULT_DO(SetupStreamForFederationLogin(
        stream, &headers, buffer->postDataBuffer, sizeof(buffer->postDataBuffer), m_ClientAccessTokenCache, credential,
        naIdTokenCacheId, m_Storage, buffer->u.workBuffer, sizeof(buffer->u.workBuffer)));

    // 通信の実行と結果の適用
    UserFederationLoginAdaptor adaptor(m_UserAccessTokenCache, m_Storage);
    NN_RESULT_DO(stream.Open());
    NN_RESULT_DO(json::ImportJsonByRapidJson(adaptor, stream, pCancellable));
    stream.Close();
    NN_RESULT_DO(adaptor.Adapt(stream.GetHttpCode()));

    // 意図しない NA で login された (通常ない)
    NintendoAccountId loggedInNaId;
    NN_RESULT_THROW_UNLESS(adaptor.TryGetNintendoAccountId(&loggedInNaId), ResultBaasDataBroken());
    NN_RESULT_THROW_UNLESS(naId == loggedInNaId, ResultBaasDataBroken());

    // 成功時は DA の所属が変更される場合がある。
    NN_RESULT_DO(adaptor.LoadUserProfile(pOutProfile));
    *pOutLoggedInAs = adaptor.GetUserId();
    NN_RESULT_SUCCESS;
}

/* --------------------------------------------------------
    Federation Login for Guest
*/
Result BaasLoginDriver::ExecuteFederationLoginWithNintendoAccountForGuest(
    GuestUserProfile* pOutProfile,
    const char (&nasIdToken)[detail::NasIdTokenSizeMax], const detail::ApplicationInfo& appInfo,
    CURL* curlHandle, void* rawBuffer, size_t bufferSize,
    const detail::Cancellable* pCancellable) const NN_NOEXCEPT
{
    // バッファ定義
    struct Buffer
    {
        char postDataBuffer[detail::NasIdTokenSizeMax + detail::NdasAppAuthTokenSizeMax + 128];
        union
        {
            char workBuffer[NN_ACCOUNT_MAX(RequiredBufferSizeForAuthorizationHeader, detail::NdasAppAuthTokenSizeMax + 1)];
            struct
            {
                char stringBuffer[detail::BaasIdTokenSizeMax + 8];
                char inputBuffer[detail::IoBufferSizeMin];
            } io;
        } u;
    };
    NN_STATIC_ASSERT(sizeof(Buffer) <= RequiredBufferSize);
    NN_STATIC_ASSERT(std::alignment_of<Buffer>::value <= std::alignment_of<std::max_align_t>::value);

    // 事前条件
    NN_SDK_REQUIRES_NOT_NULL(pOutProfile);
    NN_SDK_REQUIRES(appInfo);
    NN_SDK_REQUIRES(curlHandle != nullptr);
    NN_SDK_REQUIRES(rawBuffer != nullptr);
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(rawBuffer) % std::alignment_of<std::max_align_t>::value == 0);
    NN_SDK_REQUIRES(bufferSize >= sizeof(Buffer));

    // リクエストの作成
    auto* buffer = reinterpret_cast<Buffer*>(rawBuffer);

    struct curl_slist *headers = nullptr;
    NN_UTIL_SCOPE_EXIT
    {
        curl_slist_free_all(headers);
    };

    http::CurlInputStream stream(curlHandle, pCancellable);
    NN_RESULT_DO(stream.Initialize());
    stream.SetStringBuffer(buffer->u.io.stringBuffer, sizeof(buffer->u.io.stringBuffer));
    stream.SetInputBuffer(buffer->u.io.inputBuffer, bufferSize - offsetof(Buffer, u.io.inputBuffer));

    NN_RESULT_DO(SetupStreamForFederationLoginForGuest(
        stream, &headers, buffer->postDataBuffer, sizeof(buffer->postDataBuffer), m_ClientAccessTokenCache, nasIdToken,
        m_NdasOperator, appInfo, buffer->u.workBuffer, sizeof(buffer->u.workBuffer)));

    // 通信の実行と結果の適用
    UserLoginAdaptorForGuest adaptor(pOutProfile, m_UserIdTokenCache, appInfo);
    NN_RESULT_DO(stream.Open());
    NN_RESULT_DO(json::ImportJsonByRapidJson(adaptor, stream, pCancellable));
    stream.Close();
    NN_RESULT_DO(adaptor.Adapt(stream.GetHttpCode()));
    NN_RESULT_SUCCESS;
}

}}} // ~namespace nn::account::baas
