﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/baas/account_BaasLoginCache.h>

#include "../detail/account_CacheUtil.h"

#include <mutex>

#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace account { namespace baas {

bool BaasLoginCacheOperator::IsExpired(int64_t expiration) NN_NOEXCEPT
{
    return detail::CacheUtil::IsExpiredInUptime<detail::Minute<20>::sec>(expiration);
}
void BaasLoginCacheOperator::DeleteCacheData(const detail::Uuid& data, const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
{
    detail::CacheUtil::DeleteCacheFile(data, storage);
}

/* --------------------------------------------------------------------------------------------
    ClientAccessTokenCache
 */
Result ClientAccessTokenCache::Load(size_t* pOutActualSize, char* outToken, size_t bufferSize) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(pOutActualSize);
    NN_SDK_ASSERT(outToken);
    NN_SDK_ASSERT(bufferSize >= detail::BaasAccessTokenSizeMax);
    std::lock_guard<decltype(*this)> lock(*this);

    detail::Uuid data;
    NN_RESULT_DO(GetUnsafe(&data));
    return detail::CacheUtil::LoadCacheFile<detail::BaasAccessTokenSizeMax, ResultBaasTokenLengthUnacceptable>(
        pOutActualSize, outToken, bufferSize, data, GetStorageRef());
}
/* --------------------------------------------------------------------------------------------
    UserAccessTokenCache
 */
Result UserAccessTokenCache::Load(size_t* pOutActualSize, char* outToken, size_t bufferSize, const NetworkServiceAccountId& user) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(pOutActualSize);
    NN_SDK_ASSERT(outToken);
    NN_SDK_ASSERT(bufferSize >= detail::BaasAccessTokenSizeMax);
    std::lock_guard<decltype(*this)> lock(*this);

    detail::Uuid data;
    NN_RESULT_DO(FindUnsafe(&data, user));
    return detail::CacheUtil::LoadCacheFile<detail::BaasAccessTokenSizeMax, ResultBaasTokenLengthUnacceptable>(
        pOutActualSize, outToken, bufferSize, data, GetStorageRef());
}
/* --------------------------------------------------------------------------------------------
    UserIdTokenCache
 */
Result UserIdTokenCache::Load(size_t* pOutActualSize, char* outToken, size_t bufferSize, const NetworkServiceAccountId& user, const detail::ApplicationInfo& appInfo) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(pOutActualSize);
    NN_SDK_ASSERT(outToken);
    NN_SDK_ASSERT(bufferSize >= detail::BaasIdTokenSizeMax);
    std::lock_guard<decltype(*this)> lock(*this);

    detail::Uuid data;
    NN_RESULT_DO(FindUnsafe(&data, TagType(user, appInfo)));
    return detail::CacheUtil::LoadCacheFile<detail::BaasIdTokenSizeMax, ResultBaasTokenLengthUnacceptable>(
        pOutActualSize, outToken, bufferSize, data, GetStorageRef());
}

}}} // ~namespace nn::account::baas
