﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/baas/account_BaasChannelDriver.h>
#include "account_BaasAuthorizationHeader.h"
#include "account_BaasChannelAdaptor.h"
#include "account_BaasResourceResolver.h"

#include <type_traits>
#include <cstring>

#include <nn/account/detail/account_InternalConfig.h>
#include <nn/account/http/account_CurlInputStream.h>
#include <nn/account/http/account_ResultForHttp.h>
#include <nn/account/json/account_RapidJsonApi.h>
#include <nn/account/json/account_RapidJsonStream.h>

#include <nn/util/util_ScopeExit.h>

namespace nn {
namespace account {
namespace baas {

namespace
{
const size_t StringBufferSizeMin = 128u;

Result CreateChannelPutPostData(
    char* postDataBuffer, size_t postDataBufferSize,
    const npns::NotificationToken& nt) NN_NOEXCEPT
{
    char ntData[sizeof(nt.data) + 1];
    std::strncpy(ntData, nt.data, sizeof(ntData));
    ntData[strnlen(nt.data, sizeof(nt.data))] = '\0';

    char stackBuffer[256];
    json::FixedSizeOneTimeAllocatorForRapidJson allocator(stackBuffer, sizeof(stackBuffer));
    json::StringOutputBufferForRapidJson stringBuffer(postDataBuffer, postDataBufferSize);
    json::DefaultRapidJsonMemoryWriter writer(stringBuffer, &allocator, 16);

    NN_RESULT_THROW_UNLESS(writer.StartObject(), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.Key("service"), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.String("npns"), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.Key("deviceToken"), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.String(ntData), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.Key("deviceAttributes"), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.StartObject(), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.EndObject(), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.EndObject(), json::ResultJsonConstructionFailure());

    size_t jsonLength;
    NN_RESULT_DO(stringBuffer.Finalize(&jsonLength));
    NN_SDK_ASSERT(jsonLength < postDataBufferSize);
    NN_UNUSED(jsonLength);

    NN_RESULT_SUCCESS;
}
Result SetupStreamForChannelPut(
    http::CurlInputStream& stream,
    curl_slist** headers, char* postDataBuffer, size_t postDataBufferSize,
    const UserAccessTokenCache& userAccessTokenCache,
    const NetworkServiceAccountId& userId, uint64_t daId, const npns::NotificationToken& nt,
    char* workBuffer, size_t workBufferSize)
{
    NN_RESULT_DO(CreateChannelPutPostData(postDataBuffer, postDataBufferSize, nt));

    // Authorization ヘッダの準備
    char* authHeader = reinterpret_cast<char*>(workBuffer);
    NN_RESULT_DO(CreateAuthorizationHeaderWithUserAccessToken(authHeader, workBufferSize, userAccessTokenCache, userId));
    *headers = curl_slist_append(*headers, authHeader);
    NN_RESULT_THROW_UNLESS(*headers != nullptr, http::ResultCurlSlistAppendFailure());
    *headers = curl_slist_append(*headers, "Content-Type: application/json");
    NN_RESULT_THROW_UNLESS(*headers != nullptr, http::ResultCurlSlistAppendFailure());

    // リクエスト生成
    NN_RESULT_DO(stream.SetHeaders(*headers));
    NN_RESULT_DO(stream.SetHttpPut(postDataBuffer, false));

    // URL の作成
    char url[256];
    auto l = BaasResourceResolver::GetUrlForChannel(url, sizeof(url), userId, daId);
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);
    NN_RESULT_DO(stream.SetUrl(url));

    NN_RESULT_SUCCESS;
}

Result SetupStreamForChannelGet(
    http::CurlInputStream& stream,
    curl_slist** headers,
    const UserAccessTokenCache& userAccessTokenCache,
    const NetworkServiceAccountId& userId, uint64_t daId,
    char* workBuffer, size_t workBufferSize)
{
    // Authorization ヘッダの準備
    char* authHeader = reinterpret_cast<char*>(workBuffer);
    NN_RESULT_DO(CreateAuthorizationHeaderWithUserAccessToken(authHeader, workBufferSize, userAccessTokenCache, userId));
    *headers = curl_slist_append(*headers, authHeader);
    NN_RESULT_THROW_UNLESS(*headers != nullptr, http::ResultCurlSlistAppendFailure());

    // リクエスト生成
    NN_RESULT_DO(stream.SetHeaders(*headers));

    // URL の作成
    char url[256];
    auto l = BaasResourceResolver::GetUrlForChannel(url, sizeof(url), userId, daId);
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);
    NN_RESULT_DO(stream.SetUrl(url));

    NN_RESULT_SUCCESS;
}

Result SetupStreamForChannelDelete(
    http::CurlInputStream& stream,
    curl_slist** headers,
    const UserAccessTokenCache& userAccessTokenCache,
    const NetworkServiceAccountId& userId, uint64_t daId,
    char* workBuffer, size_t workBufferSize)
{
    // Authorization ヘッダの準備
    char* authHeader = reinterpret_cast<char*>(workBuffer);
    NN_RESULT_DO(CreateAuthorizationHeaderWithUserAccessToken(authHeader, workBufferSize, userAccessTokenCache, userId));
    *headers = curl_slist_append(*headers, authHeader);
    NN_RESULT_THROW_UNLESS(*headers != nullptr, http::ResultCurlSlistAppendFailure());

    // リクエスト生成
    NN_RESULT_DO(stream.SetHeaders(*headers));
    NN_RESULT_DO(stream.SetHttpDelete());

    // URL の作成
    char url[256];
    auto l = BaasResourceResolver::GetUrlForChannel(url, sizeof(url), userId, daId);
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);
    NN_RESULT_DO(stream.SetUrl(url));

    NN_RESULT_SUCCESS;
}
} // ~namespace nn::account::baas::<anonymous>

Result BaasChannelDriver::PutChannel(
    const NetworkServiceAccountId& id, const uint64_t loginId,
    const npns::NotificationToken& nt,
    CURL* curlHandle, void* rawBuffer, size_t bufferSize,
    const detail::Cancellable* pCancellable) const NN_NOEXCEPT
{
    struct Buffer
    {
        char postDataBuffer[1024]; // PUTするJSON。大体これくらい
        union
        {
            char workBuffer[RequiredBufferSizeForAuthorizationHeader];
            struct
            {
                char stringBuffer[128]; // NT が入ればよい
                char inputBuffer[detail::IoBufferSizeMin];
            } io;
        } u;
    };
    NN_STATIC_ASSERT(std::alignment_of<Buffer>::value <= std::alignment_of<std::max_align_t>::value);

    NN_SDK_REQUIRES_NOT_NULL(curlHandle);
    NN_SDK_REQUIRES_NOT_NULL(rawBuffer);
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(rawBuffer) % std::alignment_of<std::max_align_t>::value == 0);
    NN_SDK_REQUIRES(bufferSize >= sizeof(Buffer));

    auto* buffer = reinterpret_cast<Buffer*>(rawBuffer);

    struct curl_slist* headers = nullptr;
    NN_UTIL_SCOPE_EXIT
    {
        curl_slist_free_all(headers);
    };

    http::CurlInputStream stream(curlHandle, pCancellable);
    NN_RESULT_DO(stream.Initialize());
    stream.SetStringBuffer(buffer->u.io.stringBuffer, sizeof(buffer->u.io.stringBuffer));
    stream.SetInputBuffer(buffer->u.io.inputBuffer, bufferSize - offsetof(Buffer, u.io.inputBuffer));

    NN_RESULT_DO(SetupStreamForChannelPut(
        stream, &headers, buffer->postDataBuffer, sizeof(buffer->postDataBuffer),
        m_UserAccessTokenCache, id, loginId, nt, buffer->u.workBuffer, sizeof(buffer->u.workBuffer)));

    // 通信の実行と結果の適用
    ChannelAdaptor adaptor(id, loginId);
    NN_RESULT_DO(stream.Open());
    NN_RESULT_DO(json::ImportJsonByRapidJson(adaptor, stream, pCancellable));
    stream.Close();
    NN_RESULT_DO(adaptor.Adapt(stream.GetHttpCode()));

    // 結果の確認
    NN_RESULT_THROW_UNLESS(
        std::strncmp(nt.data, adaptor.GetNt().data, sizeof(nt.data)) == 0,
        ResultBaasDataBroken());
    NN_RESULT_SUCCESS;
}

Result BaasChannelDriver::GetChannel(
    npns::NotificationToken* pOutNt,
    const NetworkServiceAccountId& id, const uint64_t loginId,
    CURL* curlHandle, void* rawBuffer, size_t bufferSize,
    const detail::Cancellable* pCancellable) const NN_NOEXCEPT
{
    struct Buffer
    {
        union
        {
            char workBuffer[RequiredBufferSizeForAuthorizationHeader];
            struct
            {
                char stringBuffer[128]; // NT が入ればよい
                char inputBuffer[detail::IoBufferSizeMin];
            } io;
        } u;
    };
    NN_STATIC_ASSERT(std::alignment_of<Buffer>::value <= std::alignment_of<std::max_align_t>::value);

    NN_SDK_REQUIRES_NOT_NULL(curlHandle);
    NN_SDK_REQUIRES_NOT_NULL(rawBuffer);
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(rawBuffer) % std::alignment_of<std::max_align_t>::value == 0);
    NN_SDK_REQUIRES(bufferSize >= sizeof(Buffer));

    auto* buffer = reinterpret_cast<Buffer*>(rawBuffer);

    struct curl_slist* headers = nullptr;
    NN_UTIL_SCOPE_EXIT
    {
        curl_slist_free_all(headers);
    };

    http::CurlInputStream stream(curlHandle, pCancellable);
    NN_RESULT_DO(stream.Initialize());
    stream.SetStringBuffer(buffer->u.io.stringBuffer, sizeof(buffer->u.io.stringBuffer));
    stream.SetInputBuffer(buffer->u.io.inputBuffer, bufferSize - offsetof(Buffer, u.io.inputBuffer));

    NN_RESULT_DO(SetupStreamForChannelGet(
        stream, &headers,
        m_UserAccessTokenCache, id, loginId, buffer->u.workBuffer, sizeof(buffer->u.workBuffer)));

    // 通信の実行と結果の適用
    ChannelAdaptor adaptor(id, loginId);
    NN_RESULT_DO(stream.Open());
    NN_RESULT_DO(json::ImportJsonByRapidJson(adaptor, stream, pCancellable));
    stream.Close();
    NN_RESULT_DO(adaptor.Adapt(stream.GetHttpCode()));

    // 結果の取得
    *pOutNt = adaptor.GetNt();
    NN_RESULT_SUCCESS;
}

Result BaasChannelDriver::DeleteChannel(
    const NetworkServiceAccountId& id, const uint64_t loginId,
    CURL* curlHandle, void* rawBuffer, size_t bufferSize,
    const detail::Cancellable* pCancellable) const NN_NOEXCEPT
{
    struct Buffer
    {
        union
        {
            char workBuffer[RequiredBufferSizeForAuthorizationHeader];
            struct
            {
                char stringBuffer[StringBufferSizeMin]; // 使わない
                char inputBuffer[detail::IoBufferSizeMin];
            } io;
        } u;
    };

    NN_SDK_REQUIRES_NOT_NULL(curlHandle);
    NN_SDK_REQUIRES_NOT_NULL(rawBuffer);
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(rawBuffer) % std::alignment_of<std::max_align_t>::value == 0);
    NN_SDK_REQUIRES(bufferSize >= sizeof(Buffer));

    auto* buffer = reinterpret_cast<Buffer*>(rawBuffer);

    struct curl_slist* headers = nullptr;
    NN_UTIL_SCOPE_EXIT
    {
        curl_slist_free_all(headers);
    };

    http::CurlInputStream stream(curlHandle, pCancellable);
    NN_RESULT_DO(stream.Initialize());
    stream.SetStringBuffer(buffer->u.io.stringBuffer, sizeof(buffer->u.io.stringBuffer));
    stream.SetInputBuffer(buffer->u.io.inputBuffer, bufferSize - offsetof(Buffer, u.io.inputBuffer));

    NN_RESULT_DO(SetupStreamForChannelDelete(
        stream, &headers,
        m_UserAccessTokenCache, id, loginId, buffer->u.workBuffer, sizeof(buffer->u.workBuffer)));

    // 通信の実行と結果の適用
    BaasDeleteAdaptor adaptor;
    NN_RESULT_DO(stream.Open());
    auto result = json::ImportJsonByRapidJson(adaptor, stream, pCancellable);
    NN_RESULT_THROW_UNLESS(
        result.IsSuccess() || json::ResultJsonEmpty::Includes(result), // 成功の場合 Empty が返る。
        result);
    stream.Close();
    NN_RESULT_DO(adaptor.Adapt(stream.GetHttpCode()));

    NN_RESULT_SUCCESS;
}

} // ~namespace nn::account::baas
}
}

