﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/detail/account_IAsyncContext.sfdl.h>
#include <nn/account/account_AsyncContext.h>

#include <nn/sf/sf_ISharedObject.h>

#include "detail/account_ShimLibraryUtility.h"

namespace nn { namespace account {

AsyncContext& AsyncContext::Swap(AsyncContext& rhs) NN_NOEXCEPT
{
    std::swap(m_User, rhs.m_User);
    std::swap(m_Ptr, rhs.m_Ptr);
    return *this;
}
detail::IAsyncContext* AsyncContext::GetPointerUnsafe() const NN_NOEXCEPT
{
    return m_Ptr;
}
AsyncContext::AsyncContext(AsyncContext&& rhs) NN_NOEXCEPT
    : m_User(rhs.m_User)
    , m_Ptr(rhs.m_Ptr)
{
    rhs.m_User = InvalidUid;
    rhs.m_Ptr = nullptr;
}
AsyncContext& AsyncContext::operator=(AsyncContext&& rhs) NN_NOEXCEPT
{
    AsyncContext tmp(std::move(rhs));
    tmp.Swap(*this);
    return *this;
}
AsyncContext::AsyncContext(detail::IAsyncContext* ptr) NN_NOEXCEPT
    : m_User(InvalidUid)
    , m_Ptr(ptr)
{
}
AsyncContext::AsyncContext(detail::IAsyncContext* ptr, const Uid& user) NN_NOEXCEPT
    : m_User(user)
    , m_Ptr(ptr)
{
}
AsyncContext::AsyncContext() NN_NOEXCEPT
    : m_User(InvalidUid)
    , m_Ptr(nullptr)
{
}
AsyncContext::~AsyncContext() NN_NOEXCEPT
{
    if (*this)
    {
        bool done;
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->HasDone(&done));
        sf::ReleaseSharedObject(m_Ptr);
        m_Ptr = nullptr;
        // タスクが未完了の状態で AsyncContext を破棄しようとすると、もれなく ABORT する。
        NN_ABORT_UNLESS(
            done,
            "[nn::account] -----------------------------------------------\n"
            "  ABORT: Asynchronous operation incomplete or not cancelled.\n");
    }
}
AsyncContext::operator bool() const NN_NOEXCEPT
{
    return m_Ptr != nullptr;
}
Result AsyncContext::GetSystemEvent(os::SystemEvent* pOut) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, AsyncContext);

    sf::NativeHandle h;
    NN_RESULT_DO(m_Ptr->GetSystemEvent(&h));
    pOut->AttachReadableHandle(h.GetOsHandle(), h.IsManaged(), os::EventClearMode_ManualClear);
    h.Detach();
    NN_RESULT_SUCCESS;
}
Result AsyncContext::Cancel() NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, AsyncContext);

    return m_Ptr->Cancel();
}
Result AsyncContext::HasDone(bool* pOut) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, AsyncContext);

    return m_Ptr->HasDone(pOut);
}
Result AsyncContext::GetResult() NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, AsyncContext);

    return m_Ptr->GetResult();
}

}} // ~namespace nn::account
