﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include <nn/nn_DeviceCode.h>
#include <nn/nn_SdkAssert.h>

#include <nn/i2c/driver/i2c_I2cDeviceProperty.h>
#include <nn/i2c/driver/i2c_II2cDriver.h>
#include <nn/i2c/i2c_Type.h>
#include <nn/os/os_InterruptEvent.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/util/util_IntrusiveList.h>

#include "i2cTegra_RegisterI2c.h"

namespace nnd { namespace i2c { namespace tegra { namespace detail {

class BusPowerManager;

class I2cBusAccessor : public nn::i2c::driver::II2cDriver
{
    NN_DDSF_CAST_SAFE_DECL;

    NN_DISALLOW_COPY(I2cBusAccessor);
    NN_DISALLOW_MOVE(I2cBusAccessor);

private:
    nn::util::IntrusiveListNode m_ListNode;
    friend class nn::util::IntrusiveListMemberNodeTraits<I2cBusAccessor, &I2cBusAccessor::m_ListNode>;

public:
    typedef nn::util::IntrusiveList<I2cBusAccessor, nn::util::IntrusiveListMemberNodeTraits<I2cBusAccessor, &I2cBusAccessor::m_ListNode>> List;

public:
    I2cBusAccessor() NN_NOEXCEPT
    {
        // メンバ変数の初期化以外何もしない
    }

    //! デバイスツリーから取得した属性の値で I2cBusAccessor を初期化します。
    void Initialize(uint32_t address, uint32_t length, uint32_t interrupt, bool isPowerBus, nn::i2c::SpeedMode speedMode, BusPowerManager* pBusPowerManager) NN_NOEXCEPT;

    //! I2cBusAccessor にデバイスコードを割り当てます。
    void RegisterDeviceCode(nn::DeviceCode deviceCode) NN_NOEXCEPT;

    virtual void InitializeDriver() NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result InitializeDevice(nn::i2c::driver::I2cDeviceProperty* pDevice) NN_NOEXCEPT NN_OVERRIDE;
    virtual void FinalizeDevice(nn::i2c::driver::I2cDeviceProperty* pDevice) NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result Send(nn::i2c::driver::I2cDeviceProperty* pDevice, const void* pInData, size_t dataBytes, nn::i2c::TransactionOption inOption) NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result Receive(void* pOutData, nn::i2c::driver::I2cDeviceProperty* pDevice, size_t dataBytes, nn::i2c::TransactionOption inOption) NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::os::SdkMutex& GetTransactionOrderMutex() NN_NOEXCEPT NN_OVERRIDE;
    virtual void SuspendBus() NN_NOEXCEPT NN_OVERRIDE;
    virtual void SuspendPowerBus() NN_NOEXCEPT NN_OVERRIDE;
    virtual void ResumeBus() NN_NOEXCEPT NN_OVERRIDE;
    virtual void ResumePowerBus() NN_NOEXCEPT NN_OVERRIDE;

    virtual const nn::DeviceCode& GetDeviceCode() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_DeviceCode;
    }

    nn::i2c::SpeedMode GetSpeedMode() const NN_NOEXCEPT
    {
        return m_SpeedMode;
    }

    uint32_t GetBaseAddress() const NN_NOEXCEPT
    {
        return m_BaseAddress;
    }

    uint32_t GetLength() const NN_NOEXCEPT
    {
        return m_Length;
    }

    uint32_t GetInterruptNumber() const NN_NOEXCEPT
    {
        return m_InterruptNumber;
    }

    void RemoveFrom(List* pList) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pList);
        pList->erase(pList->iterator_to(*this));
    }

    bool IsLinkedToList() const NN_NOEXCEPT
    {
        return m_ListNode.IsLinked();
    }

private:
    /**
     * @brief Transaction の向き
     */
    enum Xfer
    {
        Xfer_Write = 0,
        Xfer_Read  = 1,
    };

    //! I2cBusAccessor の内部状態
    enum class State
    {
        NotInitialized,     //!< デフォルトコンストラクタによる初期化後
        Initializing,       //!< 初期化中
        Initialized,        //!< 初期化完了
        Suspended,          //!< SuspendBus 実行後、SuspendPowerBus 実行前/ResumePowerBus 実行後、ResumeBus 実行前
        PowerBusSuspended,  //!< SuspendPowerBus 実行後、ResumePowerBus 実行前
    };

private:
    void ExecuteInitialConfig() NN_NOEXCEPT;

    nn::Result Send(const nn::Bit8* pInData, size_t dataBytes, nn::i2c::TransactionOption inOption, nn::Bit16 slaveAddress, nn::i2c::AddressingMode addressingMode) NN_NOEXCEPT;
    nn::Result Receive(nn::Bit8* pOutData, size_t dataBytes, nn::i2c::TransactionOption inOption, nn::Bit16 slaveAddress, nn::i2c::AddressingMode addressingMode) NN_NOEXCEPT;

    //! [Ref1] 5.7.1 I2C コントローラをリセットします。
    void ResetController() const NN_NOEXCEPT;

    //! [Ref1] 31.1  バスをクリアします。
    void BusClear() const NN_NOEXCEPT;

    //! @ref nn::i2c::SpeedMode に応じた設定値をクロック関連のレジスタに反映させます。
    void SetClockRegisters(nn::i2c::SpeedMode speedMode) NN_NOEXCEPT;

    //! パケットベースモード向けレジスタの初期設定を行います。
    void SetPacketModeRegisters() NN_NOEXCEPT;

    //! Packet Transfer Status レジスタの状態を @ref nn::Result にマップしたものを取得します。
    nn::Result GetTransactionResult() const NN_NOEXCEPT;

    //! [31.4.4] Error Handling に従ったエラーハンドリングを行います。
    void HandleTransactionError(nn::Result result) NN_NOEXCEPT;

    //! @ref GetTransactionResult と @ref HandleTransactionError と割り込みの disable を行う Utility 関数です。
    inline nn::Result CheckAndHandleError() NN_NOEXCEPT
    {
        auto result = GetTransactionResult();
        HandleTransactionError(result);

        if ( result.IsFailure() )
        {
            DisableInterruptMask();
            nn::os::ClearInterruptEvent(&m_InterruptEvent);
            return result;
        }
        return nn::ResultSuccess();
    }

    //! デバッグ向けにレジスタをダンプします。
    void DumpRegisters(int line) const NN_NOEXCEPT;

    void WriteHeader(Xfer xfer, size_t dataBytes, nn::i2c::TransactionOption inOption, nn::Bit16 slaveAddress, nn::i2c::AddressingMode addressingMode) NN_NOEXCEPT;
    inline void DisableInterruptMask() NN_NOEXCEPT
    {
        Write32(&(m_pRegI2c->interruptMaskRegister), 0x00);
        DummyRead(&(m_pRegI2c->interruptMaskRegister));
    }

    nn::Result FlushFifos() NN_NOEXCEPT;

private:
    //! I2C コントローラのレジスタへのアクセサ
    I2cRegTable* m_pRegI2c { nullptr };

    //! スピードモード
    nn::i2c::SpeedMode m_SpeedMode { nn::i2c::SpeedMode_Fast };

    //! I2C コントローラからの割り込みイベント
    nn::os::InterruptEventType m_InterruptEvent;

    //! バスのユーザー数
    int m_UserCount { 0 };

    //! m_UserCount とそれに関連する操作をロック対象とする SdkMutex
    nn::os::SdkMutex m_UserCountMutex { };

    //! IO レジスタをロック対象とする SdkMutex
    nn::os::SdkMutex m_RegisterMutex { };

    //! バスパワーマネージャへのポインタ
    BusPowerManager* m_pBusPowerManager { nullptr };

    //! I2cBusAccessor の内部状態
    State m_State { State::NotInitialized };

    //! 一連の送受信処理を行う間にロックするべきミューテックス
    // TODO: Bus 毎で Transaction を Mutex しなくてもいい気がする
    nn::os::SdkMutex m_TransactionOrderMutex { };

    //! 電源制御に必要なバス（PowerBus）か否か
    bool m_IsPowerBus { false };

    //! コントローラのレジスタのベースアドレス@n
    //! @ref I2cBusAccessor を特定する主キー
    uint32_t m_BaseAddress { 0 };

    //! コントローラのレジスタの領域の長さ
    uint32_t m_Length { 0 };

    //! 割り込み番号
    uint32_t m_InterruptNumber { 0 };

    //! バスに対応するデバイスコード
    nn::DeviceCode m_DeviceCode { 0 };
};

}}}} // nnd::i2c::tegra::detail
